from hashlib import md5


class ItemType(object):
    DEPARTMENT = 'department'
    SERVICE = 'service'
    PERSON = 'person'
    QUEUE = 'queue'
    REPO = 'repo'
    COMMIT = 'commit'
    RESOURCE = 'resource'
    RELEASE = 'release'
    HANDLE = 'handle'
    ST_TICKET = 'st_ticket'
    MICROSERVICE = 'microservice'


class Item(object):
    def __init__(self, item_type, project_id, value, ancestor_id, created_at=None, updated_at=None, **kwargs):
        self.type = item_type
        self.project_id = project_id
        self.value = value
        if type(ancestor_id) is list:
            self.ancestors = ancestor_id
        else:
            self.ancestors = [ancestor_id]
        self.created_at = created_at
        self.updated_at = updated_at
        self.__set_id()
        self.fields = kwargs

    def append_ancestor(self, ancestor_id):
        if type(ancestor_id) is list:
            self.ancestors = list(set(self.ancestors).union(ancestor_id))
        elif ancestor_id not in self.ancestors:
            self.ancestors.append(ancestor_id)

    def __set_id(self):
        m = md5()
        if type(self.value) is int:
            value = str(self.value)
        else:
            value = self.value
        data = self.type + self.project_id + value
        data = data.encode('utf-8')
        m.update(data)
        self.__id = m.hexdigest()

    def get_id(self):
        return self.__id

    def get_value(self):
        return self.value

    def get_type(self):
        return self.type

    def serialize(self):
        d = {
            'type': self.type,
            'project_id': self.project_id,
            'value': self.value,
            'ancestors': self.ancestors,
            'uuid': self.__id,
            'fields': self.fields,
        }

        if self.created_at:
            d['created_at'] = self.created_at
        if self.updated_at:
            d['updated_at'] = self.updated_at

        return d

    @classmethod
    def unserialize(cls, item):
        item_type = item['type']
        project_id = item['project_id']
        value = item['value']
        ancestors = item.get('ancestors', list())
        created_at = item.get('created_at')
        updated_at = item.get('updated_at')
        kwargs = item.get('fields', dict())

        return cls(item_type, project_id, value, ancestors, created_at, updated_at, **kwargs)
