# -*- coding: utf-8 -*-

import requests
import json
import copy

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter


class ItsRuchkaNameParameter(SandboxStringParameter):

    """
        Имя ручки для изменения
    """
    name = 'its_ruchka_name'
    description = 'Имя ITS ручки'


class ItsValueNameParameter(SandboxStringParameter):

    """
        Новое значение для изменения
    """
    name = 'its_value_name'
    description = 'Значение для изменения'


class HTTPNotFound(Exception):
    pass


class ITS(object):
    API_HOST = 'http://its.yandex-team.ru/v1/values/'
    REQUEST_HEADERS = {
        'Content-Type': 'application/json; charset=utf-8',
        'Accept': 'application/json',
    }

    def __getRequest(self, headers={}):
        r_headers = copy.deepcopy(headers)
        r_headers.update(self.REQUEST_HEADERS)
        try:
            r = requests.get("%s" % self.url, headers=r_headers)
            r.raise_for_status()
        except requests.exceptions.HTTPError as e:
            if r.status_code == 404:
                raise HTTPNotFound(e)
            else:
                raise Exception(e)
        return r

    def __setValue(self, data, headers={}, params=''):
        r_headers = copy.deepcopy(headers)
        r_headers.update(self.REQUEST_HEADERS)
        r = requests.post(
            "%s" % self.url,
            data=json.dumps({
                'value': data
            }),
            headers=r_headers,
            params=params
        )
        r.raise_for_status()
        return r.status_code

    def __postRequest(self, data, headers={}, params=''):
        try:
            r = self.__getRequest(headers=headers)
        except HTTPNotFound:
            return self.__setValue(data, headers, params)
        else:
            if r.json()['user_value'] == data:
                return False
            headers.update({
                'If-Match': r.headers['etag'],
            })
            return self.__setValue(data, headers, params)

    @property
    def data(self):
        return self.__getRequest().json()

    @data.setter
    def data(self, value):
        self.__postRequest(value)

    def __init__(self, token, url):
        self.url = self.API_HOST + url
        self.REQUEST_HEADERS.update({
            'Authorization': 'OAuth %s' % (token),
        })


class SetItsValueTask(SandboxTask):
    """
        Отслеживаем новый шардмап в svn директории и деплоим его в сервис.
        После успешного завершения даём ссылку на релиз шардмапа для выкатки
        его в продакшен.
    """

    type = 'ITS_SET_VALUE'

    input_parameters = [
        ItsRuchkaNameParameter,
        ItsValueNameParameter
    ]

    def on_execute(self):
        nanny_token = self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token')
        its = ITS(token=nanny_token, url=self.ctx[ItsRuchkaNameParameter.name])
        its.data = self.ctx[ItsValueNameParameter.name]


__Task__ = SetItsValueTask
