#  -*- coding: utf-8 -*-

import os
import logging

from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter
import sandbox.projects.TestReportUnit as Unit


class NReqs(SandboxStringParameter):
    name = 'n_reqs'
    description = 'number of firing reqs'
    default_value = 5555


class ExtraParam(SandboxStringParameter):
    name = 'extra'
    description = 'empty or gw/tr/hm for gw/tr/hm log preparations'
    choices = [
        ('Russia', ''), ('Gateway', 'gw'), ('Turkey', 'tr'), ('Hamster', 'hm')
    ]
    default_value = choices[0][1]


class UpdateTestEnv(SandboxBoolParameter):
    required = True
    name = 'update_testenv'
    description = 'Use resource in auto update testenv mechanism'
    default_value = False
    do_not_copy = True


class AccessLog(ResourceSelector):
    name = 'access_log'
    description = 'access_log:'
    resource_type = resource_types.PLAIN_TEXT


class ReqansLog(ResourceSelector):
    name = 'reqans_log'
    description = 'reqans_log:'
    resource_type = resource_types.PLAIN_TEXT


class ShootLogPrepare(SandboxTask, object):
    """
        Препарат аксесс логов для обстрела
    """

    type = 'SHOOT_LOG_PREPARE'
    resource_type = resource_types.SHOOT_LOG_PREPARE

    execution_space = 5000
    res_folder = "."

    input_parameters = [NReqs, ExtraParam, UpdateTestEnv, AccessLog, ReqansLog]

    def on_execute(self):

        self.scripts_dir = 'scripts_dev'

        copy_path(Arcadia.get_arcadia_src_dir('arcadia:/arc/trunk/arcadia/web/report/scripts/dev@HEAD'), self.scripts_dir)

        self.set_env()
        run_process(['REPORT_DIR=`pwd` ./shoot_log_prepare.sh %s' % (self.ctx[ExtraParam.name])],
                    work_dir=self.scripts_dir, timeout=10000, shell=True, check=True, log_prefix='fire', outputs_to_one_file=False)

        # создаем ресурс SHOOT_LOG_PREPARE
        self.make_resource()

    def set_env(self):
        env_vars = {
            "SHOOT_DIR": os.path.join(self.abs_path(), self.res_folder),
            "SHOOT_NREQS": self.ctx[NReqs.name],
        }

        if AccessLog.name in self.ctx and ReqansLog.name in self.ctx:
            Unit.TestReportUnit.expand_resource(self.ctx[AccessLog.name], os.path.join(env_vars["SHOOT_DIR"], "_access_log"))
            # access_log -> _access_log/access_log
            if os.path.exists('access_log'):
                os.remove('access_log')
            os.symlink(os.path.join(env_vars["SHOOT_DIR"], '_access_log', 'access_log'), 'access_log')

            Unit.TestReportUnit.expand_resource(self.ctx[ReqansLog.name], os.path.join(env_vars["SHOOT_DIR"], "_reqans_log"))
            # reqans_log -> _reqans_log/reqans_log
            if os.path.exists('reqans_log'):
                os.remove('reqans_log')
            os.symlink(os.path.join(env_vars["SHOOT_DIR"], '_reqans_log', 'reqans_log'), 'reqans_log')

            env_vars["SKIP_RSYNC"] = "1"

        for varname, varval in env_vars.iteritems():
            os.environ[varname] = varval

    def make_resource(self):
        name = "access_log_parsed"
        if not os.path.exists(name) or not os.path.getsize(name):
            raise SandboxTaskFailureError('Res %s is empty.' % name)

        if not self.ctx.get('shoot_log_prepare_resource_id'):
            # создаем ресурс
            attrs = {'arch': self.client_info['arch'], 'ttl': 365}
            if self.ctx[UpdateTestEnv.name]:
                attrs['type'] = 'report_core%s' % self.ctx[ExtraParam.name]

            unit_resource = self.create_resource(
                description=self.descr,
                resource_path=name,
                resource_type=self.resource_type,
                arch=None,
                attributes=attrs
            )
            logging.info('Create resource %s id: %s' % (self.resource_type, unit_resource.id))
            self.ctx['shoot_log_prepare_resource_id'] = unit_resource.id

    def cleanup(self):
        return


__Task__ = ShootLogPrepare
