from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter, ResourceSelector, SandboxIntegerParameter
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import utils

import time


class SimilarOrgsDoAll(SandboxTask):
    """
        Calculates geosearch similar orgs from hypotheses
    """
    type = 'SIMILAR_ORGS_DO_ALL'

    class SpravHypothesesCalcer(LastReleasedResource):
        name = 'sprav_hypos_calcer'
        description = 'Sprav hypotheses calcer: '
        resource_type = resource_types.SIMILAR_ORGS_HYPOS_SPRAV_EXECUTABLE

    class SimilarOrgsCalcer(LastReleasedResource):
        name = 'similar_orgs_calcer'
        description = 'Similar orgs calcer: '
        resource_type = resource_types.CALC_SIMILAR_ORGS_EXECUTABLE

    class BusinessIndex(LastReleasedResource):
        name = 'business_index'
        description = 'Business index: '
        resource_type = resource_types.MAPS_DATABASE_BUSINESS

    class RubricList(ResourceSelector):
        name = 'rubric_list'
        description = 'Accepted rubrics: '
        resource_type = resource_types.GEOSEARCH_RUBRIC_ID_LIST
        default_value = None

    class GeoidList(ResourceSelector):
        name = 'geoid_list'
        description = 'Accepted geoids: '
        resource_type = resource_types.GEOSEARCH_GEOID_LIST
        default_value = None

    class SimilarOrgsHypotheses(LastReleasedResource):
        name = 'similar_orgs_hypotheses'
        description = 'Similar orgs hypotheses: '
        resource_type = resource_types.SIMILAR_ORGS_HYPOTHESES
        default_value = None

    class SimilarOrgsModel(ResourceSelector):
        name = 'similar_orgs_model'
        description = 'Similar orgs model: '
        resource_type = resource_types.SIMILAR_ORGS_MODEL
        default_value = None

    class MaxSimilarOrgs(SandboxIntegerParameter):
        name = 'max_similar_orgs'
        description = 'Max similar orgs: '
        default_value = 20

    class MaxDistance(SandboxIntegerParameter):
        name = 'max_distance'
        description = 'Max distance for similar orgs: '
        default_value = 3000

    class PolynomName(SandboxStringParameter):
        name = 'polynom_name'
        description = 'PolynomName: '
        default_value = ''

    class ExportPreparer(LastReleasedResource):
        name = 'export_preparer'
        description = 'Export preparer: '
        resource_type = resource_types.CREATE_SIMILAR_ORGS_EXPORT_EXECUTABLE

    class PhotosExport(LastReleasedResource):
        name = 'photos_export'
        description = 'Photos export: '
        resource_type = resource_types.BUSINESS_SNIPPET_PHOTOS

    class SaasDumper(LastReleasedResource):
        name = 'saas_dumper'
        description = 'SaaS dumper: '
        resource_type = resource_types.DUMP_TO_INDEXERPROXY_EXECUTABLE

    class SaasIndexer(SandboxStringParameter):
        description = "SaaS indexer"
        name = 'saas_indexer'
        choices = [
            ('testing', 'saas-indexerproxy-maps-prestable.yandex.net'),
            ('stable', 'saas-indexerproxy-maps-kv.yandex.net'),
        ]
        default_value = 'saas-indexerproxy-maps-prestable.yandex.net'

    class ExperimentPrefix(SandboxStringParameter):
        description = 'Experiment prefix'
        name = 'experiment_prefix'
        default_value = ''

    input_parameters = (
        SpravHypothesesCalcer,
        SimilarOrgsCalcer,
        BusinessIndex,
        RubricList,
        GeoidList,
        SimilarOrgsHypotheses,
        SimilarOrgsModel,
        MaxSimilarOrgs,
        MaxDistance,
        PolynomName,
        ExportPreparer,
        PhotosExport,
        SaasDumper,
        SaasIndexer,
        ExperimentPrefix,
    )

    def calc_hypotheses(self):
        hypotheses = utils.get_or_default(self.ctx, self.SimilarOrgsHypotheses)
        if not hypotheses:
            self.ctx['calc_hypotheses_task'] = self.create_subtask('CALC_SIMILAR_ORGS_HYPOTHESES', 'Calculating hypotheses').id

    def calc_similar_orgs(self):
        if 'calc_hypotheses_task' in self.ctx:
            self.ctx['similar_orgs_hypotheses'] = channel.sandbox.list_resources(resource_type=resource_types.SIMILAR_ORGS_HYPOTHESES, task_id=self.ctx['calc_hypotheses_task'], omit_failed=True)[0].id
        self.ctx['calc_similar_orgs_task'] = self.create_subtask('CALC_SIMILAR_ORGS', 'Calculating similar orgs').id

    def publish_similar_orgs(self):
        self.ctx['similar_orgs'] = channel.sandbox.list_resources(resource_type=resource_types.SIMILAR_ORGS_LIST, task_id=self.ctx['calc_similar_orgs_task'], omit_failed=True)[0].id
        self.ctx['cluster'] = 'hahn.yt.yandex.net'
        self.ctx['table'] = '//home/geosearch/similar_orgs/saas_export_' + str(int(time.time()))
        self.ctx['publish_similar_orgs_task'] = self.create_subtask('PUBLISH_SIMILAR_ORGS', 'Publishing similar orgs').id

    def on_execute(self):
        with self.memoize_stage.calc_hypotheses:
            self.calc_hypotheses()
            utils.check_subtasks_fails()
        utils.check_subtasks_fails()

        with self.memoize_stage.calc_similar_orgs:
            self.calc_similar_orgs()
            utils.check_subtasks_fails()
        utils.check_subtasks_fails()

        with self.memoize_stage.publish_similar_orgs:
            self.publish_similar_orgs()
            utils.check_subtasks_fails()
        utils.check_subtasks_fails()


__Task__ = SimilarOrgsDoAll
