# coding: utf-8

import os
import shutil
import logging

from sandbox import sdk2
from sandbox.common.types import misc as ctm

from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.sandboxsdk import environments

from sandbox.projects.Sovetnik.SovetnikResource import SovetnikScript, SovetnikScriptTypes

cache_script_name = "create-node-modules-resource.py"


class SovetnikBuild(sdk2.Task):
    """ Common task for all Sovetnik's builds. """
    work_dir_name = 'sovetnik'

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        ram = 2 * 1024
        disk_space = 2 * 1024
        environments = (
            environments.NodeJS('7.10.0'),
            environments.GCCEnvironment(),
        )

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('SSH parameters') as ssh_parameters:
            ssh_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's SSH private key",
                required=True,
                default='SOVETNIK',
            )
            ssh_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's SSH private key",
                required=True,
                default='robot-sovetnik-ssh',
            )

    def get_path_to_work_dir(self):
        """ Returns path to a working directory. """
        if self.work_dir_name is None:
            raise AttributeError('Name of working directory is not defined')

        return str(self.path(self.work_dir_name))

    def _get_node_path(self):
        """ Returns path to `node` executable of the prepared environment. """
        return self.Requirements.environments[0].node

    def _get_npm_path(self):
        """ Returns path to `npm` executable of the prepared environment. """
        return self.Requirements.environments[0].npm

    def clone_git_repository(self):
        """
        Clones a Git repository from remote source to a working directory.

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()

        with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            if os.path.exists(path_to_work_dir):
                shutil.rmtree(path_to_work_dir)

            git = Git(self.Parameters.git_ssh_url)
            git.clone(path_to_work_dir, self.Parameters.git_release_branch)

            cmd = [git.executable, 'submodule', 'update', '--init']
            sp.check_output(cmd, cwd=path_to_work_dir)

            cmd = [git.executable, 'rev-parse', '--short', 'HEAD']
            self.Context.git_commit = sp.check_output(cmd, cwd=path_to_work_dir).strip()

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()

        os.environ['NPM_PATH'] = self._get_npm_path()

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call(['make', 'build'], cwd=path_to_work_dir, stdout=pl.stdout, stderr=pl.stderr)

    def on_execute(self):
        """
        Clones a Git repository from remote source to a working directory.
        Builds Sovetnik's package.

        :return: Nothing.
        """
        with self.memoize_stage.clone_git_repository(commit_on_entrance=False):
            self.clone_git_repository()

        with self.memoize_stage.build(commit_on_entrance=False):
            self.build()

    def find_node_modules_resource(self):
        resource = SovetnikScript.find(script_type=SovetnikScriptTypes.NODE_MODULES_CACHE).first()

        if resource is None:
            return False

        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        return resource_path

    def prepare_node_modules(self):
        path_to_work_dir = self.get_path_to_work_dir()

        with sdk2.helpers.ProcessLog(self, logger='preparation') as plp:
            path_to_cache_script = self.find_node_modules_resource()
            if path_to_cache_script is False:
                logging.debug("Cannot find resource, running npm install")
                sp.check_call(['npm', 'install', '--registry=https://npm.yandex-team.ru'], cwd=path_to_work_dir,
                              stdout=plp.stdout, stderr=plp.stderr)
            else:
                path_to_cache_script = str(path_to_cache_script)
                logging.debug("Sync resource to path: {}".format(path_to_cache_script))
                local_resource_name = os.path.join(path_to_work_dir, cache_script_name)
                logging.debug('Current dir: {}'.format(
                    os.listdir('/')))

                sp.check_call(['cp', '-R', path_to_cache_script, local_resource_name], cwd=path_to_work_dir,
                              stdout=plp.stdout, stderr=plp.stderr)

                sandbox_key = sdk2.Vault.data('sovetnik-sandbox-key')

                sp.check_call(['python', local_resource_name, sandbox_key], cwd=path_to_work_dir,
                              stdout=plp.stdout, stderr=plp.stderr)
