# coding: utf-8

import os
import shutil
import glob
import logging

from sandbox import sdk2
from sandbox.common.types import misc as ctm

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.sandboxsdk import environments
from sandbox.projects.common.arcadia.sdk import mount_arc_path

from sandbox.projects.Sovetnik.SovetnikResource import SovetnikScript, SovetnikScriptTypes

cache_script_name = "create-node-modules-resource.py"


class SovetnikBuildArcadia(sdk2.Task):
    """ Common task for all Sovetnik's builds. """
    work_dir_name = 'sovetnik'
    build_dir_name = 'build_dir'

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        ram = 2 * 1024
        disk_space = 2 * 1024
        environments = (
            environments.NodeJS('7.10.0'),
            environments.GCCEnvironment(),
        )

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('SSH parameters') as ssh_parameters:
            ssh_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's SSH private key",
                required=True,
                default='SOVETNIK',
            )
            ssh_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's SSH private key",
                required=True,
                default='robot-sovetnik-ssh',
            )

    def get_path_to_work_dir(self):
        """ Returns path to a working directory. """
        if self.work_dir_name is None:
            raise AttributeError('Name of working directory is not defined')

        return str(self.path(self.work_dir_name))

    def _get_node_path(self):
        """ Returns path to `node` executable of the prepared environment. """
        return self.Requirements.environments[0].node

    def _get_npm_path(self):
        """ Returns path to `npm` executable of the prepared environment. """
        return self.Requirements.environments[0].npm

    def checkout_arcadia_repository(self):
        logging.debug('Checkout arcadia repo {}'.format(self.Parameters.repo_path))
        with mount_arc_path(self.Parameters.arc_path, use_arc_instead_of_aapi=True) as arcadia:
            self.work_dir_name = os.path.join(arcadia, self.Parameters.repo_path)

            self.build_dir_name = os.path.join(str(self.path('')).rstrip('/'), 'build_dir')
            logging.debug('Copy arcadia to local build dir > {} to {}'.format(self.work_dir_name,
                                                                              self.build_dir_name))
            shutil.copytree(self.work_dir_name, self.build_dir_name)

            self.work_dir_name = self.build_dir_name
            logging.debug("Build dir content > ")
            logging.debug(glob.glob('{build_dir}/*'.format(build_dir=self.build_dir_name)))

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()

        os.environ['NPM_PATH'] = self._get_npm_path()

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call(['make', 'build'], cwd=path_to_work_dir, stdout=pl.stdout, stderr=pl.stderr)

    def on_execute(self):
        """
        Clones Arcadia repository from remote source to a working directory.
        Builds Sovetnik's package.

        :return: Nothing.
        """
        with self.memoize_stage.checkout_arcadia_repository(commit_on_entrance=False):
            self.checkout_arcadia_repository()

        with self.memoize_stage.build(commit_on_entrance=False):
            self.build()

    def find_node_modules_resource(self):
        resource = SovetnikScript.find(script_type=SovetnikScriptTypes.NODE_MODULES_CACHE).first()

        if resource is None:
            return False

        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path

        return resource_path

    def prepare_node_modules(self):
        path_to_work_dir = self.get_path_to_work_dir()

        with sdk2.helpers.ProcessLog(self, logger='preparation') as plp:
            path_to_cache_script = self.find_node_modules_resource()
            if path_to_cache_script is False:
                sp.check_call(['npm', 'install', '--registry=https://npm.yandex-team.ru'], cwd=path_to_work_dir,
                              stdout=plp.stdout, stderr=plp.stderr)
            else:
                path_to_cache_script = str(path_to_cache_script)
                logging.debug("Sync resource to path: {}".format(path_to_cache_script))
                local_resource_name = os.path.join(path_to_work_dir, cache_script_name)
                logging.debug('Current dir: {}'.format(
                    os.listdir('/')))

                sp.check_call(['cp', '-R', path_to_cache_script, local_resource_name], cwd=path_to_work_dir,
                              stdout=plp.stdout, stderr=plp.stderr)

                sandbox_key = sdk2.Vault.data('sovetnik-sandbox-key')

                sp.check_call(['python', local_resource_name, sandbox_key], cwd=path_to_work_dir,
                              stdout=plp.stdout, stderr=plp.stderr)
