# coding: utf-8

import os
import re
import tarfile
import logging
import sandbox.common.types.misc as ctm
from sandbox import sdk2

from sandbox.projects.common.nanny import nanny

from sandbox.projects.market.resources import MARKET_DATA_OTHER
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikBackendPackage
from sandbox.sandboxsdk import environments


class MarketDataSovetnik(MARKET_DATA_OTHER):
    """ data-getter resource for Sovetnik's service. """


class SovetnikBuildBackendPackage(nanny.ReleaseToNannyTask2, SovetnikBuild):
    """ A task that builds SOVETNIK_BACKEND_PACKAGE resource from a Git repository. """
    pkg_name = 'sovetnik-backend.tar.gz'
    exclude_patterns = [
        r"^sovetnik-backend\/\.git$",
        r"^sovetnik-backend\/\.gitignore$",
        r"^sovetnik-backend\/\.editorconfig$",
        r"^sovetnik-backend\/\.eslintrc.+$",
        r"^sovetnik-backend\/\.eslintignore$",
        r"^sovetnik-backend\/\.prettier.+$",
        r"^sovetnik-backend\/jest.+$",
        r"^sovetnik-backend\/gruntfile\.js$",
        r"^sovetnik-backend\/[^\/]*\.log$",
        r"^sovetnik-backend\/[^\/]*\.map$",
    ]

    class Requirements(SovetnikBuild.Requirements):
        dns = ctm.DnsType.DNS64

        environments = (
            environments.NodeJS('8.12.0'),
            environments.GCCEnvironment(),
        )

    class Parameters(SovetnikBuild.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_parameters:
            git_release_branch = sdk2.parameters.String(
                label='Release branch',
                required=True,
                default='master',
            )
            git_ssh_url = sdk2.parameters.Url(
                label='Git remote SSH URL',
                required=True,
                default='git@github.yandex-team.ru:sovetnik/sovetnik-backend.git',
            )

    def get_path_to_pkg(self):
        """ Returns path to package. """
        if self.pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(self.pkg_name))

    def _exclude_filter(self, tarinfo):
        """
        A function that takes a TarInfo object argument and returns the TarInfo object.
        If it instead returns None the TarInfo object will be excluded from the archive.
        """
        if self.exclude_patterns is None:
            raise AttributeError('Exclude patterns are not defined')

        for pattern in self.exclude_patterns:
            if re.search(pattern, tarinfo.name):
                return None

        return tarinfo

    def pack(self):
        """
        Creates package of working directory.

        :returns: Nothing.
        """
        path_to_pkg = self.get_path_to_pkg()
        path_to_work_dir = self.get_path_to_work_dir()
        build_folder_name = 'dist'
        path_to_dist_dir = os.path.join(path_to_work_dir, build_folder_name)

        logging.debug('path to package: {}'.format(path_to_pkg))
        logging.debug('path to working directory: {}'.format(path_to_work_dir))

        with tarfile.open(path_to_pkg, 'w:gz') as tar:
            for entry in os.listdir(path_to_dist_dir):
                name = os.path.join(path_to_dist_dir, entry)
                arcname = 'sovetnik-backend/{}'.format(entry)

                tar.add(name, arcname, filter=self._exclude_filter)

    def create_resource(self):
        """ Creates SOVETNIK_BACKEND_PACKAGE resource. """
        path = self.get_path_to_pkg()
        git_commit = self.Context.git_commit
        description = "Sovetnik's backend package #{}".format(git_commit)

        resource = SovetnikBackendPackage(
            self,
            path=path,
            git_commit=git_commit,
            description=description,
        )

        sdk2.ResourceData(resource).ready()

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()

        os.environ['NPM_PATH'] = self._get_npm_path()

        os.environ['MARKET_INTERNAL_KEY'] = sdk2.Vault.data(
            'sovetnik-market-internal-key')
        os.environ['MARKET_PARTNER_KEY'] = sdk2.Vault.data(
            'sovetnik-market-partner-key')
        os.environ['YT_OAUTH_ACCESS_TOKEN'] = sdk2.Vault.data(
            'sovetnik-yt-oauth')
        os.environ['COOKIE_PARSER_SECRET'] = sdk2.Vault.data(
            'sovetnik-cookie-parser-secret')
        os.environ['CRYPTO_PASSWORD'] = sdk2.Vault.data(
            'sovetnik-crypto-password')
        os.environ['AVIA_PASSWORD'] = sdk2.Vault.data(
            'sovetnik-avia-secret')
        os.environ['ROBOT_SOVETNIK_DOC_TOKEN'] = sdk2.Vault.data('ROBOT_SOVETNIK_WIKI_OAUTH_TOKEN')

        self.prepare_node_modules()

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call(['make', 'build'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

    def on_execute(self):
        """
        Clones a Git repository from remote source to a working directory.
        Builds Sovetnik's package.
        Creates package of working directory.
        Creates SOVETNIK_BACKEND_PACKAGE resource.

        :return: Nothing.
        """
        super(SovetnikBuildBackendPackage, self).on_execute()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()
