import os
import logging
import tarfile
import sandbox.common.types.misc as ctm
import sandbox.common.errors as errors

from sandbox import sdk2

from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikResource
from sandbox.projects.Sovetnik.SovetnikBuildClientIdsPackage.yql_query import load_client_ids


class SovetnikClientIdsPackage(SovetnikResource):
    """ Resource that contains a list of Market shops. """
    ttl = 5


class SovetnikBuildClientIdsPackage(SovetnikBuild):
    """ Task to retrieve all Market shops from YT. """
    pkg_name = 'sovetnik-client-ids.tar.gz'
    client_ids_file = 'client-ids.migration'

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4 * 1024
        disk_space = 4 * 1024
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('8.12.0'),
            environments.GCCEnvironment(),
            environments.PipEnvironment('yandex-yt', use_wheel=True),
            environments.PipEnvironment('yql', use_wheel=True),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_parameters:
            git_release_branch = sdk2.parameters.String(
                label='Release branch',
                required=True,
                default='master',
            )
            git_ssh_url = sdk2.parameters.Url(
                label='Git remote SSH URL',
                required=True,
                default='git@github.yandex-team.ru:sovetnik/sovetnik-backend.git',
            )

        with sdk2.parameters.Group('SSH parameters') as ssh_parameters:
            ssh_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's SSH private key",
                required=True,
                default='SOVETNIK',
            )
            ssh_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's SSH private key",
                required=True,
                default='robot-sovetnik-ssh',
            )

        with sdk2.parameters.Group('Run parameters') as run_parameters:
            with sdk2.parameters.RadioGroup('Run environment') as run_environment:
                run_environment.values['development'] = run_environment.Value(
                    value='development', default=True)
                run_environment.values['production'] = run_environment.Value(value='production')

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yql_vault_token = sdk2.parameters.String(
                'Your yql token name in vault',
                default='YQL_TOKEN',
                required=True
            )
            yql_period = sdk2.parameters.String(
                'Period of time for YQL request',
                default='P30D',
                required=True
            )

            with sdk2.parameters.RadioGroup('Yt host') as yt_host:
                yt_host.values['hahn'] = yt_host.Value(
                    value='Hahn', default=True)

            with sdk2.parameters.String("Environment type", required=True) as environment_type:
                environment_type.values["stable"] = environment_type.Value(
                    "stable")
                environment_type.values["testing"] = environment_type.Value("testing")

    def create_json_file(self, file_name, data):
        """
        Creates migration file with a list of objects
        with the following structure:

        clientId1
        clientId2
        clientId3
        clientId4

        :returns: Nothing.
        """
        with open(file_name, 'w') as f:
            last_item = str(data[-1][0])
            logging.debug('last item in list: {}'.format(last_item))
            for row in data:
                client_id = str(row[0])
                if client_id == 'client_id':
                    continue
                elif client_id == last_item:
                    f.write('{}'.format(client_id) + '\n')
                else:
                    f.write('{}'.format(client_id) + '\n')
            f.write('\n')

    def create_resource(self):
        """
        Creates SOVETNIK_CLIENT_IDS_PACKAGE resource.

        :returns: Nothing.
        """
        path_to_pkg = self.get_path_to_pkg()
        path_to_work_dir = os.path.join(self.get_path_to_work_dir(), 'resources')
        path = os.path.join(path_to_work_dir, path_to_pkg)

        logging.debug('create resource in path: {}'.format(path))
        description = "Sovetnik client ids package"

        resource = SovetnikClientIdsPackage(
            self,
            path=path,
            description=description,
        )

        sdk2.ResourceData(resource).ready()

    def get_path_to_pkg(self):
        """ Returns path to package. """
        if self.pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(self.pkg_name))

    def create_tarfile(self, tarfile_name, file_name):
        """
        Create tarfile with a given name from a given file.

        :returns: Nothing.
        """
        with tarfile.open('{}.tar.gz'.format(tarfile_name), 'w:gz') as tar:
            arcname = 'resources/{}'.format(file_name)
            tar.add(file_name, arcname)

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()
        logging.debug("Start build: {}".format(path_to_work_dir))

        os.environ['NPM_PATH'] = self._get_npm_path()

        os.environ['REDIS_PASS'] = sdk2.Vault.data(
            'sovetnik-redis-password')

        os.environ['MONGODB_PASS'] = sdk2.Vault.data(
            'sovetnik-mongo-password')

        os.environ['NODE_ENV'] = self.Parameters.run_environment

        self.prepare_node_modules()

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call(['make', 'prepare-experiments'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

    def pack(self):
        """
        Creates a package inside a working directory.

        :returns: Nothing.
        """
        path_to_pkg = self.get_path_to_pkg()
        path_to_work_dir = os.path.join(self.get_path_to_work_dir(), 'resources')
        tarfile_name = 'sovetnik-client-ids'

        logging.debug('path to package: {}'.format(os.path.join(path_to_work_dir, path_to_pkg)))
        logging.debug('pack stage dir contents: {}'.format(
            os.listdir(path_to_work_dir)))

        self.create_tarfile(tarfile_name, os.path.join(path_to_work_dir, self.client_ids_file))

    def on_execute(self):
        logging.debug("Start Client ids task")

        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(
            db=self.Parameters.yt_host,
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_vault_token),
        )

        with self.memoize_stage.clone_git_repository(commit_on_entrance=False):
            self.clone_git_repository()

        with self.memoize_stage.yql_query(commit_on_entrance=False):
            request = yql_client.query(load_client_ids.format(interval_start=self.Parameters.yql_period),
                                       syntax_version=1)
            request.run()
            request.get_results(wait=True)

            if not request.is_success:
                raise errors.TaskError('YQL query failed')

            request.table.fetch_full_data()

            path_to_work_dir = os.path.join(self.get_path_to_work_dir(), 'resources')

            self.create_json_file(os.path.join(
                path_to_work_dir, self.client_ids_file), request.table.rows)

            logging.info('created file {}'.format(self.client_ids_file))
            logging.debug('execute stage dir contents: {}'.format(
                os.listdir(path_to_work_dir)))

        with self.memoize_stage.build(commit_on_entrance=False):
            self.build()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()

        logging.debug("End Client ids task")
