# coding: utf-8

import os
import tarfile
import logging

from sandbox import sdk2

from sandbox.projects.common.nanny import nanny

from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikResource

from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp


class SovetnikDomainUglifyPackage(SovetnikResource):
    """ A resource that contains tar.gz package with files for Sovetnik's landings service. """
    ttl = 60
    # custom attributes
    git_commit = sdk2.Attributes.String('The last Git commit', required=True)


class SovetnikBuildDomainUglifyPackage(nanny.ReleaseToNannyTask2, SovetnikBuild):
    """ A task that build SOVETNIK_DOMAIN_UGLIFY_PACKAGE resource from a Git repository. """
    pkg_name = 'sovetnik-domain-uglify.tar.gz'

    class Requirements(SovetnikBuild.Requirements):
        environments = (
            environments.NodeJS('10.14.2'),
            environments.GCCEnvironment(version='5.3.0'),
        )

    class Parameters(SovetnikBuild.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_parameters:
            git_release_branch = sdk2.parameters.String(
                label='Release branch',
                required=True,
                default='master',
            )
            git_ssh_url = sdk2.parameters.Url(
                label="Git remote SSH URL",
                required=True,
                default='git@github.yandex-team.ru:sovetnik/domain-uglify.git',
            )

    def get_path_to_pkg(self):
        """ Returns path to package. """
        if self.pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(self.pkg_name))

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()
        npm = self._get_npm_path()
        logging.debug("Start build: {}".format(path_to_work_dir))

        os.environ['NPM_PATH'] = npm

        self.prepare_node_modules()

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call([npm, 'run', 'build', '--if-present'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

    def publish(self):
        path_to_work_dir = self.get_path_to_work_dir()
        logging.debug("Start publish: {}".format(path_to_work_dir))

        os.environ['NPM_USER'] = 'robot-sovetnik'
        os.environ['NPM_EMAIL'] = 'robot-sovetnik@yandex-team.ru'
        os.environ['NPM_PASS'] = sdk2.Vault.data('sovetnik-npm-password')
        os.environ['NPM_REGISTRY'] = 'https://npm.yandex-team.ru'
        os.environ['NODE_ENV'] = 'development'
        os.environ['NPM_RC_PATH'] = '{}/.npmrc'.format(path_to_work_dir)

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call(['make', 'publish'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

    def pack(self):
        """
        Creates package of working directory.

        :returns: Nothing.
        """
        path_to_pkg = self.get_path_to_pkg()
        path_to_work_dir = self.get_path_to_work_dir()
        path_to_build_dir = os.path.join(path_to_work_dir, 'build')

        logging.debug('path to package: {}'.format(path_to_pkg))
        logging.debug('path to working directory: {}'.format(path_to_work_dir))
        logging.debug('path to build directory: {}'.format(path_to_build_dir))

        with tarfile.open(path_to_pkg, 'w:gz') as tar:
            for entry in os.listdir(path_to_build_dir):
                name = os.path.join(path_to_build_dir, entry)
                arcname = 'sovetnik-domain-uglify/{}'.format(entry)

                tar.add(name, arcname)

    def create_resource(self):
        """ Creates SOVETNIK_LANDINGS_PACKAGE resource. """
        path = self.get_path_to_pkg()
        git_commit = self.Context.git_commit
        description = "Sovetnik's domain uglify package #{}".format(git_commit)

        resource = SovetnikDomainUglifyPackage(
            self,
            path=path,
            git_commit=git_commit,
            description=description,
        )

        sdk2.ResourceData(resource).ready()

    def on_execute(self):
        """
        Clones a Git repository from remote source to a working directory.
        Builds Sovetnik's package.
        Creates package of working directory.
        Creates SOVETNIK_DOMAIN_UGLIFY_PACKAGE resource.

        :return: Nothing.
        """
        super(SovetnikBuildDomainUglifyPackage, self).on_execute()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.publish()
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()
