# coding: utf-8

import os
import logging

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.common.nanny import nanny

from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikResource


class SovetnikDomainsPackage(SovetnikResource):
    """ A resource that contains tar.gz package with files for Sovetnik's domains service. """
    ttl = 60
    # custom attributes
    git_commit = sdk2.Attributes.String('The last Git commit', required=True)


class SovetnikBuildDomainsPackage(nanny.ReleaseToNannyTask2, SovetnikBuild):
    """ A task that builds SOVETNIK_DOMAINS_PACKAGE resource from a Git repository. """
    pkg_name = 'sovetnik-domains.tar.gz'

    class Requirements(SovetnikBuild.Requirements):
        environments = (
            environments.NodeJS('8.9.4'),
            environments.GCCEnvironment(version='5.3.0'),
        )

    class Parameters(SovetnikBuild.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_parameters:
            git_release_branch = sdk2.parameters.String(
                label='Release branch',
                required=True,
                default='master',
            )
            git_ssh_url = sdk2.parameters.Url(
                label="Git remote SSH URL",
                required=True,
                default='git@github.yandex-team.ru:sovetnik/sovetnik-domains.git',
            )

    def get_path_to_pkg(self):
        """ Returns path to package. """
        if self.pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(self.pkg_name))

    def pack(self):
        """
        Creates package of working directory.

        :returns: Nothing.
        """
        path_to_pkg = self.get_path_to_pkg()
        path_to_work_dir = self.get_path_to_work_dir()

        logging.debug('path to working directory: {}'.format(path_to_work_dir))
        logging.debug('path to package: {}'.format(path_to_pkg))

        os.environ['PACKAGE_ARCHIVE_NAME'] = path_to_pkg

        with sdk2.helpers.ProcessLog(self, logger='archive') as pl:
            sp.check_call(['make', 'archive'], cwd=path_to_work_dir, stdout=pl.stdout, stderr=pl.stderr)

    def create_resource(self):
        """ Creates SOVETNIK_DOMAINS_PACKAGE resource. """
        path = self.get_path_to_pkg()
        git_commit = self.Context.git_commit
        description = "Sovetnik's domains package #{}".format(git_commit)

        resource = SovetnikDomainsPackage(
            self,
            path=path,
            git_commit=git_commit,
            description=description,
        )

        sdk2.ResourceData(resource).ready()

    def on_execute(self):
        """
        Clones a Git repository from remote source to a working directory.
        Builds Sovetnik's domains package.
        Creates package of working directory.
        Creates SOVETNIK_DOMAINS_PACKAGE resource.

        :return: Nothing.
        """
        super(SovetnikBuildDomainsPackage, self).on_execute()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()
