import os
import logging
import tarfile
import requests

from sandbox import sdk2

from sandbox.projects.common.nanny import nanny

from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikGeodataResource, SovetnikCurrencyResource


class SovetnikBuildExternalResourcesPackage(nanny.ReleaseToNannyTask2, SovetnikBuild):
    """ Task to retrieve all external resources """

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Resource type", required=True) as resource_type:
            resource_type.values["geo"] = resource_type.Value(
                "SovetnikGeodataResource")
            resource_type.values["money"] = resource_type.Value(
                "SovetnikCurrencyResource")

        with sdk2.parameters.String("Environment type", required=True) as environment_type:
            environment_type.values["stable"] = environment_type.Value(
                "stable")
            environment_type.values["testing"] = environment_type.Value(
                "testing")

        with sdk2.parameters.Group('Resource parameters') as yt_parameters:
            resource_url = sdk2.parameters.String(
                label='Url for external resource',
                required=True,
                default='http://example.com/geodump.json'
            )
            resource_name = sdk2.parameters.String(
                label='Name of the resource to create',
                required=True,
                default='geodump.json'
            )
            resource_path = sdk2.parameters.String(
                label='Path  of the  resource (the same as in backend)',
                required=True,
                default='resources/geodump.json'
            )

        with sdk2.parameters.Group("Additional parameters") as additional_block:
            create_nanny_release = sdk2.parameters.Bool(
                "Create Nanny release on success", default=True)

    def download_file(self, file_name, data):
        logging.debug('start downloading')

        result = requests.get(data)
        path = file_name

        with open(path, 'wb') as f:
            f.write(result.content)

        logging.debug('finish downloading')

    def create_resource(self):
        """
        Creates geoexport resource resource.

        :returns: Nothing.
        """
        resource_name = self.Parameters.resource_name
        path_to_resource = self.get_path_to_pkg('{}.tar.gz'.format(resource_name))
        resource_type = self.Parameters.resource_type

        logging.debug('create resource in path: {}'.format(path_to_resource))
        logging.debug('resource_type: {}'.format(resource_type))

        if resource_type == 'geo' or resource_type == 'SovetnikGeodataResource':
            resource = SovetnikGeodataResource(
                self,
                path=path_to_resource,
                description="Sovetnik external package with {}".format(resource_name),
            )
            sdk2.ResourceData(resource).ready()
        elif resource_type == 'money' or resource_type == 'SovetnikCurrencyResource':
            resource = SovetnikCurrencyResource(
                self,
                path=path_to_resource,
                description="Sovetnik external package with {}".format(resource_name),
            )
            sdk2.ResourceData(resource).ready()

    def get_path_to_pkg(self, pkg_name):
        """ Returns path to package. """
        if pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(pkg_name))

    def create_tarfile(self, file_name, target_name):
        """
        Create tarfile with a given name from a given file.

        :returns: Nothing.
        """
        with tarfile.open('{}.tar.gz'.format(file_name), 'w:gz') as tar:
            arcname = '{}'.format(target_name)  # target_name=resources/geodupm.json
            tar.add(file_name, arcname)  # file_name=/56789/4353t/geodump.json

    def pack(self):
        """
        Creates a package inside a working directory.

        :returns: Nothing.
        """
        path_to_work_dir = os.getcwd()

        logging.debug('current workdir is: {}'.format(path_to_work_dir))

        logging.debug('pack stage dir contents: {}'.format(
            os.listdir(path_to_work_dir)))

        self.create_tarfile(self.Parameters.resource_name, self.Parameters.resource_path)

    def on_execute(self):
        logging.debug("Start Building SovetnikBuildExternalResources")

        self.download_file(self.Parameters.resource_name, self.Parameters.resource_url)

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()

        logging.debug("End  Building SovetnikBuildExternalResources")

    def on_success(self, prev_status):
        SovetnikBuild.on_success(self, prev_status)

        if self.owner != 'SOVETNIK':
            logging.warning("Owner is %s. Skip any futher action", self.owner)
        else:
            if self.Parameters.create_nanny_release:
                nanny.ReleaseToNannyTask2.on_release(self, dict(
                    release_comments=None,
                    release_status=self.Parameters.environment_type,
                    release_subject="Auto release from {}".format(self.Parameters.resource_name),
                    releaser=self.author
                ))
