# coding: utf-8

import os
import tarfile
import logging

from sandbox import sdk2

from sandbox.projects.common.nanny import nanny

from sandbox.projects.Sovetnik.SovetnikBuildArcadia import SovetnikBuildArcadia
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikLandingsPackage

from sandbox.sandboxsdk import environments


class SovetnikBuildLandingsPackageArcadia(nanny.ReleaseToNannyTask2, SovetnikBuildArcadia):
    """ A task that build SOVETNIK_LANDINGS_PACKAGE resource from a Git repository. """
    pkg_name = 'sovetnik-landings.tar.gz'

    class Requirements(SovetnikBuildArcadia.Requirements):
        environments = (
            environments.NodeJS('10.14.2'),
            environments.GCCEnvironment(version='5.3.0'),
        )

    class Parameters(SovetnikBuildArcadia.Parameters):
        with sdk2.parameters.Group('Настройки репозитория') as repo_block:
            arc_path = sdk2.parameters.String(
                'Arcadia url:',
                required=True,
                default_value='arcadia-arc:/#trunk',
                description='Arc путь. EX: arcadia-arc:/#trunk или arcadia-arc:/#users/bogdansky/SOVETNIK-4106'
            )

            repo_path = sdk2.parameters.String(
                'Repo path:',
                required=True,
                description='Путь к репозиторию Sovetnik Landings. EX: market/sovetnik/landings',
                default_value='market/sovetnik/landings'
            )

    def get_path_to_pkg(self):
        """ Returns path to package. """
        if self.pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(self.pkg_name))

    def build(self):
        """
        Build Sovetnik's landings service.

        :returns: Nothing.
        """
        self.prepare_node_modules()

        super(SovetnikBuildLandingsPackageArcadia, self).build()

    def pack(self):
        """
        Creates package of working directory.

        :returns: Nothing.
        """
        path_to_pkg = self.get_path_to_pkg()
        path_to_work_dir = self.get_path_to_work_dir()
        path_to_build_dir = os.path.join(path_to_work_dir, 'build')

        logging.debug('path to package: {}'.format(path_to_pkg))
        logging.debug('path to working directory: {}'.format(path_to_work_dir))
        logging.debug('path to build directory: {}'.format(path_to_build_dir))

        with tarfile.open(path_to_pkg, 'w:gz') as tar:
            for entry in os.listdir(path_to_build_dir):
                name = os.path.join(path_to_build_dir, entry)
                arcname = 'sovetnik-static/landing/{}'.format(entry)

                tar.add(name, arcname)

    def create_resource(self):
        """ Creates SOVETNIK_LANDINGS_PACKAGE resource. """
        path = self.get_path_to_pkg()
        description = "Sovetnik's landings package"

        resource = SovetnikLandingsPackage(
            self,
            path=path,
            description=description,
        )

        sdk2.ResourceData(resource).ready()

    def on_execute(self):
        """
        Clones a Git repository from remote source to a working directory.
        Builds Sovetnik's package.
        Creates package of working directory.
        Creates SOVETNIK_LANDINGS_PACKAGE resource.

        :return: Nothing.
        """
        super(SovetnikBuildLandingsPackageArcadia, self).on_execute()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()
