import os
import logging
import tarfile

import sandbox.common.errors as errors

from sandbox import sdk2

from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import environments

from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikResource
from sandbox.projects.Sovetnik.SovetnikBuildMarketShopsPackage.yql_query import load_shops


class SovetnikMarketShopsPackage(SovetnikResource):
    """ Resource that contains a list of Market shops. """
    ttl = 5


class SovetnikBuildMarketShopsPackage(nanny.ReleaseToNannyTask2, SovetnikBuild):
    """ Task to retrieve all Market shops from YT. """
    pkg_name = 'sovetnik-shops.tar.gz'
    market_shops_file = 'market_shops.json'

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        environments = [
            environments.PipEnvironment('yandex-yt', use_wheel=True),
            environments.PipEnvironment('yql', use_wheel=True),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yql_vault_token = sdk2.parameters.String(
                'Your YQL token name in Vault',
                default='YQL_TOKEN',
                required=True
            )

            with sdk2.parameters.RadioGroup('YT host') as yt_host:
                yt_host.values['hahn'] = yt_host.Value(
                    value='Hahn', default=True)
                yt_host.values['banach'] = yt_host.Value(value='Banach')

        with sdk2.parameters.String("Environment type", required=True) as environment_type:
            environment_type.values["stable"] = environment_type.Value(
                "stable")
            environment_type.values["testing"] = environment_type.Value(
                "testing")

        with sdk2.parameters.Group("Additional parameters") as additional_block:
            create_nanny_release = sdk2.parameters.Bool(
                "Create Nanny release on success", default=True)

    def create_json_file(self, file_name, data):
        """
        Creates json file with an array of objects
        with the following structure:

        [
            { "host": "market_shop_domain_1" },
            { "host": "market_shop_domain_2" },
            { "host": "market_shop_domain_N" }
        ]

        :returns: Nothing.
        """
        with open(file_name, 'w') as f:
            f.write('[\n')
            last_item = str(data[-1][0])
            logging.debug('last item in list: {}'.format(last_item))
            for row in data:
                host = str(row[0])
                if host == 'None':
                    continue
                elif host == last_item:
                    f.write('{' + '"host": "{}"'.format(host) + '}\n')
                else:
                    f.write('{' + '"host": "{}"'.format(host) + '},\n')
            f.write(']\n')

    def create_resource(self):
        """
        Creates SOVETNIK_SHOPS_PACKAGE resource.

        :returns: Nothing.
        """
        path = self.get_path_to_pkg()
        description = "Sovetnik shops package"

        resource = SovetnikMarketShopsPackage(
            self,
            path=path,
            description=description,
        )

        sdk2.ResourceData(resource).ready()

    def get_path_to_pkg(self):
        """ Returns path to package. """
        if self.pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(self.pkg_name))

    def create_tarfile(self, tarfile_name, file_name):
        """
        Create tarfile with a given name from a given file.

        :returns: Nothing.
        """
        with tarfile.open('{}.tar.gz'.format(tarfile_name), 'w:gz') as tar:
            arcname = 'resources/{}'.format(file_name)
            tar.add(file_name, arcname)

    def pack(self):
        """
        Creates a package inside a working directory.

        :returns: Nothing.
        """
        path_to_pkg = self.get_path_to_pkg()
        tarfile_name = 'sovetnik-shops'

        logging.debug('path to package: {}'.format(path_to_pkg))
        logging.debug('pack stage dir contents: {}'.format(
            os.listdir(os.getcwd())))

        self.create_tarfile(tarfile_name, self.market_shops_file)

    def on_execute(self):
        logging.debug("Start Market shops task")

        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(
            db=self.Parameters.yt_host,
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_vault_token),
        )

        with self.memoize_stage.yql_query(commit_on_entrance=False):
            request = yql_client.query(load_shops, syntax_version=1)
            request.run()
            request.get_results(wait=True)

            if not request.is_success:
                raise errors.TaskError('YQL query failed')

            request.table.fetch_full_data()

            home_dir = os.getcwd()

            self.create_json_file(os.path.join(
                home_dir, self.market_shops_file), request.table.rows)

            logging.info('created file {}'.format(self.market_shops_file))
            logging.debug('execute stage dir contents: {}'.format(
                os.listdir(os.getcwd())))

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()

        logging.debug("End Market shops task")

    def on_success(self, prev_status):
        SovetnikBuild.on_success(self, prev_status)

        if self.owner != 'SOVETNIK':
            logging.warning("Owner is %s. Skip any futher action", self.owner)
        else:
            if self.Parameters.create_nanny_release:
                nanny.ReleaseToNannyTask2.on_release(self, dict(
                    release_comments=None,
                    release_status=self.Parameters.environment_type,
                    release_subject="Auto release from sandbox Market shops task",
                    releaser=self.author
                ))
