import os
import logging
import tarfile
import sandbox.common.types.misc as ctm

from sandbox import sdk2

from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikMobileUCPackage, SovetnikMobileYabroPackage, SovetnikMobileSmartSearchIphonePackage, SovetnikMobileSmartSearchAndroidPackage


class SovetnikBuildMobileScripts(SovetnikBuild):
    pkg_name = 'sovetnik-mobile-scripts.tar.gz'

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4 * 1024
        disk_space = 4 * 1024
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('10.14.2'),
            environments.GCCEnvironment(version='5.3.0'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_parameters:
            git_release_branch = sdk2.parameters.String(
                label='Release branch',
                required=True,
                default='master',
            )
            git_ssh_url = sdk2.parameters.Url(
                label='Git remote SSH URL',
                required=True,
                default='git@github.yandex-team.ru:sovetnik/mobile-app.git',
            )

        with sdk2.parameters.Group('SSH parameters') as ssh_parameters:
            ssh_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's SSH private key",
                required=True,
                default='SOVETNIK',
            )
            ssh_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's SSH private key",
                required=True,
                default='robot-sovetnik-ssh',
            )

        with sdk2.parameters.Group('Run parameters') as run_parameters:
            with sdk2.parameters.RadioGroup('Run environment') as run_environment:
                run_environment.values['development'] = run_environment.Value(
                    value='development', default=True)
                run_environment.values['production'] = run_environment.Value(value='production')

    def create_resource(self):
        """
        Creates SOVETNIK_MOBILE_SCRIPTS resource.

        :returns: Nothing.
        """
        path_to_uc = self.get_path_to_pkg('uc_browser.tar.gz')
        path_to_yabro = self.get_path_to_pkg('yabro.tar.gz')
        path_to_smart_search_iphone = self.get_path_to_pkg('smart_search_iphone.tar.gz')
        path_to_smart_search_android = self.get_path_to_pkg('smart_search_android.tar.gz')

        logging.debug('create resource in path uc: {}'.format(path_to_uc))
        logging.debug('create resource in path yabro: {}'.format(path_to_yabro))

        resource = SovetnikMobileYabroPackage(
            self,
            path=path_to_yabro,
            description="Sovetnik mobile scripts package for Yabro",
        )

        sdk2.ResourceData(resource).ready()

        resource = SovetnikMobileUCPackage(
            self,
            path=path_to_uc,
            description="Sovetnik mobile scripts package for UC Web",
        )

        sdk2.ResourceData(resource).ready()

        resource = SovetnikMobileSmartSearchIphonePackage(
            self,
            path=path_to_smart_search_iphone,
            description="Sovetnik mobile scripts package for smartsearch iphone"
        )

        sdk2.ResourceData(resource).ready()

        resource = SovetnikMobileSmartSearchAndroidPackage(
            self,
            path=path_to_smart_search_android,
            description="Sovetnik mobile scripts package for smartsearch android"
        )

        sdk2.ResourceData(resource).ready()

    def get_path_to_pkg(self, pkg_name):
        """ Returns path to package. """
        if pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(pkg_name))

    def create_tarfile(self, tarfile_name, file_name):
        """
        Create tarfile with a given name from a given file.

        :returns: Nothing.
        """
        with tarfile.open('{}.tar.gz'.format(tarfile_name), 'w:gz') as tar:
            tar.add(file_name, tarfile_name)

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()
        logging.debug("Start build: {}".format(path_to_work_dir))

        os.environ['NPM_PATH'] = self._get_npm_path()

        self.prepare_node_modules()

        os.environ['NODE_ENV'] = self.Parameters.run_environment

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call([os.environ['NPM_PATH'], 'run', 'build'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

    def pack(self):
        """
        Creates a package inside a working directory.

        :returns: Nothing.
        """
        path_to_work_dir = os.path.join(self.get_path_to_work_dir(), 'build')

        logging.debug('pack stage dir contents: {}'.format(
            os.listdir(path_to_work_dir)))

        directories = os.listdir(path_to_work_dir)

        for dirname in directories:
            self.create_tarfile(dirname, os.path.join(path_to_work_dir, dirname))

    def on_execute(self):
        logging.debug("Start Building SovetnikBuildMobileScripts")

        super(SovetnikBuildMobileScripts, self).on_execute()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()

        logging.debug("End  Building SovetnikBuildMobileScripts")
