# coding: utf-8

import os
import re
import tarfile
import logging
import sandbox.common.types.misc as ctm
from sandbox import sdk2

from sandbox.projects.common.nanny import nanny

from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.Sovetnik.SovetnikBuildArcadia import SovetnikBuildArcadia
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikRedirPackage
from sandbox.sandboxsdk import environments


class SovetnikBuildRedirPackageArcadia(nanny.ReleaseToNannyTask2, SovetnikBuildArcadia):
    """ A task that builds SOVETNIK_REDIR_PACKAGE resource from Arcadia repository. """
    pkg_name = 'sovetnik-redir.tar.gz'
    exclude_patterns = [
        r"^sovetnik-redir\/\.editorconfig$",
        r"^sovetnik-redir\/\.svn$",
        r"^sovetnik-redir\/\.eslintrc.+$",
        r"^sovetnik-redir\/\.eslintignore$",
        r"^sovetnik-redir\/\.prettier.+$",
        r"^sovetnik-redir\/jest.+$",
        r"^sovetnik-redir\/gruntfile\.js$",
        r"^sovetnik-redir\/[^\/]*\.log$",
        r"^sovetnik-redir\/[^\/]*\.map$",
    ]

    class Requirements(SovetnikBuildArcadia.Requirements):
        dns = ctm.DnsType.DNS64

        environments = (
            environments.NodeJS('8.12.0'),
            environments.GCCEnvironment(),
        )

    class Parameters(SovetnikBuildArcadia.Parameters):
        with sdk2.parameters.Group('Настройки репозитория') as repo_block:
            arc_path = sdk2.parameters.String(
                'Arcadia url:',
                required=True,
                default_value='arcadia-arc:/#trunk',
                description='Arc путь. EX: arcadia-arc:/#trunk или arcadia-arc:/#users/bogdansky/SOVETNIK-4106'
            )

            repo_path = sdk2.parameters.String(
                'Repo path:',
                required=True,
                description='Путь к репозиторию Sovetnik Redir. EX: market/sovetnik/redir',
                default_value='market/sovetnik/redir'
            )

    def get_path_to_pkg(self):
        """ Returns path to package. """
        if self.pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(self.pkg_name))

    def _exclude_filter(self, tarinfo):
        """
        A function that takes a TarInfo object argument and returns the TarInfo object.
        If it instead returns None the TarInfo object will be excluded from the archive.
        """
        if self.exclude_patterns is None:
            raise AttributeError('Exclude patterns are not defined')

        for pattern in self.exclude_patterns:
            if re.search(pattern, tarinfo.name):
                return None

        return tarinfo

    def pack(self):
        """
        Creates package of working directory.

        :returns: Nothing.
        """
        path_to_pkg = self.get_path_to_pkg()
        path_to_work_dir = self.get_path_to_work_dir()

        logging.debug('path to package: {}'.format(path_to_pkg))
        logging.debug('path to working directory: {}'.format(path_to_work_dir))

        with tarfile.open(path_to_pkg, 'w:gz') as tar:
            for entry in os.listdir(path_to_work_dir):
                name = os.path.join(path_to_work_dir, entry)
                arcname = 'sovetnik-redir/{}'.format(entry)
                tar.add(name, arcname, filter=self._exclude_filter)

    def create_resource(self):
        """ Creates SOVETNIK_REDIR_PACKAGE resource. """
        path = self.get_path_to_pkg()
        description = "Sovetnik's redirector package"

        resource = SovetnikRedirPackage(
            self,
            path=path,
            description=description,
        )

        sdk2.ResourceData(resource).ready()

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        logging.debug('NOW START BUILD')

        path_to_work_dir = self.get_path_to_work_dir()
        logging.debug('path_to_work_dir = {}'.format(path_to_work_dir))

        os.environ['NPM_PATH'] = self._get_npm_path()
        os.environ['SOVETNIK_CRYPTO'] = sdk2.Vault.data(
            'sovetnik-crypto-password')
        os.environ['CRYPTO'] = sdk2.Vault.data(
            'sovetnik-crypto-password')

        self.prepare_node_modules()

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call(['make', 'build'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

    def on_execute(self):
        """
        Clones Arcadia repository from remote source to a working directory.
        Builds Sovetnik's package.
        Creates package of working directory.
        Creates SOVETNIK_REDIR_PACKAGE resource.

        :return: Nothing.
        """
        super(SovetnikBuildRedirPackageArcadia, self).on_execute()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()
