# coding: utf-8

import os
import tarfile
import logging

from sandbox import sdk2

from sandbox.projects.common.nanny import nanny

from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild
from sandbox.projects.Sovetnik.SovetnikResource import SovetnikSettingsPagePackage

from sandbox.sandboxsdk import environments


class SovetnikBuildSettingsPagePackage(nanny.ReleaseToNannyTask2, SovetnikBuild):
    """ A task that build SOVETNIK_SETTINGS_PAGE_PACKAGE resource from a Git repository. """
    pkg_name = 'sovetnik-settings.tar.gz'

    class Requirements(SovetnikBuild.Requirements):
        environments = (
            environments.NodeJS('10.14.2'),
            environments.GCCEnvironment(version='5.3.0'),
        )

    class Parameters(SovetnikBuild.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_parameters:
            git_release_branch = sdk2.parameters.String(
                label='Release branch',
                required=True,
                default='master',
            )
            git_ssh_url = sdk2.parameters.Url(
                label="Git remote SSH URL",
                required=True,
                default='git@github.yandex-team.ru:sovetnik/settings.git',
            )

    def get_path_to_pkg(self):
        """ Returns path to package. """
        if self.pkg_name is None:
            raise AttributeError('Package name is not defined')

        return str(self.path(self.pkg_name))

    def build(self):
        """
        Build Sovetnik's settings service.

        :returns: Nothing.
        """
        self.prepare_node_modules()

        super(SovetnikBuildSettingsPagePackage, self).build()

    def pack(self):
        """
        Creates package of working directory.

        :returns: Nothing.
        """
        path_to_pkg = self.get_path_to_pkg()
        path_to_work_dir = self.get_path_to_work_dir()
        path_to_build_dir = os.path.join(path_to_work_dir, 'build/desktop')

        logging.debug('path to package: {}'.format(path_to_pkg))
        logging.debug('path to working directory: {}'.format(path_to_work_dir))
        logging.debug('path to build directory: {}'.format(path_to_build_dir))

        with tarfile.open(path_to_pkg, 'w:gz') as tar:
            for entry in os.listdir(path_to_build_dir):
                name = os.path.join(path_to_build_dir, entry)
                arcname = 'sovetnik-static/sovetnik/{}'.format(entry)

                tar.add(name, arcname)

    def create_resource(self):
        """ Creates SOVETNIK_SETTINGS_PAGE_PACKAGE resource. """
        path = self.get_path_to_pkg()
        git_commit = self.Context.git_commit
        description = "Sovetnik's settings package #{}".format(git_commit)

        resource = SovetnikSettingsPagePackage(
            self,
            path=path,
            description=description,
        )

        sdk2.ResourceData(resource).ready()

    def on_execute(self):
        """
        Clones a Git repository from remote source to a working directory.
        Builds Sovetnik's package.
        Creates package of working directory.
        Creates SOVETNIK_SETTINGS_PAGE_PACKAGE resource.

        :return: Nothing.
        """
        super(SovetnikBuildSettingsPagePackage, self).on_execute()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.pack()

        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()
