# coding: utf-8

import os
import shutil
import logging
import yaml
import json
import re
import time
import glob

from sandbox import sdk2
from sandbox.common.types import misc
from sandbox.common.errors import TaskFailure, TemporaryError, SubprocessError
from sandbox.projects.common.debpkg import DebRelease
from sandbox.projects.common.arcadia.sdk import mount_arc_path

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.sandboxsdk.process import run_process
from sandbox.common.utils import server_url

from sandbox.projects.Statbox import STATBOX_PBUILDER_IMAGE
from dist import Dist
import conductor

import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

PACKAGES = {
    'static': 'yandex-sovetnik-www-static'
}


# region Exceptions
class ChangelogEmpty(Exception):
    pass


class PushFailed(Exception):
    pass


# endregion


class SovetnikStaticDebPackage(sdk2.Resource):
    releasable = True


class SovetnikDebuildPackage(sdk2.Task):
    class Context(sdk2.Task.Context):
        workdir_mobile = 'workdir'
        workdir = 'workdir'
        build_dir = 'build_dir'
        build_dmove = ''
        build_source = ''
        build_packages = ''
        build_version = ''

        pb_image = ''

        build_options = {
            'distr': 'precise',
            'hookdir': 'hooks.d',
            'repo': {
                'name': 'statbox-common',
                'check_if_exist': True,
                'dupload': True,
                'user': 'robot-statinfra'
            },
            'conductor': {
                'create_ticket': True,
                'branch': 'testing',
                'projects': ['STATBOX'],
            }
        }

        baked_build_options = {}

    # region Requirements & Parameters
    class Requirements(sdk2.Task.Requirements):
        privileged = True
        dns = misc.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('SSH params') as ssh_parameters:
            ssh_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's SSH private key",
                required=True,
                default='SOVETNIK',
            )
            ssh_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's SSH private key",
                required=True,
                default='robot-sovetnik-ssh',
            )

        with sdk2.parameters.Group('Conductor params') as conductor_parameters:
            conductor_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's Conductor private key",
                required=True,
                default='SOVETNIK',
            )
            conductor_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's Conductor private key",
                required=True,
                default='robot-sovetnik-real-conductor-oauth',
            )

        with sdk2.parameters.Group('GPG keys params') as gpg_parameters:
            gpg_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's GPG keys",
                required=True,
                default='SOVETNIK',
            )
            gpg_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's GPG keys",
                required=True,
                default='robot-sovetnik',
            )

        with sdk2.parameters.Group('Package name') as package_name_params:
            result_package_name = sdk2.parameters.String(
                label='Name of package to build',
                required=True,
                default='yandex-sovetnik-static',
            )

        with sdk2.parameters.Group('YABro sign params') as yabro_parameters:
            yabro_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's yabro key",
                required=True,
                default='SOVETNIK',
            )
            yabro_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's yabro key",
                required=True,
                default='ya-bro-password',
            )
        with sdk2.parameters.Group('YT token') as yt_parameters:
            yt_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's YT Token",
                required=True,
                default='SOVETNIK',
            )
            env_YT_OAUTH_ACCESS_TOKEN = sdk2.parameters.String(
                label="Vault name of YT_OAUTH_ACCESS_TOKEN",
                required=True,
                default='env.YT_OAUTH_ACCESS_TOKEN',
            )

        with sdk2.parameters.Group('Настройки репозитория ') as repo_static_block:
            arc_path_static = sdk2.parameters.String(
                'Arc url:',
                required=True,
                default_value='arcadia-arc:/#trunk',
                description='Arc путь. EX: arcadia-arc:/#trunk или arcadia-arc:/#users/bogdansky/SOVETNIK-4106'
            )

            repo_path_static = sdk2.parameters.String(
                'Static repo path in Arcadia:',
                required=True,
                description='Путь к репозиторию Sovetnik Static. EX: market/sovetnik/static',
                default_value='market/sovetnik/static'
            )

            repo_path_mobile = sdk2.parameters.String(
                'Mobile repo path in Arcadia:',
                required=True,
                description='Путь к репозиторию Sovetnik Mobile App. EX: market/sovetnik/mobile-app',
                default_value='market/sovetnik/mobile-app'
            )

        with sdk2.parameters.Group('Build params') as build_params:
            build_param_path = sdk2.parameters.String(
                'Path to directory with .build.yml',
                default=''
            )

            build_distr = sdk2.parameters.String(
                'Name of image for pdebuild',
                default='trusty'
            )

            build_hooks = sdk2.parameters.String(
                'Dir name for pdebuild hooks',
                default='hooks.d'
            )

            build_override_params = sdk2.parameters.Bool(
                'Override build params',
                default=True
            )

            with build_override_params.value[True]:
                build_repo_user = sdk2.parameters.String(
                    'Name of user for sign and dupload',
                    default='robot-sovetnik'
                )

                build_dupload = sdk2.parameters.Bool(
                    'Dupload packages',
                    default=True
                )

                build_repo_check_if_exists = sdk2.parameters.Bool(
                    'Check if exist in repo',
                    default=True
                )

                build_repo_name = sdk2.parameters.String(
                    'Repo name',
                    default='verstka'
                )

                build_conductor_create_ticket = sdk2.parameters.Bool(
                    'Create ticket in conductor',
                    default=True
                )

                with sdk2.parameters.String('Branch in conductor', default='testing') as build_conductor_branch:
                    for branch in ['unstable', 'testing', 'prestable', 'stable', 'hotfix', 'fallback']:
                        build_conductor_branch.values[branch] = build_conductor_branch.Value(branch)

                build_conductor_projects = sdk2.parameters.String(
                    'Projects for tickets in conductor',
                    default='PORTAL'
                )

    # endregion

    # region Utils
    def get_path_to_work_dir(self):
        if self.Context.workdir is None:
            raise AttributeError('Name of working directory is not defined')

        return str(self.path(self.Context.workdir))

    def get_abs_path(self):
        return str(self.path(''))

    def checkout_arcadia_repository(self, arc_path, repo_path):
        logging.debug('Checkout arcadia repo {}'.format(repo_path))
        with mount_arc_path(arc_path, use_arc_instead_of_aapi=True) as arcadia:
            self.Context.workdir = os.path.join(arcadia, repo_path)
            self.Context.workdir_mobile = os.path.join(arcadia, self.Parameters.repo_path_mobile)

            static_mobile = os.path.join(self.Context.workdir, 'mobile-app')
            logging.debug('Copy mobile files from mobile to static > {} to {}'.format(self.Context.workdir_mobile,
                                                                                      static_mobile))
            shutil.copytree(self.Context.workdir_mobile, static_mobile)

            self.Context.build_dir = os.path.join(str(self.path('')).rstrip('/'), 'build_dir')
            logging.debug('Copy all from mounted arc to local build dir > {} to {}'.format(self.Context.workdir,
                                                                                           self.Context.build_dir))
            shutil.copytree(self.Context.workdir, self.Context.build_dir)
            logging.debug('Copy hidden .build.yml file from arcadia to build dir')
            shutil.copy(os.path.join(self.Context.workdir.rstrip('/'), '.build.yml'),
                        os.path.join(self.Context.build_dir.rstrip('/'), '.build.yml'))

            self.Context.workdir = self.Context.build_dir
            logging.debug("Build dir content > ")
            logging.debug(glob.glob('{build_dir}/*'.format(build_dir=self.Context.build_dir)))

    def read_build_file(self, build_file):
        if not self.Parameters.build_override_params:
            logging.info('Params overrided')
            return self.Context.build_options

        try:
            with open(build_file, 'r') as stream:
                logging.info('Try to read build.yml')
                return yaml.load(stream)
        except IOError:
            logging.info('build.yml not found')
            raise TaskFailure('build.yml not found, and not overrided in params')

    def enrich_build_options(self):
        repo_params = self.Context.build_options.get('repo')
        conductor_params = self.Context.build_options.get('conductor')

        return {
            'distr': self.Parameters.build_distr or self.Context.build_options.get('distr'),
            'repo': {
                'name': self.Parameters.build_repo_name or repo_params.get('name'),
                'check_if_exist': self.Parameters.build_repo_check_if_exists or repo_params.get('check_if_exist'),
                'dupload': self.Parameters.build_dupload or repo_params.get('dupload'),
                'user': self.Parameters.build_repo_user or repo_params.get('user')
            },
            'conductor': {
                'create_ticket': self.Parameters.build_conductor_create_ticket or conductor_params.get('create_ticket'),
                'branch': self.Parameters.build_conductor_branch or conductor_params.get('branch'),
                'projects': self.Parameters.build_conductor_projects or conductor_params.get('projects')
            }
        }

    def prepare_environ(self):
        logging.info('->[PrepareEnviron] Start')

        for v in ['TEMP', 'TMPDIR', 'TMP']:
            if v in os.environ:
                del os.environ[v]

        repo_user = self.Context.baked_build_options.get('repo', {}).get('user')
        os.environ['DEBFULLNAME'] = repo_user
        os.environ['EMAIL'] = '{}@yandex-team.ru'.format(repo_user)
        os.environ['YA_BRO_PASSWORD'] = sdk2.Vault.data(
            self.Parameters.yabro_vault_owner,
            self.Parameters.yabro_vault_name
        )

        os.environ['YT_OAUTH_ACCESS_TOKEN'] = sdk2.Vault.data(
            self.Parameters.yt_vault_owner,
            self.Parameters.env_YT_OAUTH_ACCESS_TOKEN,
        )

        # it used for arcadia
        os.environ['SVN_SSH'] = 'ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null'

        logging.info('Environ: ')
        logging.info(os.environ)

        run_process(['sudo', 'apt-get', 'update'])

        run_process(
            [
                'sudo', 'apt-get', 'install', 'gnupg2', 'pbuilder',
                '-q0', '--assume-yes', '--force-yes', '-o', "Dpkg::Options::='--force-confdef'"
            ],
            shell=True, log_prefix='install_pbuilder'
        )

        run_process(
            [
                'sudo', 'apt-get', 'install', 'gnupg2', '--assume-yes', '--force-yes',
            ],
            shell=True, log_prefix='install_gnupg'
        )

        run_process(
            [
                'sudo', 'apt-get', 'install', 'devscripts', '--assume-yes', '--force-yes',
            ],
            shell=True, log_prefix='install_dev_scripts'
        )

        logging.info('->[PrepareEnviron] End')

    def get_base_tgz(self, ubuntu_version='precise'):
        logging.info('->[GetBaseTGZ] Start')
        resource = STATBOX_PBUILDER_IMAGE.find(state=ctr.State.READY, attrs={'release': ubuntu_version}).first()
        resource_data = sdk2.ResourceData(resource)

        logging.info('Resource:')
        logging.info(resource)
        logging.info(resource_data)
        logging.info(resource_data.path)

        if resource is None:
            raise TaskFailure('Cannot find suitable resource')

        logging.info('->[GetBaseTGZ] End')
        return resource_data

    def linkify(self, url):
        return '<a href={}>{}</a>'.format(url, url)

    # endregion

    # region Hooks
    def on_prepare(self):
        logging.info('[Prepare] Start')
        self.checkout_arcadia_repository(self.Parameters.arc_path_static, self.Parameters.repo_path_static)
        logging.debug('Workdir is {}'.format(self.get_path_to_work_dir()))
        logging.debug('What is in workdir after checkout before build')
        logging.debug(glob.glob('{work_dir}/*'.format(
            work_dir=self.get_path_to_work_dir())))
        build_file = os.path.join(
            self.get_path_to_work_dir().rstrip('/'),
            self.Parameters.build_param_path.strip('/'),
            '.build.yml'
        )

        logging.info('Path to .build.yml: {}'.format(build_file))

        self.Context.build_options = self.read_build_file(build_file)
        self.Context.baked_build_options = self.enrich_build_options()
        self.Context.build_dmove = self.Context.baked_build_options.get('conductor', {}).get('branch')

        logging.info('Build options:')
        logging.info(json.dumps(self.Context.build_options, indent=4))
        logging.info('\nBaked options:')
        logging.info(json.dumps(self.Context.baked_build_options, indent=4))

        ubuntu_version = self.Context.baked_build_options.get('distr')
        self.Context.pb_image = str(self.get_base_tgz(ubuntu_version).path)

        logging.info('[Prepare] End')

    def on_execute(self):
        logging.info('[Execute] Start')

        self.prepare_environ()
        try:
            self.build()
        except ChangelogEmpty:
            self.set_info("Changelog is empty. It's not a problem if it is autobuild. See previous build")
            return
        except PushFailed:
            self.set_info("Changelog wasn't pushed. Skipping this build")
            return

        if self.Parameters.build_dupload:
            self.dupload()

            if self.Parameters.build_conductor_create_ticket:
                self.create_conductor_ticket()
            else:
                self.dmove(self.Context.build_dmove)

        logging.info('[Execute] End')

    # endregion

    # region Main functions
    def build(self):
        logging.info('->[Build] Start')

        workdir = self.get_path_to_work_dir()
        abs_path = self.get_abs_path()

        control_path = os.path.join(
            workdir,
            'debian',
            'control'
        )

        changelog_path = os.path.join(
            workdir,
            'debian',
            'changelog'
        )

        with open(control_path, 'r') as f:
            debian_control = f.read()

            self.Context.build_source = re.search(r'^Source:\s*(.*)\n', debian_control).group(1)
            self.Context.build_packages = re.findall('^Package:[-> ]*(.+)$', debian_control, re.MULTILINE)

        with open(changelog_path, 'r') as f:
            debian_changelog = f.read()

            self.Context.build_version, distribution = re.search(r'^[\.\w_-]+ \((.+?)\) (\w+)',
                                                                 debian_changelog).groups()

        logging.info('Read debian files:')
        logging.info(json.dumps({
            'debian_control': control_path,
            'debian_changelog': changelog_path,
            'build_source': self.Context.build_source,
            'build_packages': self.Context.build_packages,
            'build_version': self.Context.build_version,
            'distribution': distribution
        }, indent=4))

        hook_dir = self.Context.baked_build_options.get('hookdir', os.path.join(workdir, 'hooks.d'))

        build_cmd = [
            'pdebuild', '--use-pdebuild-internal',
            # '--auto-debsign', '--debsign-k', self.Context.baked_build_options.get('repo', {}).get('user'),
            '--buildresult', abs_path,
            '--',
            '--basetgz', self.Context.pb_image,
            '--hookdir', hook_dir,
            '--buildplace', abs_path,
            '--aptcache', abs_path,
            '--debug'
        ]

        logging.info('Build command:')
        logging.info(json.dumps(build_cmd, indent=4))

        repo_name = self.Context.baked_build_options.get('repo', {}).get('name')

        if self.Context.baked_build_options.get('repo', {}).get('check_if_exist'):
            if_exists = Dist().if_exist(
                self.Context.build_packages[0],
                self.Context.build_version,
                repo_name
            )

            if if_exists:
                for binary_package in self.Context.build_packages:
                    self.set_info(
                        'Package {}={} is already in {}'.format(
                            binary_package, self.Context.build_version, repo_name
                        )
                    )

                raise TaskFailure('Nothing to build, package(s) already in {}'.format(repo_name))

        private_gpg_key = "{}-gpg-private".format(self.Parameters.gpg_vault_name)
        public_gpg_key = "{}-gpg-public".format(self.Parameters.gpg_vault_name)
        gpg_key_owner = self.Parameters.gpg_vault_owner

        logging.info('GPG info:')
        logging.info(json.dumps({
            'Private GPG key': private_gpg_key,
            'Public GPG key': public_gpg_key,
            'GPG keys owner': gpg_key_owner,
        }, indent=4))

        process_info = run_process(build_cmd, log_prefix='pdebuild', work_dir=workdir, check=False)

        if process_info.returncode > 0:
            stdout = open(process_info.stdout_path, 'r').read()

            if 'Some index files failed to download' in stdout:
                raise TemporaryError('Temporary problems with repos')
            else:
                raise SubprocessError('Build package failed')

        private_gpg_key = "{}-gpg-private".format(self.Parameters.gpg_vault_name)
        public_gpg_key = "{}-gpg-public".format(self.Parameters.gpg_vault_name)
        gpg_key_owner = self.Parameters.gpg_vault_owner

        abs_path = self.get_abs_path()
        workdir = self.get_path_to_work_dir()

        key_file = os.path.join(abs_path, 'key.gpg')
        manual_key = sdk2.Vault.data(gpg_key_owner, private_gpg_key)
        with open(key_file, 'w') as f:
            f.write(
                manual_key
            )

        run_process(['cat', key_file], work_dir=workdir, log_prefix='gpg_cat')
        run_process(['gpg', '--version'], work_dir=workdir, log_prefix='gpg_version')
        run_process(['gpg', '--list-keys'], work_dir=workdir, log_prefix='gpg_list')
        run_process(['gpg', '--import', key_file], work_dir=workdir, log_prefix='gpg')
        run_process(['gpg', '--list-keys'], work_dir=workdir, log_prefix='gpg_list_import')
        gpg_passphrase = sdk2.Vault.data(
            self.Parameters.yabro_vault_owner,
            'robot-sovetnik-gpg-pass'
        )

        for binary_package in self.Context.build_packages:
            self.set_info('{} {} builded'.format(binary_package, self.Context.build_version))
            # yandex-sovetnik-static_1.80.0.dsc
            dsc_path = '{abs_path}/{package_name}_{version}.dsc'.format(
                abs_path=abs_path, package_name=self.Parameters.result_package_name, version=self.Context.build_version)
            changes_path = '{abs_path}/{package_name}_{version}_amd64.changes'.format(
                abs_path=abs_path, package_name=self.Parameters.result_package_name, version=self.Context.build_version)

            logging.debug(glob.glob('{abs_path}/*'.format(
                abs_path=abs_path)))

            for file_to_sign in [dsc_path, changes_path]:
                logging.debug(file_to_sign)
                logging.debug(os.path.exists(file_to_sign))
                run_process(['debsign',
                             '-k{}@yandex-team.ru'.format(self.Context.baked_build_options.get('repo', {}).get('user')),
                             "-pgpg --batch --passphrase {}".format(gpg_passphrase),
                             file_to_sign], log_prefix='debsign', work_dir=workdir)

            for cpu_type in ['all', 'amd64']:
                resource_path = '{abs_path}/{binary_package}_{version_wo_epoch}_{cpu_type}.deb'.format(
                    abs_path=abs_path,
                    binary_package=binary_package,
                    version_wo_epoch=self.Context.build_version,
                    cpu_type=cpu_type)

                if os.path.exists(resource_path):
                    resource = SovetnikStaticDebPackage(
                        self,
                        path=resource_path,
                        description='Package ' + binary_package + '=' + self.Context.build_version,
                        attrs={'package_name': binary_package, 'version': self.Context.build_version},
                        owner=self.owner
                    )

                    sdk2.ResourceData(resource).ready()
                else:
                    logging.info('{} not found in {}'.format(os.path.basename(resource_path), os.getcwd()))

        logging.info('->[Build] End')

    def dupload(self):
        logging.info('->[DUpload] Start')

        repo_name = self.Context.baked_build_options.get('repo', {}).get('name')
        abs_path = self.get_abs_path()

        dupload_conf = {repo_name: {
            'fqdn': '{repo}.dupload.dist.yandex.ru'.format(
                repo=repo_name),
            'method': 'scpb',
            'incoming': '/repo/{}/mini-dinstall/incoming/'.format(repo_name),
            'dinstall_runs': 0,
            'login': self.Context.baked_build_options.get('repo', {}).get('user'),
            'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
        }}

        upload_cmd = "dupload --force --to {repo} {abs_path}/{source_package}_{version_stripped}_a*.changes --nomail".format(
            repo=repo_name,
            abs_path=abs_path,
            source_package=self.Context.build_source,
            version_stripped=self.Context.build_version
        )

        logging.info('Dupload config:')
        logging.info(json.dumps(dupload_conf, indent=4))

        logging.info('Upload cmd: {}'.format(upload_cmd))

        with DebRelease(dupload_conf):
            with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
                run_process(
                    'set -x; chmod 0644 *.{dsc,deb,tar.gz,changes}',
                    shell=True, log_prefix='chmod'
                )

                for i in range(5):
                    logging.info('Try #{}'.format(i))

                    dupload_process = run_process(
                        upload_cmd,
                        shell=True, log_prefix='dupload', check=False
                    )

                    if dupload_process.returncode == 0:
                        for binary_package in self.Context.build_packages:
                            self.set_info(
                                'Package {}={} successfully uploaded to {}'.format(
                                    binary_package, self.Context.build_version, repo_name)
                            )
                        break
                    else:
                        time.sleep(10)
                else:
                    raise TaskFailure("Package(s) weren't uploaded")

        logging.info('->[DUpload] End')

    def create_conductor_ticket(self):
        logging.info('->[CreateConductorTicket] Start')

        changelog_msg = run_process(
            'dpkg-parsechangelog -S changes',
            stdout=sp.PIPE, work_dir=self.get_path_to_work_dir()
        )

        changelog_msg = '\n'.join(changelog_msg.stdout.read().split('\n')[2:])

        logging.info('Changelog message:')
        logging.info(changelog_msg)

        conductor_auth = sdk2.Vault.data(self.Parameters.conductor_vault_owner,
                                         self.Parameters.conductor_vault_name)

        ticket_created = conductor.create_conductor_ticket(self,
                                                           conductor_auth,
                                                           [PACKAGES.get('static')],
                                                           self.Context.build_version,
                                                           server_url(),
                                                           branch=self.Context.baked_build_options.get('conductor').get(
                                                               'branch'),
                                                           changelog=changelog_msg)

        def find_url(string):
            return re.search(r'URL: \b([\w\.:/-]+)\b', string).group(1)

        url = find_url(ticket_created)
        self.set_info(ticket_created.replace(url, self.linkify(url)), do_escape=False)

        logging.info('->[CreateConductorTicket] End')

    def dmove(self, to_branch):
        logging.info('->[DMove] Start')
        logging.info('To branch: {}'.format(to_branch))

        with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            from_branch = 'unstable'

            for _ in range(12):
                j = Dist().find_package(self.ctx['build_packages'][0], self.ctx['build_version'])
                if j:
                    from_branch = j[0]['environment']
                    break
                else:
                    time.sleep(5)

            if from_branch == to_branch:
                self.set_info('Package(s) already in {}'.format(to_branch))
                return

            dmove_cmd = [
                'ssh',
                'robot-sovetnik@dupload.dist.yandex.ru',
                'sudo',
                'dmove',
                self.Context.baked_build_options.get('repo', {}).get('name'),
                to_branch,
                self.Context.build_packages[0],
                self.Context.build_version,
                from_branch
            ]

            logging.info('DMove CMD:')
            logging.info(json.dumps(dmove_cmd, indent=4))

            for _ in range(12):
                _dmove = run_process(dmove_cmd, log_prefix='dmove_{}'.format(to_branch), check=False)

                if _dmove.returncode == 0:
                    self.set_info('Package(s) dmoved to {} (from {})'.format(to_branch, from_branch))
                    break
                else:
                    time.sleep(5)
            else:
                raise TaskFailure("Package(s) weren't dmoved")

        logging.info('->[DMove] End')
    # endregion
