import os
import logging

import sandbox.common.types.misc as ctm

from sandbox import sdk2

from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.Sovetnik.utils.s3 import S3Client
from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild


TESTING_HOST = 'http://s3.mdst.yandex.net'
PRODUCTION_HOST = 'http://s3.mds.yandex.net'
BUCKET_NAME = 'sovetnik'


class SovetnikDeliverStaticBuildToS3(SovetnikBuild):

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4 * 1024
        disk_space = 4 * 1024
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('10.14.2'),
            environments.GCCEnvironment(version='5.3.0'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_parameters:
            git_release_branch = sdk2.parameters.String(
                label='Release branch',
                required=True,
                default='master',
            )
            git_ssh_url = sdk2.parameters.Url(
                label='Git remote SSH URL',
                required=True,
                default='git@github.yandex-team.ru:sovetnik/static.git',
            )

        with sdk2.parameters.Group('SSH parameters') as ssh_parameters:
            ssh_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's SSH private key",
                required=True,
                default='SOVETNIK',
            )
            ssh_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's SSH private key",
                required=True,
                default='robot-sovetnik-ssh',
            )

        with sdk2.parameters.Group('Run parameters') as run_parameters:
            with sdk2.parameters.RadioGroup('Run environment') as run_environment:
                run_environment.values['development'] = run_environment.Value(value='development')
                run_environment.values['production'] = run_environment.Value(value='production', default=True)

        with sdk2.parameters.Group('S3 parameters') as s3_params:
            s3_vault_owner = sdk2.parameters.String(
                label='Vault owner of s3 keys',
                required=True,
                default='SOVETNIK',
            )
            s3_access_key_id = sdk2.parameters.String(
                label='Vault name of s3 key_id',
                required=True,
                default='prod-robot-sovetnik-s3-access-key-id',
            )
            s3_access_key_secret = sdk2.parameters.String(
                label='Vault name of s3 access_key',
                required=True,
                default='prod-robot-sovetnik-s3-access-secret-key',
            )

    def build(self):
        """
        Builds sovetnik chrome prod build (zip) for assessors.

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()
        logging.debug("Start build: {}".format(path_to_work_dir))

        os.environ['NPM_PATH'] = self._get_npm_path()

        self.prepare_node_modules()

        os.environ['NODE_ENV'] = self.Parameters.run_environment

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call([os.environ['NPM_PATH'], 'run', 'zip:chrome'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

    def upload_to_s3(self):
        path_to_work_dir = os.path.join(self.get_path_to_work_dir(), 'build/sovetnik/extension/chrome.zip')

        logging.debug('path to chrome build (zip): {}'.format(path_to_work_dir))

        s3_endpoint = TESTING_HOST
        if self.Parameters.run_environment == 'production':
            s3_endpoint = PRODUCTION_HOST

        aws_secret_key_id = sdk2.Vault.data(self.Parameters.s3_vault_owner, self.Parameters.s3_access_key_id)
        aws_secret_access_key = sdk2.Vault.data(self.Parameters.s3_vault_owner, self.Parameters.s3_access_key_secret)

        s3_client = S3Client(
            key_id=aws_secret_key_id,
            access_key=aws_secret_access_key,
            bucket_name=BUCKET_NAME,
            s3_endpoint=s3_endpoint
        )

        s3_client.upload(path_to_work_dir, '_/zip/sovetnik-chrome.zip')

    def on_execute(self):
        logging.debug('Start build and upload')

        super(SovetnikDeliverStaticBuildToS3, self).on_execute()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.upload_to_s3()

        logging.debug('End build and upload')
