import os
import logging

import sandbox.common.types.misc as ctm

from sandbox import sdk2

from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.Sovetnik.utils.s3 import S3Client
from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild


TESTING_HOST = 'http://s3.mdst.yandex.net'
PRODUCTION_HOST = 'http://s3.mds.yandex.net'
BUCKET_NAME = 'sovetnik'
SCRIPTS = {
    'browser': {
        'active': False,
        'name': 'ecomerce-context-sovetnik-browser.js',
        'path': 'build/static/js/ecomerce-context-sovetnik-browser.js',
        'build_command': 'build:browser'
    },
    'browser_native': {
        'active': False,
        'name': 'ecomerce-context-sovetnik-browser-native.js',
        'path': 'build/static/js/ecomerce-context-sovetnik-browser-native.js',
        'build_command': 'build:native'
    }
}


class SovetnikDeliverDesktopScriptsToYastatic(SovetnikBuild):

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4 * 1024
        disk_space = 4 * 1024
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('8.12.0'),
            environments.GCCEnvironment(),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_parameters:
            git_release_branch = sdk2.parameters.String(
                label='Release branch',
                required=True,
                default='master',
            )
            git_ssh_url = sdk2.parameters.Url(
                label='Git remote SSH URL',
                required=True,
                default='git@github.yandex-team.ru:sovetnik/static.git',
            )

        with sdk2.parameters.Group('SSH parameters') as ssh_parameters:
            ssh_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's SSH private key",
                required=True,
                default='SOVETNIK',
            )
            ssh_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's SSH private key",
                required=True,
                default='robot-sovetnik-ssh',
            )

        with sdk2.parameters.Group('Run parameters') as run_parameters:
            with sdk2.parameters.RadioGroup('Run environment') as run_environment:
                run_environment.values['development'] = run_environment.Value(
                    value='development', default=True)
                run_environment.values['production'] = run_environment.Value(value='production')

        with sdk2.parameters.Group('S3 parameters') as s3_params:
            s3_vault_owner = sdk2.parameters.String(
                label='Vault owner of s3 keys',
                required=True,
                default='SOVETNIK',
            )
            s3_access_key_id = sdk2.parameters.String(
                label='Vault name of s3 key_id',
                required=True,
                default='robot-sovetnik-s3-access-key-id',
            )
            s3_access_key_secret = sdk2.parameters.String(
                label='Vault name of s3 access_key',
                required=True,
                default='robot-sovetnik-s3-access-secret-key',
            )
        with sdk2.parameters.Group("Yandex Browser's parameters") as ya_bro_parameters:
            ya_bro_password_vault_owner = sdk2.parameters.String(
                label="Vault owner of Yandex Browser's password",
                required=True,
                default='SOVETNIK',
            )
            ya_bro_password_vault_name = sdk2.parameters.String(
                label="Vault name of Yandex Browser's password",
                required=True,
                default='ya-bro-password',
            )
        with sdk2.parameters.Group("Items to build") as items_names:
            browser_native = sdk2.parameters.Bool(
                label='ecomerce-context-sovetnik-browser-native.js',
                default=True)
            browser = sdk2.parameters.Bool(
                label='ecomerce-context-sovetnik-browser.js',
                default=True
            )

    def fill_in_scripts_to_build(self):
        SCRIPTS['browser_native']['active'] = self.Parameters.browser_native
        SCRIPTS['browser']['active'] = self.Parameters.browser

    def build(self):
        """
        Builds desktop scrtipts

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()
        logging.debug("Start build: {}".format(path_to_work_dir))

        os.environ['NPM_PATH'] = self._get_npm_path()
        logging.debug('npm version ')

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call(['npm', '-v'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

        self.prepare_node_modules()

        os.environ['NODE_ENV'] = self.Parameters.run_environment

        os.environ['YA_BRO_PASSWORD'] = sdk2.Vault.data(
            self.Parameters.ya_bro_password_vault_owner,
            self.Parameters.ya_bro_password_vault_name,
        )

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            for value in SCRIPTS.values():
                if value['active']:
                    sp.check_call([os.environ['NPM_PATH'], 'run', value['build_command']], cwd=path_to_work_dir,
                                  stdout=pl.stdout, stderr=pl.stderr)

    def upload_to_s3(self):
        s3_endpoint = TESTING_HOST
        if self.Parameters.run_environment == 'production':
            s3_endpoint = PRODUCTION_HOST

        aws_secret_key_id = sdk2.Vault.data(self.Parameters.s3_vault_owner, self.Parameters.s3_access_key_id)
        aws_secret_access_key = sdk2.Vault.data(self.Parameters.s3_vault_owner, self.Parameters.s3_access_key_secret)

        s3_client = S3Client(
            key_id=aws_secret_key_id,
            access_key=aws_secret_access_key,
            bucket_name=BUCKET_NAME,
            s3_endpoint=s3_endpoint
        )

        for value in SCRIPTS.values():
            if value['active']:
                path_to_work_dir = os.path.join(self.get_path_to_work_dir(), value['path'])
                logging.debug('path to script: {}'.format(path_to_work_dir))
                s3_client.upload(path_to_work_dir, '_/js/' + value['name'])

    def on_execute(self):
        logging.debug('Start build and upload')
        self.fill_in_scripts_to_build()
        logging.debug('Scripts to build: {}'.format(SCRIPTS))

        super(SovetnikDeliverDesktopScriptsToYastatic, self).on_execute()

        with self.memoize_stage.pack(commit_on_entrance=False):
            self.upload_to_s3()

        logging.debug('End build and upload')
