# coding: utf-8

import os
import logging
import shutil
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.Sovetnik.SovetnikBuildArcadia import SovetnikBuildArcadia
from sandbox.projects.common.arcadia.sdk import mount_arc_path

delete_old_users = "procedures:delete-old:users"
delete_old_records = "procedures:delete-old:records"
delete_old_urls = "procedures:delete-old:urls"
delete_old_records_and_refs = "procedures:delete-old:records-and-refs"

create_bins = "procedures:ab:bins"
move_client_ids = "procedures:ab:move-client-ids"
check_client_ids = "procedures:ab:check-client-ids"
split_users = "procedures:ab:split-users"
show_bins = "procedures:ab:show-bins"

sync_domains = "procedures:sync-domains"

sync_shop_domains = "procedures:sync-shop-domains"

cache_script_name = "create-node-modules-resource.py"


class SovetnikProceduresArcadia(SovetnikBuildArcadia):
    work_dir_name = 'workdir'
    build_dir_name = 'build_dir'

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4 * 1024
        disk_space = 4 * 1024
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('8.12.0'),
            environments.GCCEnvironment(),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Настройки репозитория') as repo_block:
            arc_path = sdk2.parameters.String(
                'Arcadia url:',
                required=True,
                default_value='arcadia-arc:/#trunk',
                description='Arc путь. EX: arcadia-arc:/#trunk или arcadia-arc:/#users/bogdansky/SOVETNIK-4106'
            )

            repo_path = sdk2.parameters.String(
                'Repo path:',
                required=True,
                description='Путь к репозиторию Sovetnik Backend. EX: market/sovetnik/backend',
                default_value='market/sovetnik/backend'
            )

        with sdk2.parameters.Group('SSH parameters') as ssh_parameters:
            ssh_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's SSH private key",
                required=True,
                default='SOVETNIK',
            )
            ssh_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's SSH private key",
                required=True,
                default='robot-sovetnik-ssh',
            )

        with sdk2.parameters.Group('Run parameters') as run_parameters:
            with sdk2.parameters.RadioGroup('Run environment') as run_environment:
                run_environment.values['development'] = run_environment.Value(
                    value='development', default=True)
                run_environment.values['production'] = run_environment.Value(value='production')

        with sdk2.parameters.RadioGroup('Procedures to run') as procedures:
            need_to_run_records = sdk2.parameters.Bool(delete_old_records, default=False)
            need_to_run_users = sdk2.parameters.Bool(delete_old_users, default=False)
            need_to_run_records_and_refs = sdk2.parameters.Bool(delete_old_records_and_refs, default=False)
            need_to_run_urls = sdk2.parameters.Bool(delete_old_urls, default=False)

            need_to_run_create_bins = sdk2.parameters.Bool(create_bins, default=False)
            need_to_run_move_client_ids = sdk2.parameters.Bool(move_client_ids, default=False)
            need_to_run_check_client_ids = sdk2.parameters.Bool(check_client_ids, default=False)
            need_to_run_split_users = sdk2.parameters.Bool(split_users, default=False)
            need_to_run_show_bins = sdk2.parameters.Bool(show_bins, default=False)

            need_to_run_sync_domains = sdk2.parameters.Bool(sync_domains, default=False)

            need_to_run_sync_shop_domains = sdk2.parameters.Bool(sync_shop_domains, default=False)

    def checkout_arcadia_repository(self):
        logging.debug('Checkout arcadia repo {}'.format(self.Parameters.repo_path))
        with mount_arc_path(self.Parameters.arc_path, use_arc_instead_of_aapi=True) as arcadia:
            self.work_dir_name = os.path.join(arcadia, self.Parameters.repo_path)
            self.build_dir_name = os.path.join(str(self.path('')).rstrip('/'), 'build_dir')
            logging.debug('Copy arc to local dir > {} to {}'.format(self.work_dir_name, self.build_dir_name))
            shutil.copytree(self.work_dir_name, self.build_dir_name)
            self.work_dir_name = self.build_dir_name

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        path_to_work_dir = self.work_dir_name
        logging.debug("Start build: {}".format(path_to_work_dir))

        npm = self._get_npm_path()
        os.environ['NPM_PATH'] = npm

        os.environ['MONGODB_PASS'] = sdk2.Vault.data(
            'sovetnik-mongo-password')

        os.environ['REDIS_PASS'] = sdk2.Vault.data(
            'sovetnik-redis-password')

        os.environ['POSTGRES_PORT'] = sdk2.Vault.data('env.POSTGRES_PORT')
        os.environ['POSTGRES_DB'] = sdk2.Vault.data('env.POSTGRES_DB')
        os.environ['POSTGRES_USER'] = sdk2.Vault.data('env.POSTGRES_USER')
        os.environ['POSTGRES_PASSWORD'] = sdk2.Vault.data('env.POSTGRES_PASSWORD')

        self.prepare_node_modules()

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            os.environ['NODE_ENV'] = self.Parameters.run_environment
            if self.Parameters.need_to_run_records:
                logging.debug('run {} command'.format(delete_old_records))

                sp.check_call([npm, 'run', delete_old_records, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            if self.Parameters.need_to_run_users:
                logging.debug('run {} command'.format(delete_old_users))

                sp.check_call([npm, 'run', delete_old_users, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            if self.Parameters.need_to_run_records_and_refs:
                logging.debug('run {} command'.format(delete_old_records_and_refs))

                sp.check_call([npm, 'run', delete_old_records_and_refs, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            if self.Parameters.need_to_run_urls:
                logging.debug('run {} command'.format(delete_old_urls))

                sp.check_call([npm, 'run', delete_old_urls, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            if self.Parameters.need_to_run_create_bins:
                logging.debug('run {} command'.format(create_bins))

                sp.check_call([npm, 'run', create_bins, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            if self.Parameters.need_to_run_move_client_ids:
                """
                    @todo clone users list here
                """
                logging.debug('unable to run {} command'.format(move_client_ids))

            if self.Parameters.need_to_run_check_client_ids:
                logging.debug('run {} command'.format(check_client_ids))

                sp.check_call([npm, 'run', check_client_ids, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            if self.Parameters.need_to_run_split_users:
                logging.debug('run {} command'.format(split_users))

                sp.check_call([npm, 'run', split_users, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            if self.Parameters.need_to_run_show_bins:
                logging.debug('run {} command'.format(show_bins))

                sp.check_call([npm, 'run', show_bins, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            if self.Parameters.need_to_run_sync_domains:
                logging.debug('run {} command'.format(sync_domains))

                sp.check_call([npm, 'run', sync_domains, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            if self.Parameters.need_to_run_sync_shop_domains:
                logging.debug('run {} command'.format(sync_shop_domains))

                sp.check_call([npm, 'run', sync_shop_domains, '--if-present'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

    def on_execute(self):
        logging.debug("Start procedures task")

        with self.memoize_stage.checkout_arcadia_repository(commit_on_entrance=False):
            self.checkout_arcadia_repository()

        with self.memoize_stage.build(commit_on_entrance=False):
            self.build()

        logging.debug("End procedures task")
