import os
import logging
import tarfile
import sandbox.common.types.misc as ctm

from sandbox import sdk2

from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.Sovetnik.SovetnikBuild import SovetnikBuild
from sandbox.projects.Sovetnik.SovetnikBuildMarketShopsPackage import SovetnikMarketShopsPackage


class SovetnikSyncDomainsPackage(nanny.ReleaseToNannyTask2, SovetnikBuild):
    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4 * 1024
        disk_space = 4 * 1024
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('8.12.0'),
            environments.GCCEnvironment(),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_parameters:
            git_release_branch = sdk2.parameters.String(
                label='Release branch',
                required=True,
                default='master',
            )
            git_ssh_url = sdk2.parameters.Url(
                label='Git remote SSH URL',
                required=True,
                default='git@github.yandex-team.ru:sovetnik/sovetnik-domains.git',
            )

        with sdk2.parameters.Group('SSH parameters') as ssh_parameters:
            ssh_vault_owner = sdk2.parameters.String(
                label="Vault owner of Sovetnik's SSH private key",
                required=True,
                default='SOVETNIK',
            )
            ssh_vault_name = sdk2.parameters.String(
                label="Vault name of Sovetnik's SSH private key",
                required=True,
                default='robot-sovetnik-ssh',
            )

        with sdk2.parameters.Group('Run parameters') as run_parameters:
            with sdk2.parameters.RadioGroup('Run environment') as run_environment:
                run_environment.values['development'] = run_environment.Value(
                    value='development', default=True)
                run_environment.values['production'] = run_environment.Value(value='production')

            migrate_shops = sdk2.parameters.Bool(
                "Run Market Shops migration",
                default=False
            )

    def find_resource(self):
        resource = SovetnikMarketShopsPackage.find().first()

        if resource is None:
            return False

        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        return resource_path

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        path_to_work_dir = self.get_path_to_work_dir()
        logging.debug("Start build: {}".format(path_to_work_dir))
        resource_name = 'sovetnik-shops.tar.gz'

        if self.Parameters.migrate_shops is True:
            path_to_resource = self.find_resource()
            if path_to_resource is False:
                logging.debug("Cannot find resource")
            else:
                path_to_resource = str(path_to_resource)
                logging.debug("Sync resource to path: {}".format(path_to_resource))
                local_resource_name = os.path.join(path_to_work_dir, resource_name)

                sp.check_call(['cp', path_to_resource, local_resource_name])
                with tarfile.open(local_resource_name, 'r:gz') as tar:
                    tar.extractall(path_to_work_dir)

                logging.debug('Resource dir contents: {}'.format(
                    os.listdir(path_to_work_dir)))

        os.environ['NPM_PATH'] = self._get_npm_path()

        env_vars = [
            'YT_OAUTH_ACCESS_TOKEN',
            'POSTGRES_PORT',
            'POSTGRES_DB',
            'POSTGRES_USER',
            'POSTGRES_PASSWORD',
            'Q_OAUTH_ACCESS_TOKEN',
        ]

        for var in env_vars:
            os.environ[var] = sdk2.Vault.data('env.{}'.format(var))

        if self.Parameters.run_environment == 'production':
            os.environ['POSTGRES_HOST'] = sdk2.Vault.data('env.POSTGRES_HOST_PROD')
        else:
            os.environ['POSTGRES_HOST'] = sdk2.Vault.data('env.POSTGRES_HOST')

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call([os.environ['NPM_PATH'], 'install', '-registry=https://npm.yandex-team.ru'],
                          cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

        os.environ['NODE_ENV'] = self.Parameters.run_environment
        os.environ['ONE_TIME_SCRIPT'] = 'true'

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            if self.Parameters.migrate_shops is True:
                sp.check_call([os.environ['NPM_PATH'], 'run', 'sync:market-shops'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            sp.check_call([os.environ['NPM_PATH'], 'run', 'sync:yt'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

    def on_execute(self):
        logging.debug("Start sync task")

        with self.memoize_stage.clone_git_repository(commit_on_entrance=False):
            self.clone_git_repository()

        with self.memoize_stage.build(commit_on_entrance=False):
            self.build()

        logging.debug("End sync task")
