# coding: utf-8

import os
import logging
import tarfile
import sandbox.common.types.misc as ctm
import shutil

from sandbox import sdk2

from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.Sovetnik.SovetnikBuildArcadia import SovetnikBuildArcadia
from sandbox.projects.Sovetnik.SovetnikBuildMarketShopsPackage import SovetnikMarketShopsPackage
from sandbox.projects.common.arcadia.sdk import mount_arc_path


class SovetnikSyncDomainsPackageArcadia(nanny.ReleaseToNannyTask2, SovetnikBuildArcadia):
    workdir = 'workdir'
    build_dir = 'build_dir'

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4 * 1024
        disk_space = 4 * 1024
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('8.12.0'),
            environments.GCCEnvironment(),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Настройки репозитория') as repo_block:
            arc_path = sdk2.parameters.String(
                'Arcadia url:',
                required=True,
                default_value='arcadia-arc:/#trunk',
                description='Arc путь. EX: arcadia-arc:/#trunk или arcadia-arc:/#users/bogdansky/SOVETNIK-4106'
            )

            repo_path = sdk2.parameters.String(
                'Repo path:',
                required=True,
                description='Путь к репозиторию Sovetnik Domains. EX: market/sovetnik/domains',
                default_value='market/sovetnik/domains'
            )

        with sdk2.parameters.Group('Параметры запуска') as run_parameters:
            with sdk2.parameters.RadioGroup('Run environment') as run_environment:
                run_environment.values['development'] = run_environment.Value(
                    value='development', default=True)
                run_environment.values['production'] = run_environment.Value(value='production')

            migrate_shops = sdk2.parameters.Bool(
                "Run Market Shops migration",
                default=False
            )

    def checkout_arcadia_repository(self):
        logging.debug('Checkout arcadia repo {}'.format(self.Parameters.repo_path))
        with mount_arc_path(self.Parameters.arc_path, use_arc_instead_of_aapi=True) as arcadia:
            self.workdir = os.path.join(arcadia, self.Parameters.repo_path)
            self.build_dir = os.path.join(str(self.path('')).rstrip('/'), 'build_dir')
            logging.debug('Copy arc to local dir > {} to {}'.format(self.workdir, self.build_dir))
            shutil.copytree(self.workdir, self.build_dir)
            self.workdir = self.build_dir

    def find_resource(self):
        resource = SovetnikMarketShopsPackage.find().first()

        if resource is None:
            return False

        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        return resource_path

    def build(self):
        """
        Builds Sovetnik's package.

        :returns: Nothing.
        """
        path_to_work_dir = self.workdir
        logging.debug("Start build: {}".format(path_to_work_dir))
        resource_name = 'sovetnik-shops.tar.gz'

        if self.Parameters.migrate_shops is True:
            path_to_resource = self.find_resource()
            if path_to_resource is False:
                logging.debug("Cannot find resource")
            else:
                path_to_resource = str(path_to_resource)
                logging.debug("Sync resource to path: {}".format(path_to_resource))
                local_resource_name = os.path.join(path_to_work_dir, resource_name)

                sp.check_call(['cp', path_to_resource, local_resource_name])
                with tarfile.open(local_resource_name, 'r:gz') as tar:
                    tar.extractall(path_to_work_dir)

                logging.debug('Resource dir contents: {}'.format(
                    os.listdir(path_to_work_dir)))

        os.environ['NPM_PATH'] = self._get_npm_path()

        env_vars = [
            'YT_OAUTH_ACCESS_TOKEN',
            'POSTGRES_PORT',
            'POSTGRES_DB',
            'POSTGRES_USER',
            'POSTGRES_PASSWORD',
            'Q_OAUTH_ACCESS_TOKEN',
        ]

        for var in env_vars:
            os.environ[var] = sdk2.Vault.data('env.{}'.format(var))

        if self.Parameters.run_environment == 'production':
            os.environ['POSTGRES_HOST'] = sdk2.Vault.data('env.POSTGRES_HOST_PROD')
        else:
            os.environ['POSTGRES_HOST'] = sdk2.Vault.data('env.POSTGRES_HOST')

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call([os.environ['NPM_PATH'], 'install', '-registry=https://npm.yandex-team.ru'],
                          cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

        os.environ['NODE_ENV'] = self.Parameters.run_environment
        os.environ['ONE_TIME_SCRIPT'] = 'true'

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            if self.Parameters.migrate_shops is True:
                sp.check_call([os.environ['NPM_PATH'], 'run', 'sync:market-shops'], cwd=path_to_work_dir,
                              stdout=pl.stdout, stderr=pl.stderr)

            sp.check_call([os.environ['NPM_PATH'], 'run', 'sync:yt'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr)

    def on_execute(self):
        logging.debug("Start sync task")

        with self.memoize_stage.checkout_arcadia_repository(commit_on_entrance=False):
            self.checkout_arcadia_repository()

        with self.memoize_stage.build(commit_on_entrance=False):
            self.build()

        logging.debug("End sync task")
