import logging
import boto3


logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


class S3Client:

    def __init__(self, access_key, key_id, bucket_name, s3_endpoint):
        session = boto3.session.Session()
        self.bucket_name = bucket_name
        self.s3_endpoint = s3_endpoint
        self.s3 = session.client(
            service_name='s3',
            endpoint_url=s3_endpoint,
            aws_access_key_id=key_id,
            aws_secret_access_key=access_key
        )

    def exists(self, s3_key):
        try:
            self.s3.head_object(Bucket=self.bucket_name,
                                Key=s3_key)
            return True
        except Exception:
            return False

    def upload(self, local_path, s3_path):
        logger.info('Attempt to upload {0} with path {1}'.format(
            s3_path, local_path))
        try:
            self.s3.upload_file(local_path, self.bucket_name, s3_path)
        except IOError:
            logger.error('File {0} does not exist'.format(local_path))

        if self.exists(s3_path):

            logger.info('Successfully uploaded resource:'
                        ' http://{0}/{1}/{2}'.format(self.s3_endpoint,
                                                     self.bucket_name,
                                                     s3_path))
        else:
            raise Exception('Unable to upload to S3')

    def delete(self, upload_key):
        logger.info('Removing {0} from S3'.format(upload_key))
        try:
            self.s3.delete_object(Bucket=self.bucket_name,
                                  Key=upload_key)
            return True
        except Exception as e:
            logger.error('Can not delete object %s \n error:  %d',
                         (upload_key, e))
            return False

    def download(self, s3_path, local_path,):
        logger.info('Attempt to download {0} to {1}'.format(
            s3_path, local_path))

        if self.exists(s3_path):
            try:
                self.s3.download_file(self.bucket_name, s3_path, local_path)
                logger.info('Successfully downloaded resource from'
                            ' http://{0}/{1}/{2} to {3}'.format(self.s3_endpoint,
                                                                self.bucket_name,
                                                                s3_path, local_path))

            except IOError:
                logger.error('File {0} does not exist'.format(local_path))
