# -*- coding: utf-8 -*-

import pymongo
import logging
import re
import time
import os

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.sandboxapi import TASK_FINISHED, TASK_FAILURE, TASK_UNKNOWN, TASK_DELETED, TASK_STOPPED, TASK_STOPPING, TASK_NOT_READY
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxSubprocessError

from sandbox.projects.common.gencfg.task import IGencfgTask
from sandbox.projects.common.gencfg import utils as config_generator_utils

logger = logging.getLogger(__name__)


class SpawnTestConfigGenerator(IGencfgTask):
    type = 'SPAWN_TEST_CONFIG_GENERATOR'

    def get_downloader_path(self):
        return self.abs_path('get_gencfg_repo.py')

    def get_gencfg_path(self):
        return self.abs_path('gencfg')

    # can not run these tasks in parallel, so check if sandbox has other running tasks of the same type
    def check_running_tasks(self):
        finished_statuses = [TASK_FINISHED, TASK_FAILURE, TASK_UNKNOWN, TASK_DELETED, TASK_STOPPED, TASK_STOPPING, TASK_NOT_READY]
        tasks = channel.sandbox.list_tasks(task_type=self.type, limit=30)

        if len(filter(lambda x: (x.id != self.id) and (x.status not in finished_statuses), tasks)) > 0:
            return False

        return True

    def get_mongo_client(self):
        mongo_db_path = 'hb-dbs01.search.yandex.net,hb-dbs03.search.yandex.net,hb-dbs04.search.yandex.net'
        mongo_options = {
            'replicaSet': 'heartbeat',
            'read_preference': pymongo.ReadPreference.PRIMARY,
            'socketTimeoutMS': 10000,
            'connectTimeoutMS': 600,
            'waitQueueTimeoutMS': 1000,
        }

        return pymongo.MongoReplicaSetClient(mongo_db_path, **mongo_options)

    def generate_failed_message(self, task):
        author = "%s@yandex-team.ru" % task.ctx.get('last_commit_author', 'sereglond').partition('@')[0]
        if author == 'robot-gencfg@yandex-team.ru':
            author = '%s@yandex-team.ru' % self.author

        modified_commit = task.ctx['last_commit']
        modified_commit_url = "https://arc.yandex-team.ru/wsvn/arc?op=comp&compare[]=%2F@{}&compare[]=%2F@{}".format(int(modified_commit) - 1, int(modified_commit))

        message = """Check if your commit< %s> breaks gencfg build.
Sandbox task: %s\n""" % (modified_commit_url, task.url)

        # add broken goals with urls
        resources = channel.sandbox.list_resources(resource_type='CONFIG_BUILD_LOGS', task_id=task.id)
        if resources is not None and len(resources) != 0:
            log_http_url = channel.sandbox.get_resource_http_links(resources[0].id)
            if len(log_http_url):
                log_http_url = log_http_url[0].rpartition('/')[0]
            else:
                log_http_url = None
        else:
            log_http_url = None

        for broken_goal in task.ctx['build_broken_goals']:
            if log_http_url is None:
                message += "    Broken goal <%s> .\n" % broken_goal
            else:
                message += "    Broken goal <%s> : %s .\n" % (broken_goal, "%s/%s.log" % (log_http_url, broken_goal))

        subject = "[sandbox] [%s] Broken gencfg build" % (self.type)

        return (author, subject, message)

    def initCtx(self):
        self.ctx['kill_timeout'] = 600

    def on_enqueue(self):
        self.ctx['scheduled_tasks'] = []
        self.ctx['mailed_failed_tasks'] = []

    def spawn_new_test_tasks(self):
        """
            Check if we have new commits and if have ones, start sandbox tasks
        """

        mongoclient = self.get_mongo_client()
        mongocoll = mongoclient['heartbeat']['sandbox_storage']
        entry = mongocoll.find_one({'task_type': self.type})

        last_checked_commit = int(entry['last_checked_commit'])

        TRIES = 3
        for i in range(TRIES):
            try:
                p = run_process(["svn", "update", "--non-interactive"], work_dir=os.path.join(self.get_gencfg_path(), 'db'), log_prefix="update_svn")
                p = run_process(["svn", "log", "-q", "-r", "%s:HEAD" % (last_checked_commit)], work_dir=os.path.join(self.get_gencfg_path(), 'db'), log_prefix="get_unchecked_revisions")
                break
            except SandboxSubprocessError:
                if i == TRIES - 1:
                    raise
                else:
                    time.sleep(5)

        new_commits = re.findall("^r(\d+) ", open(p.stdout_path).read(), re.MULTILINE)
        new_commits = map(lambda x: int(x), new_commits)
        new_commits = filter(lambda x: x != last_checked_commit, new_commits)

        # start sandbox task for every commit that still not tested
        for commit in new_commits:
            subtask = channel.sandbox.create_task('TEST_CONFIG_GENERATOR',
                                                  self.author,
                                                  'Test commit <a href="https://a.yandex-team.ru/commit/{commit}">{commit}</a>'.format(commit=commit),
                                                  context={'last_commit': commit}, priority=("SERVICE", "HIGH"),
                                                  parameters={'ram': 75 * 1024})
            self.ctx['scheduled_tasks'].append(subtask.id)

        if len(new_commits) > 0:
            entry['last_checked_commit'] = new_commits[-1]
            mongocoll.remove({'task_type': 'SPAWN_TEST_CONFIG_GENERATOR'})
            mongocoll.insert(entry)

    def update_tesk_task_status(self):
        # got through all task, that have already finished and mail corresponding users
        alltasks = channel.sandbox.list_tasks(task_type='TEST_CONFIG_GENERATOR', limit=200)
        alltasks.sort(cmp=lambda x, y: cmp(x.id, y.id))
        for i in range(1, len(alltasks)):
            curtask = alltasks[i]
            prevtask = alltasks[i - 1]

            if curtask.status != TASK_FINISHED or prevtask.status != TASK_FINISHED:
                continue
            if curtask.ctx['build_status'] != 'FAILURE' or prevtask.ctx['build_status'] != 'SUCCESS':
                continue
            if curtask.id < 32350111:  # skip old
                continue
            if curtask.ctx.get('mail_sent', False):
                continue

            author, subj, msg = self.generate_failed_message(curtask)

            # some log
            logger.info("Send message to <%s>" % author)

            channel.sandbox.send_email(author, '', subj, msg)
            channel.sandbox.set_task_context_value(curtask.id, 'mail_sent', True)

    def on_execute(self):
        logger = logging.getLogger(__name__)

        if not self.check_running_tasks():
            logger.info("Found at least one working task, exiting ...")
            return

        config_generator_utils.clone_gencfg_all(self.get_gencfg_path(), 'full')

        finisht = int(time.time()) + 9 * 60 * 60
        while time.time() < finisht:
            self.spawn_new_test_tasks()
            self.update_tesk_task_status()
            time.sleep(5)


__Task__ = SpawnTestConfigGenerator
