# -*- coding: utf-8 -*-

import datetime

from os.path import join as pj

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects import resource_types
from sandbox.projects.common import task_env
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.utils import server_url
from sandbox.common.types import task as ctt
import sandbox.common.types.misc as ctm

from sandbox.projects.logs.release_helpers import GetLastGoodReleaseInfo

BAOBAB_SOURCE = 'quality/logs/baobab/api/python_arc/library'
BAOBAB_ARTIFACT = 'baobab.so'

RESULT_DIR = 'baobab.build'
BAOBAB_SO = '{}/baobab.so'.format(RESULT_DIR)
BAOBAB_SO_DEBUG = '{}/baobab.so.debug'.format(RESULT_DIR)

YT_PATH = '//statbox/resources-versioned/baobab.so'
YT_ATTR_BUILT_BY = 'built_by'
# Имя токена в Vault должно быть в формате yt-<yt-rw-user>
# Владелец токена - STATINFRA
TOKENS = {'robot-statinfra': ('hume', 'freud'),
          'robot-stat-service': ('hahn', 'arnold')}


class StatboxBaobabResource(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = False
    auto_backup = True


class StatboxBaobabApiCook(sdk2.Task):
    """Сборка baobab.so и загрузка на YT"""

    class Requirements(task_env.TinyRequirements):
        disk_space = 2 * 1024  # 2 Gb
        environments = PipEnvironment('yandex-yt'),

    def build_baobab(self, arcadia_url):
        if self.Context.ya_make_task_id != ctm.NotExists:
            return

        subtask_type = sdk2.Task["YA_MAKE_2"]

        # TODO: ttl somehow. or after - set ttl-attrs myself
        params = dict(
            arch='linux',
            checkout_arcadia_from_url=arcadia_url,
            targets=BAOBAB_SOURCE,
            arts=pj(BAOBAB_SOURCE, BAOBAB_ARTIFACT),
            result_rt=StatboxBaobabResource.name,
            build_system='ya',
            build_type='release',
            use_aapi_fuse=True,
            aapi_fallback=True,
            check_return_code=True,
            result_single_file=True,
            strip_binaries=False,
            def_flags=dict(USE_SYSTEM_PYTHON='2.7')
        )

        subtask = subtask_type(
            self,
            description='Building baobab library',
            **params
        )

        sdk2.Task.server.task[subtask.id].update(
            requirements=dict(disk_space=8 * 1024 ** 2)
        )

        subtask.enqueue()

        self.Context.ya_make_task_id = subtask.id

        raise sdk2.WaitTask(
            [subtask],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
        )

    def on_execute(self):
        import yt.wrapper as yt

        arcadia_url = GetLastGoodReleaseInfo.get_last_success_resource()

        self.build_baobab(arcadia_url)

        self.set_info("Seems it's built")

        baobab = sdk2.Resource.find(
            resource_type=StatboxBaobabResource,
            task_id=self.Context.ya_make_task_id
        ).limit(1).first()

        baobab_data = sdk2.ResourceData(baobab)

        sp.check_call(['mkdir', '-p', RESULT_DIR])
        sp.check_call(['cp', str(baobab_data.path), BAOBAB_SO])
        sp.check_call(['chmod', '755', BAOBAB_SO])
        sp.check_call(['cp', BAOBAB_SO, BAOBAB_SO_DEBUG])

        with sdk2.helpers.ProcessLog(self, logger="strip") as pl:
            sp.check_call(['strip', '--strip-debug', '--strip-unneeded', BAOBAB_SO], shell=False, stdout=pl.stdout, stderr=pl.stdout)
            sp.check_call(['strip', '--only-keep-debug', BAOBAB_SO_DEBUG], shell=False, stdout=pl.stdout, stderr=pl.stdout)

        self.set_info("baobab.so was stripped and now it's ready to upload")

        baobab_res = sdk2.ResourceData(resource_types.OTHER_RESOURCE(self, 'baobab.so', BAOBAB_SO))
        baobab_res_debug = sdk2.ResourceData(resource_types.OTHER_RESOURCE(self, 'baobab.so.debug', BAOBAB_SO_DEBUG))

        baobab_res.ready()
        baobab_res_debug.ready()

        today_date = datetime.date.today().strftime('%Y-%m-%d')
        self.set_info('YT path: {}/{}'.format(YT_PATH, today_date))
        sandbox_www = server_url()
        for token in TOKENS:
            yt.config['token'] = sdk2.Vault.data('STATINFRA', 'yt-{}'.format(token))
            for cluster in TOKENS[token]:
                yt.config['proxy']['url'] = '{}.yt.yandex.net'.format(cluster)
                yt.smart_upload_file(BAOBAB_SO, destination='{}/{}'.format(YT_PATH, today_date), placement_strategy='replace')
                yt.set('{}/{}/@{}'.format(YT_PATH, today_date, YT_ATTR_BUILT_BY), '{}/task/{}'.format(sandbox_www, self.id))
                yt.smart_upload_file(BAOBAB_SO_DEBUG, destination='{}/{}.debug'.format(YT_PATH, today_date), placement_strategy='replace')
                yt.set('{}/{}.debug/@{}'.format(YT_PATH, today_date, YT_ATTR_BUILT_BY), '{}/task/{}'.format(sandbox_www, self.id))
                yt.link('{}/{}'.format(YT_PATH, today_date), '//statbox/resources/baobab.so', ignore_existing=False, force=True)
                self.set_info('{}: OK'.format(cluster))
