# -*- coding: utf-8 -*-
import logging
import subprocess
import tempfile
from urlparse import urlparse

import os
import re
import requests
import time
import yaml
from sandbox.common.types import misc
from sandbox.common.types.client import Tag
from sandbox.common.types.task import Semaphores
from sandbox.common.utils import server_url
from sandbox.projects import resource_types
from sandbox.projects.common import utils

from sandbox.projects.common.debpkg import DebRelease
from sandbox.projects.common.gnupg import GpgKey
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter, SandboxGitUrlParameter

from sandbox.sandboxsdk.process import run_process, SandboxSubprocessError
from sandbox.sandboxsdk.ssh import Key
from sandbox.sandboxsdk.task import SandboxTask

import conductor
import resources
from collections import OrderedDict
from dist import Dist
from sandbox.common import errors
from .. import STATBOX_DEB_PACKAGE, STATBOX_PBUILDER_APTCACHE, STATBOX_PBUILDER_HOOKS, STATBOX_PBUILDER_IMAGE

SUPPORTED_UBUNTU_VERSIONS = ['precise', 'trusty', 'xenial']
DEFAULT_UBUNTU_VERSION = 'precise'
DEFAULT_DISTRIBUTION = 'unstable'
DEFAULT_TESTING_DISTRIBUTION = 'testing'
WAIT_REPO_TIMEOUT = 600

REPOS = [
    'common',
    'statbox-common',
    'yandex-precise',
    'yandex-trusty'
]

PYPIRC = '''
[distutils]
index-servers =
    yandex

[yandex]
repository: https://pypi.yandex-team.ru/simple/
username: {username}
password: {password}
'''

REVISOR_DEFAULT_ENV = 'default'
COMMIT_MSG = "Package's version set to"
CHANGELOG_DELIMITER = '::'

# ############################################### STATBOX_PBUILDER_IMAGE ###############################################
#
# For creating pbuilder image:
# 1) pbuilder --crease trusty.tgz --distribution trusty
# 2) sudo pbuilder --login --save-after-login --basetgz trusty.tgz
# 3) rm /dev/loop2 /dev/port /dev/loop7 /dev/midi02 /dev/tty4 /dev/loop6 /dev/tty8 /dev/loop4 /dev/ram10 /dev/dsp3
# /dev/ram12 /dev/ram8 /dev/tty9 /dev/ram7 /dev/mpu401data /dev/rmidi2 /dev/ram2 /dev/loop0 /dev/loop1 /dev/midi2
# /dev/ram9 /dev/tty3 /dev/dsp /dev/midi0 /dev/ram15 /dev/midi1 /dev/smpte3 /dev/tty2 /dev/kmem /dev/rmidi0 /dev/ram3
# /dev/mixer /dev/ram6 /dev/loop5 /dev/audio1 /dev/dsp2 /dev/agpgart /dev/full /dev/tty6 /dev/mem /dev/mixer3
# /dev/mpu401stat /dev/audio3 /dev/midi3 /dev/dsp1 /dev/sequencer /dev/midi01 /dev/audio2 /dev/midi00 /dev/audio
# /dev/ram14 /dev/rmidi3 /dev/sndstat /dev/smpte2 /dev/ram1 /dev/tty5 /dev/loop3 /dev/midi03 /dev/smpte0 /dev/mixer1
# /dev/ram4 /dev/mixer2 /dev/ram0 /dev/ram11 /dev/ram13 /dev/audioctl /dev/ram5 /dev/ram16 /dev/smpte1 /dev/tty7
# /dev/rmidi1  (SANDBOX-2958)
# 4) ya upload trusty.tgz -T STATBOX_PBUILDER_IMAGE --do-not-remove -A release=trusty
#
# ############################################### STATBOX_PBUILDER_IMAGE ###############################################

# class ArcadiaURL(SandboxArcadiaUrlParameter):
#     name = 'ArcadiaURL'
#     description = 'URL to repository with source files. Type `svn info` and copy-paste URL'
#     required = True
#     group = 'Source parameters'


class GitURL(SandboxGitUrlParameter):
    name = 'GitURL'
    description = 'URL to Arcadia folder with actions repo'
    required = True
    group = 'Source parameters'


class GitBranch(SandboxStringParameter):
    name = 'GitBranch'
    description = 'Git branch:'
    default_value = 'master'
    required = True
    group = 'Source parameters'


class BuildParamPath(SandboxStringParameter):
    name = "BuildParamPath"
    description = "Path to directory with .build.yml"
    default_value = ""
    group = 'Source parameters'


class CheckoutSSHKey(SandboxStringParameter):
    name = 'CheckoutSSHKey'
    description = 'Name of SSH key in Vault for checkout sources'
    default_value = 'robot-statinfra'
    required = True
    group = 'Source parameters'


class KeyOwner(SandboxStringParameter):
    name = 'KeyOwner'
    description = 'Owner name of SSH key in Vault for checkout sources'
    default_value = 'STATINFRA'
    required = True
    group = 'Source parameters'


class BuildDistr(SandboxStringParameter):
    name = 'BuildDistr'
    description = 'Name of image for pdebuild'
    group = 'Build parameters'
    default_value = 'precise'


# class BuildHooks(SandboxStringParameter):
#     name = 'BuildHooks'
#     description = 'Dir name for pdebuild hooks'
#     group = 'Build parameters'
#     default_value = 'hooks.d'


class BuildConductorBranch(SandboxStringParameter):
    name = 'BuildConductorBranch'
    description = 'Branch in conductor'
    choices = [(branch, branch) for branch in ['unstable', 'testing', 'prestable', 'stable', 'hotfix', 'fallback']]
    group = 'Conductor parameters'
    default_value = 'testing'


class BuildConductorProjects(SandboxStringParameter):
    name = 'BuildConductorProjects'
    description = 'Projects for tickets in conductor'
    default_value = 'STATBOX'
    group = 'Conductor parameters'


class BuildConductorCreateTicket(SandboxBoolParameter):
    name = 'BuildConductorCreateTicket'
    description = 'Create ticket in conductor'
    group = 'Conductor parameters'
    default_value = True
    # sub_fields = {
    #     'true': [BuildConductorBranch.name, BuildConductorProjects.name]
    # }


class BuildConductorAuth(SandboxStringParameter):
    name = 'BuildConductorAuth'
    description = 'Auth cookie vault key name'
    group = 'Conductor parameters'
    default_value = 'robot-statinfra'


class BuildConductorAuthOwner(SandboxStringParameter):
    name = 'BuildConductorAuthOwner'
    description = 'Auth cookie vault key owner'
    group = 'Conductor parameters'
    default_value = 'STATINFRA'


class BuildRepoName(SandboxStringParameter):
    name = 'BuildRepoName'
    description = 'Repo name'
    choices = [(repo, repo) for repo in REPOS]
    group = 'Dist parameters'
    default_value = 'statbox-common'


class BuildRepoCheckIfExist(SandboxBoolParameter):
    name = 'BuildRepoCheckIfExist'
    description = 'Check if exist in repo'
    group = 'Dist parameters'
    default_value = True


class BuildDupload(SandboxBoolParameter):
    name = 'BuildDupload'
    description = 'Dupload packages'
    default_value = True
    group = 'Dist parameters'
    # sub_fields = {
    #     'true': [BuildRepoName.name, BuildRepoCheckIfExist.name, BuildConductorCreateTicket.name]
    # }


class BuildRepoUser(SandboxStringParameter):
    name = 'BuildRepoUser'
    description = 'Name of user for sign and dupload'
    group = 'Build parameters'
    default_value = 'robot-statinfra'


class BuildReleaseType(SandboxStringParameter):
    name = 'BuildReleaseType'
    description = 'Release type'
    choices = [(release, release) for release in ['major', 'minor', 'hotfix']]
    default_value = 'minor'
    # sub_fields = {
    #     'hotfix': [GitBranch.name]
    # }
    # group = 'Build parameters'


class BuildGenerateChangelog(SandboxBoolParameter):
    name = 'BuildGenerateChangelog'
    description = 'Generate changelog'
    default_value = True
    # sub_fields = {
    #     'true': [BuildReleaseType.name]
    # }
    # group = 'Build parameters'


class BuildStictChangelog(SandboxBoolParameter):
    name = 'BuildStictChangelog'
    description = 'Strict changelog'
    default_value = True


class BuildCheckNoCommits(SandboxBoolParameter):
    name = 'BuildCheckNoCommits'
    description = 'Check if no new commits (fail if exist)'
    default_value = False


class BuildEnableTox(SandboxBoolParameter):
    name = 'BuildEnableTox'
    description = 'Enable tox executing (tox -e sandbox)'
    default_value = False


class BuildUseAptCache(SandboxBoolParameter):
    name = 'BuildUseAptCache'
    description = 'Use aptcache from previous builds'
    default_value = False  # DMP-26 temp


class BuildPackAptCache(SandboxBoolParameter):
    name = 'BuildPackAptCache'
    description = 'Pack aptcache from current build'
    default_value = True


class BuildSemaphore(SandboxStringParameter):
    name = 'BuildSemaphore'
    description = 'Semaphore name'


class BuildPyPIUpload(SandboxBoolParameter):
    name = 'BuildPyPIUpload'
    description = 'Upload python package to PyPI'
    default_value = False
    group = 'PyPI parameters'


class BuildPyPIAccess(SandboxStringParameter):
    name = 'BuildPyPIAccess'
    description = 'Access key for PyPI'
    default_value = 'robot-statinfra'
    group = 'PyPI parameters'


class BuildPyPISecret(SandboxStringParameter):
    name = 'BuildPyPISecret'
    description = 'Secret key for PyPI'
    default_value = 'robot-statinfra'
    group = 'PyPI parameters'


class BuildPyPIKeysOwner(SandboxStringParameter):
    name = 'BuildPyPIKeysOwner'
    description = 'Vault owner of pypi keys'
    default_value = 'STATINFRA'
    group = 'PyPI parameters'


class BuildParamOverride(SandboxBoolParameter):
    name = 'BuildParamOverride'
    description = 'Override build params'
    default_value = False
    group = 'Build parameters'
    sub_fields = {
        'true': [BuildDistr.name, BuildRepoUser.name,
                 BuildDupload.name, BuildRepoCheckIfExist.name, BuildRepoName.name,
                 BuildPyPIUpload.name, BuildPyPIAccess.name, BuildPyPISecret.name, BuildPyPIKeysOwner.name,
                 BuildConductorCreateTicket.name, BuildConductorBranch.name, BuildConductorProjects.name]
    }


def make_version(version, increment=None):
    version_parts = [
        int(part) if part.isdigit() else part
        for part in version.split('.')]
    if len(version_parts) < 3:
        raise Exception("Unexpected version format", version)
    if increment == 'major':
        version_parts[0] += 1
        version_parts[1] = version_parts[2] = 0
    elif increment == 'minor':
        version_parts[1] += 1
        version_parts[2] = 0
    elif increment == 'hotfix':
        version_parts[2] += 1
    return '.'.join(str(part) for part in version_parts)


def linkify(url):
    return '<a href={}>{}</a>'.format(url, url)


def juggler_push(repo, status, task_id):
    def requests_retry():
        s = requests.Session()
        r = requests.packages.urllib3.util.retry.Retry(total=3, backoff_factor=0.5)
        a = requests.adapters.HTTPAdapter(max_retries=r)
        s.mount('http://', a)
        return s
    tags = []
    if '://' not in repo:
        repo = 'ssh://{}'.format(repo.replace(':', '/'))
    service = urlparse(repo).path.lstrip('/')
    reply = requests_retry().post(
        'http://juggler-push.search.yandex.net/events',
        json={
            'source': 'sandbox',
            'events': [
                {
                    'host': 'statbox-debuilder',
                    'service': service,
                    'status': status,
                    'description': '{}/task/{}'.format(server_url(), task_id),
                    'tags': tags
                }
            ]
        },
        timeout=10,
    )
    logging.info('Juggler event (service: {}; status: {}) returned {} code: {}'.format(
        service, status, reply.status_code, reply.text
    ))


# TODO: probably make it external and use in STATBOX_TICKET_TO_STABLE also
def clean_changelog(changelog):
    def badline(line):
        skip_check_ = (True if line == 'Note: mandatory check (NEED_CHECK) was skipped' else False)
        empty_line_ = (True if line == '' else False)
        review_line = (True if line.startswith('REVIEW: ') else False)
        delim_here_ = (False if CHANGELOG_DELIMITER in line else True)
        return skip_check_ or empty_line_ or review_line or delim_here_
    clean_changelog = list()
    for line in '\n'.join(changelog).split('\n'):
        if badline(line):
            continue
        else:
            clean_changelog.append(line)
    return clean_changelog


class ChangelogEmpty(Exception):
    pass


class PushFailed(Exception):
    pass


class StatboxDebuilder(SandboxTask):
    privileged = True
    type = 'STATBOX_DEBUILDER'
    dns = misc.DnsType.DNS64
    client_tags = Tag.LINUX_XENIAL

    cores = 1
    required_ram = 31000
    execution_space = 5000   # 5G

    input_parameters = (
        BuildGenerateChangelog,
        BuildStictChangelog,
        BuildCheckNoCommits,
        BuildReleaseType,
        BuildEnableTox,
        BuildUseAptCache,
        BuildPackAptCache,
        BuildSemaphore,
        # RepoType,
        GitURL,
        GitBranch,
        # ArcadiaURL,
        BuildParamPath,
        CheckoutSSHKey,
        KeyOwner,
        BuildParamOverride,
        BuildDistr,
        # BuildHooks,
        BuildRepoUser,
        BuildDupload,
        BuildRepoCheckIfExist,
        BuildRepoName,
        BuildPyPIUpload,
        BuildPyPIAccess,
        BuildPyPISecret,
        BuildPyPIKeysOwner,
        BuildConductorCreateTicket,
        BuildConductorBranch,
        BuildConductorProjects,
        BuildConductorAuth,
        BuildConductorAuthOwner
    )
    workdir = 'workdir'
    build_options = {}

    def on_enqueue(self):
        if self.ctx.get(BuildSemaphore.name):
            semaphore = self.ctx.get(BuildSemaphore.name)
            self.semaphores(Semaphores(acquires=[Semaphores.Acquire(name=semaphore, capacity=1)]))

    def on_execute(self):
        self.build_options = self._get_build_options()
        self.source_path = self.abs_path(self.workdir)
        self.ctx['build_dmove'] = self.ctx.get(BuildConductorBranch.name)
        self._prepare_environ()
        self._clone_repo()
        try:
            self._build()
        except ChangelogEmpty:
            self.set_info("Changelog is empty. It's not a problem if it is autobuild. See previous build")
            return
        except PushFailed:
            self.set_info("Changelog wasn't pushed. Skipping this build")
            return
        if self.ctx.get(BuildDupload.name):
            self._dupload()
            self._touch_revisor()
            if self.ctx.get(BuildConductorCreateTicket.name):
                self._create_ticket()
            else:
                self._dmove(self.ctx['build_dmove'])

    def on_success(self):
        juggler_push(self.ctx.get(GitURL.name), 'OK', self.id)

    def on_failure(self):
        juggler_push(self.ctx.get(GitURL.name), 'CRIT', self.id)

    def on_break(self):
        juggler_push(self.ctx.get(GitURL.name), 'CRIT', self.id)

    def on_release(self, additional_parameters):
        self._dmove('stable')
        self._touch_revisor('production')

    def _dmove(self, tobranch):
        with Key(self, self.ctx.get(KeyOwner.name), '{}-ssh'.format(self.ctx.get(CheckoutSSHKey.name))):
            frombranch = 'unstable'
            for _ in range(12):
                j = Dist().find_package(self.ctx['build_packages'][0], self.ctx['build_version'])
                if j:
                    frombranch = j[0]['environment']
                    break
                else:
                    time.sleep(5)
            if frombranch == tobranch:
                self.set_info('Package(s) already in {}'.format(tobranch))
                return
            dmove_cmd = ['ssh',
                         'robot-statinfra@dupload.dist.yandex.ru',
                         'sudo',
                         'dmove',
                         self.ctx.get(BuildRepoName.name),
                         tobranch,
                         self.ctx['build_packages'][0],
                         self.ctx['build_version'],
                         frombranch]
            for _ in range(12):
                _dmove = run_process(dmove_cmd, log_prefix='dmove_{}'.format(tobranch), check=False)
                if _dmove.returncode == 0:
                    self.set_info('Package(s) dmoved to {} (from {})'.format(tobranch, frombranch))
                    break
                else:
                    time.sleep(5)
            else:
                raise errors.TaskError("Package(s) weren't dmoved")

    def _get_basetgz(self, ubuntu_version='precise'):
        pbimage = resources.get_newest_resource(self, STATBOX_PBUILDER_IMAGE, ubuntu_version,
                                                attrs={'release': ubuntu_version})
        logging.info('pbimage with release {} synced to {}'.format(ubuntu_version, pbimage))
        return pbimage

    def _get_hooks_pack(self):
        hooks_pack = resources.get_newest_resource(self, STATBOX_PBUILDER_HOOKS, 'STATBOX_PBUILDER_HOOKS')
        logging.info('Hooks pack successfully downloaded ({})'.format(hooks_pack))
        return hooks_pack

    def _clone_repo(self):
        self.ctx['repo_type'] = 'arcadia'
        self._clone_arcadia()
        self._clone_changelog()

    def _clone_arcadia(self):
        logging.info('Clone arcadia repo from {}'.format(self.ctx.get(GitURL.name)))
        svn.Arcadia.checkout(self.ctx.get(GitURL.name), self.workdir)

    def _clone_changelog(self):
        # source path MUST contain @revision spec, see DMP-741
        self.changelog_root = "debian_changelog_root"
        self.debian_checkout_url = self.ctx.get(GitURL.name).split("@")[0] + "/debian"
        logging.info('Clone for changelog from %s', self.debian_checkout_url)
        svn.Arcadia.checkout(self.debian_checkout_url, self.changelog_root)

        # copy latest changelog to working folder where `dch` will
        # do its secret magic
        run_process([
            "cp", "--verbose",
            os.path.join(self.changelog_root, "changelog"),
            os.path.join(self.source_path, "debian/changelog"),
        ])

    def _get_build_options(self):
        return BuildOptions(
            os.path.join(
                self.workdir.rstrip('/'),
                utils.get_or_default(self.ctx, BuildParamPath).strip('/'),
                '.build.yml'
            ),
            self.ctx
        )

    def _prepare_environ(self):
        for v in ['TEMP', 'TMPDIR', 'TMP']:
            if v in os.environ:
                del os.environ[v]
        logging.debug(os.environ)
        run_process(['sudo', 'apt-get', 'update'])
        run_process(
            [
                'sudo', 'apt-get', 'install', 'pbuilder',
                '-q0', '--assume-yes', '--force-yes', '-o', "Dpkg::Options::='--force-confdef'"
            ],
            shell=True, log_prefix='install_pbuilder'
        )

    def _touch_revisor(self, branch=None):
        with Key(self, self.ctx.get(KeyOwner.name), '{}-ssh'.format(self.ctx.get(CheckoutSSHKey.name))):
            revisor = str(tempfile.mkdtemp())
            run_process(
                [
                    'svn', 'checkout',
                    'svn+ssh://robot-statinfra@arcadia.yandex.ru/arc/trunk/arcadia/statbox/revisor', revisor
                ],
                log_prefix='revisor-checkout')
            for binary_package in self.ctx['build_packages']:
                if not branch:
                    branch = (
                        'production' if self.ctx.get(BuildConductorBranch.name) == 'stable'
                        else self.ctx.get(BuildConductorBranch.name)
                    )
                default_package_file = os.path.join(revisor, '{}.{}'.format(binary_package, REVISOR_DEFAULT_ENV))
                if os.path.exists(default_package_file):
                    package_file = default_package_file
                    self.ctx['build_dmove'] = 'stable'
                else:
                    package_file = os.path.join(revisor, '{}.{}'.format(binary_package, branch))
                if os.path.exists(package_file):
                    with open(package_file, 'w') as f:
                        f.write(self.ctx['build_version'])
                    self.set_info("{} was updated in revisor".format(binary_package))
                else:
                    self.set_info("{} is not under revisor control".format(binary_package))
            commit_message = '{} {}'.format(' '.join(self.ctx['build_packages']), self.ctx['build_version'])
            if hasattr(self, 'build_changelog'):
                commit_message += '\n\n'
                commit_message += self.build_changelog
            commit_message += '\n\nBuilded by {}/task/{}'.format(server_url(), self.id)
            svn_env = os.environ.copy()
            svn_env['LC_ALL'] = 'ru_RU.utf8'
            _svn_commit = run_process(
                ['svn', 'commit', '--message', commit_message],
                log_prefix='revisor-commit',
                work_dir=revisor,
                check=False,
                environment=svn_env)
            if _svn_commit.returncode:
                self.set_info("Revisor state wasn't updated")
            else:
                self.set_info("Revisor state updated")
            # svn_commit = _svn_commit.stdout.read()
            # if svn_commit:
            #     commit_revision = svn_commit.split()[-1].rstrip('.')
            #     commit_url = 'https://a.yandex-team.ru/arc/commit/{}'.format(commit_revision)
            #     self.set_info('Revisor state updated\nURL: {}'.format(linkify(commit_url)), do_escape=False)
            # else:
            #     self.set_info('Revisor state wasn\'t changed')

    def _create_ticket(self):
        changelog_msg = run_process('dpkg-parsechangelog -S changes', stdout=subprocess.PIPE, work_dir=self.source_path)
        changelog_msg = '\n'.join(changelog_msg.stdout.read().split('\n')[2:])
        conductor_auth = self.get_vault_data(utils.get_or_default(self.ctx, BuildConductorAuthOwner),
                                             '{}-conductor-auth'.format(
                                                 utils.get_or_default(self.ctx, BuildConductorAuth)))
        ticket_created = conductor.create_conductor_ticket(self,
                                                           conductor_auth,
                                                           self.ctx['build_packages'],
                                                           self.ctx['build_version'],
                                                           server_url(),
                                                           branch=self.ctx.get(BuildConductorBranch.name),
                                                           changelog=changelog_msg)

        def find_url(string):
            return re.search(r'URL: \b([\w\.:/-]+)\b', string).group(1)

        url = find_url(ticket_created)
        self.set_info(ticket_created.replace(url, linkify(url)), do_escape=False)

    def _dupload(self):
        dupload_conf = {
            r: {
                'fqdn': '{repo}.dupload.dist.yandex.ru'.format(repo=self.ctx.get(BuildRepoName.name)),
                'method': 'scpb',
                'incoming': '/repo/{}/mini-dinstall/incoming/'.format(r),
                'dinstall_runs': 0,
                'login': self.ctx.get(BuildRepoUser.name),
                'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
            } for r in REPOS
        }
        upload_cmd = "dupload --force --to {repo} {abs_path}/{source_package}_{version_stripped}_a*.changes".format(
            repo=self.ctx.get(BuildRepoName.name),
            abs_path=self.abs_path(),
            source_package=self.ctx['build_source'],
            version_stripped=self.ctx['build_version']
        )
        with DebRelease(dupload_conf):
            with Key(self, utils.get_or_default(self.ctx, KeyOwner), '{}-ssh'.format(self.ctx.get(BuildRepoUser.name))):
                run_process('set -x; chmod 0644 *.{dsc,deb,tar.gz,changes}', shell=True, log_prefix='chmod')
                for _ in range(5):
                    dupload_process = run_process(upload_cmd, shell=True, log_prefix='dupload', check=False)
                    if dupload_process.returncode == 0:
                        for binary_package in self.ctx['build_packages']:
                            self.set_info(
                                'Package {}={} successfully uploaded to {}'.format(
                                    binary_package, self.ctx['build_version'], self.ctx.get(BuildRepoName.name))
                            )
                        break
                    else:
                        time.sleep(10)
                else:
                    raise errors.TaskError("Package(s) weren't uploaded")

    def _generate_changelog(self):
        changelog = list()
        stop_changelog = False
        for log_entry in svn.Arcadia.log(self.workdir):
            if log_entry['msg'].startswith("{} {}".format(COMMIT_MSG, self.ctx['build_version'])):
                stop_changelog = True
            if stop_changelog:
                break
            else:
                changelog.append('{}{}{}'.format(log_entry['author'], CHANGELOG_DELIMITER, log_entry['msg']))
        if not stop_changelog:
            if len(changelog) > 10:
                changelog = changelog[:10]
                changelog.append(
                    'MR TRUNCATER{}[NOTICKETNO-0000] '
                    'PREVIOUS STATBOX_DEBUILDER MARK WAS NOT FOUND'.format(CHANGELOG_DELIMITER)
                )

        changelog = clean_changelog(changelog)
        if not changelog:
            if self.ctx.get(BuildStictChangelog.name):
                raise ChangelogEmpty
            changelog.append(
                CHANGELOG_DELIMITER.join([
                    self.ctx.get(BuildRepoUser.name), 'Building strict changelog disabled. Default message'
                ])
            )

        # build_changelog for revisor
        changedict, c = OrderedDict(), str()
        for line in changelog:
            author, message = line.split(CHANGELOG_DELIMITER)
            if author not in changedict:
                changedict[author] = list()
            changedict[author].append(message)
        firsttime = True
        for a in changedict:
            if firsttime:
                firsttime = False
            else:
                c += '\n'
            c += '[ {author} ]\n'.format(author=a)
            for m in changedict[a]:
                c += ' * {message}\n'.format(message=m)
        self.build_changelog = c.rstrip('\n')
        self.set_info(self.build_changelog)
        logging.info('build_changelog is: {}'.format(self.build_changelog))

        # append debian/changelog
        self.ctx['build_version'] = make_version(self.ctx['build_version'], self.ctx.get(BuildReleaseType.name))
        line1 = changelog.pop(0).split(CHANGELOG_DELIMITER, 1)
        os.environ['DEBFULLNAME'] = line1[0]
        run_process(
            [
                'dch', '--newversion', self.ctx['build_version'],
                '--distribution', self.ctx.get(BuildDistr.name), '--', line1[1]
            ],
            work_dir=self.source_path)
        for _line in changelog:
            line = _line.split(CHANGELOG_DELIMITER, 1)
            os.environ['DEBFULLNAME'] = line[0]
            run_process(['dch', '--append', '--multimaint-merge', '--', line[1]], work_dir=self.source_path)
        os.environ['DEBFULLNAME'] = self.ctx.get(BuildRepoUser.name)
        os.environ['EMAIL'] = '{}@yandex-team.ru'.format(os.environ['DEBFULLNAME'])
        run_process('dch --release ""', work_dir=self.source_path)

    def _push_changelog(self):
        with Key(self, self.ctx.get(KeyOwner.name), '{}-ssh'.format(self.ctx.get(CheckoutSSHKey.name))):
            # copy updated changelog to its checkout place (for commit)
            run_process([
                "cp", "--verbose",
                os.path.join(self.source_path, "debian/changelog"),
                os.path.join(self.changelog_root, "changelog"),
            ])

            try:
                svn.Arcadia.commit(
                    self.changelog_root, '{} {} SKIP_CHECK'.format(COMMIT_MSG, self.ctx['build_version']),
                    user=self.ctx.get(CheckoutSSHKey.name)
                )
            except Exception as e:
                logging.exception("Arcadia commit failed: %s", e)
                raise PushFailed
            else:
                self.set_info('Changelog was commited to {}'.format(self.debian_checkout_url))

    def _build(self):
        with open(os.path.join(self.source_path, 'debian', 'control'), 'r') as f:
            debian_control = f.read()
            self.ctx['build_source'] = re.search(r'^Source:\s*(.*)\n', debian_control).group(1)
            self.ctx['build_packages'] = re.findall('^Package:[\t ]*(.+)$', debian_control, re.MULTILINE)

        with open(os.path.join(self.source_path, 'debian', 'changelog'), 'r') as f:
            self.ctx['build_version'], distribution = re.search(r'^[\.\w_-]+ \((.+?)\) (\w+)', f.read()).groups()

        logging.info('Found version {}', self.ctx['build_version'])

        os.environ['STATBOX_DEBUILDER_WORKDIR'] = self.source_path
        # enable tox
        if self.ctx.get(BuildEnableTox.name):
            os.environ['STATBOX_DEBUILDER_ENABLE_TOX'] = BuildEnableTox.name
        # TODO: BuildGenerateChangelog, BuildReleaseType to BuildOptions
        if self.ctx.get(BuildGenerateChangelog.name):
            self._generate_changelog()
        # Это для многоплатформенных сборок, чтобы в сборку под следующую платформу не попали коммиты которые не попали
        # в предыдущую. Не работает
        # if self.ctx.get(BuildCheckNoCommits.name):
        #     if self.ctx['build_changelog']:
        #         raise errors.TaskFailure("Changelog isn't empty")
        ubuntu_version = self.build_options.params.get('distr')
        # extract hooks here
        hooks_path = os.path.join(self.source_path, 'pbuilder-hooks')
        run_process(['mkdir', hooks_path])
        hooks_pack = self._get_hooks_pack()
        run_process(['tar', '--extract', '--file', hooks_pack, '--gzip', '--keep-old-files'], work_dir=hooks_path)
        # extract aptcache
        aptcache_path = os.path.join(self.source_path, 'pbuilder-aptcache')
        run_process(['mkdir', aptcache_path])
        if False and self.ctx.get(BuildUseAptCache.name):  # DMP-26
            aptcache_pack = resources.get_newest_resource(self, STATBOX_PBUILDER_APTCACHE, 'STATBOX_PBUILDER_APTCACHE')
            run_process(['tar', '--extract', '--file', aptcache_pack], work_dir=aptcache_path)
        else:
            aptcache_pack = 'pbuilder-aptcache.tar'
        # make cmds
        build_cmd = [
            'pdebuild',
            '--use-pdebuild-internal',
            '--auto-debsign', '--debsign-k', self.build_options.params.get('repo', {}).get('user'),
            '--buildresult', self.abs_path(),
            '--',
            '--basetgz', self._get_basetgz(ubuntu_version),
            # '--hookdir', self.build_options.params.get('hookdir', '{}/hooks.d/'.format(self.workdir)),
            '--buildplace', self.abs_path(),
            '--aptcache', aptcache_path,
        ]
        if ubuntu_version != 'trusty':
            build_cmd.extend(['--hookdir', hooks_path])

        if ubuntu_version == 'xenial':
            # DMP-26 hack: disable PPA list load
            with open(os.path.join(hooks_path, 'D11-ppa-list'), 'w') as f:
                f.write('#!/bin/bash\nexit 0')

        if self.ctx.get(BuildRepoCheckIfExist.name):
            if Dist().if_exist(
                self.ctx['build_packages'][0], self.ctx['build_version'], self.ctx.get(BuildRepoName.name)
            ):
                for binary_package in self.ctx['build_packages']:
                    self.set_info(
                        'Package {}={} is already in {}'.format(
                            binary_package, self.ctx['build_version'], self.ctx.get(BuildRepoName.name)
                        )
                    )
                raise errors.TaskError(
                    'Nothing to build, package(s) already in {}'.format(self.ctx.get(BuildRepoName.name))
                )
        with GpgKey(
            self,
            utils.get_or_default(self.ctx, KeyOwner),
            "{}-gpg-private".format(self.ctx.get(BuildRepoUser.name)),
            "{}-gpg-public".format(self.ctx.get(BuildRepoUser.name)),
        ):
            logging.info('Build with cmd:\n{}'.format(build_cmd))
            # pdebuild
            process_info = run_process(build_cmd, log_prefix='pdebuild', work_dir=self.source_path, check=False)
            if process_info.returncode > 0:
                stdout = open(process_info.stdout_path, 'r').read()
                if 'Some index files failed to download' in stdout:
                    raise errors.TemporaryError('Temporary problems with repos')
                else:
                    raise SandboxSubprocessError(
                        message='Build package failed',
                        stdout_path=process_info.stdout_path_filename,
                        logs_resource=getattr(getattr(channel.task, '_log_resource', None), 'id', None),
                    )
            # upload deb as resource
            for binary_package in self.ctx['build_packages']:
                self.set_info('{} {} builded'.format(binary_package, self.ctx['build_version']))
                for cpu_type in ['all', 'amd64']:
                    resource_path = '{abs_path}/{binary_package}_{version_wo_epoch}_{cpu_type}.deb'.format(
                        abs_path=self.abs_path(),
                        binary_package=binary_package,
                        version_wo_epoch=self.ctx['build_version'],
                        cpu_type=cpu_type)
                    if os.path.exists(resource_path):
                        self.create_resource(
                            description='Package ' + binary_package + '=' + self.ctx['build_version'],
                            resource_path=resource_path,
                            resource_type=STATBOX_DEB_PACKAGE,
                            attributes={'package_name': binary_package, 'version': self.ctx['build_version']},
                            owner=self.owner)
                    else:
                        logging.info('{} not found in {}'.format(os.path.basename(resource_path), os.getcwd()))
            if self.ctx.get(BuildPackAptCache.name):
                # upload aptcache as resource
                aptcache = [debfile for debfile in os.listdir(aptcache_path) if debfile.endswith('.deb')]

                def aptcache_mtime(name):
                    return os.path.getmtime('{}/{}'.format(aptcache_path, name))

                aptcache.sort(key=aptcache_mtime, reverse=True)
                _aptcache = dict()
                for debfile in aptcache:
                    _debname, _debversion = debfile.split('_', 1)
                    if _debname not in _aptcache:
                        _aptcache[_debname] = _debversion
                aptcache = ['{}_{}'.format(debname, debversion) for debname, debversion in _aptcache.iteritems()]
                # have to do this, because root path is readonly
                aptcache_pack = os.path.join(self.source_path, os.path.basename(aptcache_pack))
                tar_aptcache_cmd = ['tar', '--create', '--file', aptcache_pack]
                tar_aptcache_cmd.extend(aptcache)
                run_process(tar_aptcache_cmd, log_prefix='TAR', work_dir=aptcache_path)
                self.create_resource('Cached apt packages for pbuilder', resource_path=aptcache_pack,
                                     resource_type=STATBOX_PBUILDER_APTCACHE, owner=self.owner)
            # upload report.html
            # TODO: make _pytest_html be parametrized
            if self.ctx.get(BuildEnableTox.name):
                # self.create_resource('Tests result', resource_path='{}/tests.xml'.format(source_path),
                #                      resource_type=resource_types.OTHER_RESOURCE, owner=self.owner)
                _pytest_html = os.path.join(self.source_path, 'pytest.html')
                if os.path.isfile(_pytest_html):
                    pytest_html_resource = self.create_resource(
                        'HTML report by pytest', resource_path=_pytest_html,
                        resource_type=resource_types.OTHER_RESOURCE, owner=self.owner
                    )
                    self.set_info(
                        'PyTest HTML report: {}'.format(linkify(pytest_html_resource.proxy_url)),
                        do_escape=False
                    )
            # with open(os.path.join(source_path, 'report.html')) as report_html:
            #     self.set_info(report_html.read(), do_escape=False)
        # push changelog
        if self.ctx.get(BuildGenerateChangelog.name):
            self._push_changelog()

        # upload to pypi
        if self.ctx.get(BuildPyPIUpload.name):
            pypi_user = self.get_vault_data(utils.get_or_default(self.ctx, BuildPyPIKeysOwner),
                                            '{}-pypi-access'.format(utils.get_or_default(self.ctx, BuildPyPIAccess)))
            pypi_pass = self.get_vault_data(utils.get_or_default(self.ctx, BuildPyPIKeysOwner),
                                            '{}-pypi-secret'.format(utils.get_or_default(self.ctx, BuildPyPISecret)))
            with open('{}/.pypirc'.format(os.environ['HOME']), 'w') as pypirc:
                pypirc.write(PYPIRC.format(username=pypi_user, password=pypi_pass))
            run_process(['python', 'setup.py', 'sdist', 'upload', '-r', 'yandex'],
                        work_dir=self.source_path, log_prefix='pypi_upload')
            self.set_info('Successfully uploaded to PyPI')


class BuildOptions(object):
    """
    Class for construct and validate build options
    """
    build_options = {
        'distr': 'precise',
        'hookdir': 'hooks.d',
        'repo': {
            'name': 'statbox-common',
            'check_if_exist': True,
            'dupload': True,
            'user': 'robot-statinfra'
        },
        'conductor': {
            'create_ticket': True,
            'branch': 'testing',
            'projects': ['STATBOX'],
        }
    }

    def __init__(self, build_file, ctx):
        logging.info('Trying to load .build.yml')
        self.ctx = ctx
        self.build_options = self._read_file_(build_file)
        self.params = self._merge_()
        logging.info(self.params)

    def _read_file_(self, build_file):
        if self.ctx.get(BuildParamOverride.name):
            logging.info('Params overrided')
            return self.build_options
        try:
            with open(build_file, 'r') as stream:
                logging.info('Try to read .build.yml')
                return yaml.load(stream)
        except IOError:
            logging.info('.build.yml not found')
            raise SandboxTaskFailureError('build.yml not found, and not overrided in params')

    def _merge_(self):
        repo_params = self.build_options.get('repo')
        conductor_params = self.build_options.get('conductor')
        build_params = {
            'distr': self.ctx.get(BuildDistr.name, self.build_options.get('distr')),
            # 'hookdir': self.build_options.get('hookdir'),
            'repo': {
                'name': self.ctx.get(BuildRepoName.name, repo_params.get('name')),
                'check_if_exist': self.ctx.get(BuildRepoCheckIfExist.name, repo_params.get('check_if_exist')),
                'dupload': self.ctx.get(BuildDupload.name, repo_params.get('dupload')),
                'user': self.ctx.get(BuildRepoUser.name, repo_params.get('user')),
            },
            'conductor': {
                'create_ticket': self.ctx.get(BuildConductorCreateTicket.name, conductor_params.get('create_ticket')),
                'branch': self.ctx.get(BuildConductorBranch.name, conductor_params.get('branch')),
                'projects': self.ctx.get(BuildConductorProjects.name, conductor_params.get('projects'))
            }
        }
        return build_params


__Task__ = StatboxDebuilder
