# import logging

# import os
import re
import shlex
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
# from sandbox.sdk2.ssh import Key as sshKey
# from sandbox.common.types import client

import sandbox.common.types.misc as ctm
from sandbox.projects.Statbox import STATFACE_TASK_RUNNER_LXC


REGISTRY = 'registry.yandex.net'


class StatboxDockrunner(sdk2.Task):
    """
    Sometimes registry is more convenient than sandbox even at its cost.
    """

    class Requirements(sdk2.Task.Requirements):
        # https://wiki.yandex-team.ru/sandbox/cookbook/#cores1multislot
        privileged = False
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

        dns = ctm.DnsType.DNS64
        disk_space = 50 * 1024

    class Parameters(sdk2.Task.Parameters):

        Container = sdk2.parameters.Container(
            'LXC Container',
            resource_type=STATFACE_TASK_RUNNER_LXC,
            platform='linux_ubuntu_18.04_bionic',
            default=870270439)

        DockerImage = sdk2.parameters.String('Docker image', required=True)
        DockerOpts = sdk2.parameters.String('Docker extra options', default='')

        Command = sdk2.parameters.String(
            'Command to execute in the container',
            required=True, multiline=True)
        BashWrap = sdk2.parameters.Bool(
            'Give the command as-is to `/bin/bash -c`', default=True)
        VaultEnv = sdk2.parameters.String(
            'Space-separated list of ENVVAR=vault_owner:vault_key',
            default='')
        RegistryAuth = sdk2.parameters.String(
            'Registry username::vault_owner::vault_name',
            default='robot-statface-api::STATFACE::robot-statface-api-qloud')

    def _get_vault_env_data(self):
        vault_env = self.Parameters.VaultEnv
        vault_env = [
            re.search(r'^([^=]+)=([^:]+):(.*)$', item).groups()
            for item in vault_env.split()
            if item]
        vault_env_data = {
            envvar: sdk2.Vault.data(vault_owner, vault_name)
            for envvar, vault_owner, vault_name in vault_env}
        return vault_env_data

    def on_execute(self):
        registry_username, registry_vault_owner, registry_vault_name = (
            self.Parameters.RegistryAuth.split('::', 2))
        registry_pass = sdk2.Vault.data(registry_vault_owner, registry_vault_name)

        vault_env = self._get_vault_env_data()

        with sdk2.helpers.ProcessLog(self, logger='docker.login') as logger:
            cmd = ('docker', 'login', '--username', registry_username, '--password-stdin', REGISTRY)
            proc = subprocess.Popen(
                cmd,
                stdin=subprocess.PIPE,
                stdout=logger.stdout,
                stderr=logger.stderr)
            # Pieces of `subprocess.check_output`
            proc.communicate(registry_pass)
            retcode = proc.poll()
            if retcode:
                raise subprocess.CalledProcessError(retcode, cmd)

        # # Effective command variations:
        # SEC1=... SEC2=... docker run -e SEC1 -e SEC2 ... "$image" /bin/bash -c "$cmd"
        # SEC1=... SEC2=... docker run -e SEC1 -e SEC2 ... "$image" $cmd

        run_env = {}
        cmd = [
            'docker', 'run',
            # '--rm',  # probably no point in adding that
            # '-i',  # no stdin here
            # '-t',  # might be useful, but only rarely.
        ]
        for envvar, secvalue in vault_env.items():
            cmd.extend(('-e', envvar))
            run_env[envvar] = secvalue

        cmd.extend(shlex.split(self.Parameters.DockerOpts))  # Note: '' -> []

        cmd.append(self.Parameters.DockerImage)

        if self.Parameters.BashWrap:
            cmd.extend((
                '/bin/bash',
                '-c',
                self.Parameters.Command))
        else:
            cmd.extend(shlex.split(self.Parameters.Command))

        # Secrets and args are valid, go get the image:
        with sdk2.helpers.ProcessLog(self, logger='docker.pull') as logger:
            subprocess.check_call(
                ('docker', 'pull', self.Parameters.DockerImage),
                stdout=logger.stdout, stderr=logger.stderr)

        # main
        with sdk2.helpers.ProcessLog(self, logger='run') as logger:
            subprocess.check_call(
                cmd,
                env=run_env,
                stdout=logger.stdout,
                stderr=logger.stderr)
