# -*- coding: utf-8 -*-

import datetime
import logging
import os
from collections import namedtuple

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.common.utils import server_url
from sandbox.common.types import task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr

from sandbox.projects.logs.release_helpers import GetLastGoodReleaseInfo


RESULT_DIR = 'libra.build'
YT_ATTR_BUILT_BY = 'built_by'
TOKENS = {'robot_ci_sessions_yt_token': ('hahn', 'arnold', 'hume', 'freud')}
NILE_UDF_PYTHON_VERSIONS = ['2.7', '3.6', '3.7', '3.8', '3.9']


class StatboxLibraResource(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = False
    auto_backup = True


class StatboxLibra3Resource(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = False
    auto_backup = True


class StatboxLibraNileUdfResource(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = False
    auto_backup = True


class StatboxLibraReleaseInfo(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    arcadia_revision = sdk2.resource.Attributes.String()
    arcadia_branch = sdk2.resource.Attributes.String()


class StatboxLibra3ReleaseInfo(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    arcadia_revision = sdk2.resource.Attributes.String()
    arcadia_branch = sdk2.resource.Attributes.String()


class StatboxLibraNileUdfReleaseInfo(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    arcadia_revision = sdk2.resource.Attributes.String()
    arcadia_branch = sdk2.resource.Attributes.String()


TargetCookInfo = namedtuple('TargetCookInfo', [
    'ya_make_description',
    'ya_make_resource_type',
    'release_resource_type',
    'ya_make_task_id_context_key',
    'source_dir',
    'source_artifact',
    'system_python_version',
    'local_result_path',
    'local_result_path_debug',
    'yt_path_versioned',
    'yt_path_link',
])

TARGETS_COOK_INFO = [
    TargetCookInfo(
        ya_make_description='Building libra library',
        ya_make_resource_type=StatboxLibraResource,
        release_resource_type=StatboxLibraReleaseInfo,
        ya_make_task_id_context_key='ya_make_task_id_libra',
        source_dir='quality/user_sessions/libra',
        source_artifact='libra.so',
        system_python_version='2.7',
        local_result_path='{}/libra.so'.format(RESULT_DIR),
        local_result_path_debug='{}/libra.so.debug'.format(RESULT_DIR),
        yt_path_versioned='//statbox/resources-versioned/libra.so',
        yt_path_link='//statbox/resources/libra.so',
    ),
    TargetCookInfo(
        ya_make_description='Building libra3 library',
        ya_make_resource_type=StatboxLibra3Resource,
        release_resource_type=StatboxLibra3ReleaseInfo,
        ya_make_task_id_context_key='ya_make_task_id_libra3',
        source_dir='quality/user_sessions/libra3',
        source_artifact='libra.so',
        system_python_version='3.7',
        local_result_path='{}/libra3.so'.format(RESULT_DIR),
        local_result_path_debug='{}/libra3.so.debug'.format(RESULT_DIR),
        yt_path_versioned='//statbox/resources-versioned/libra3.so',
        yt_path_link='//statbox/resources/libra3.so',
    ),
] + [
    TargetCookInfo(
        ya_make_description='Building libra_nile_udf{}'.format(system_python_version),
        ya_make_resource_type=StatboxLibraNileUdfResource,
        release_resource_type=StatboxLibraNileUdfReleaseInfo,
        ya_make_task_id_context_key='ya_make_task_id_libra_nile_udf{}'.format(system_python_version.replace('.', '_')),
        source_dir='statbox/nile/libra_python_udf/py{}'.format(system_python_version[0]),
        source_artifact='liblibra_python{}_udf.so'.format('' if system_python_version == '2.7' else '3'),
        system_python_version=system_python_version,
        local_result_path='{}/libra_nile_udf{}.so'.format(RESULT_DIR, system_python_version),
        local_result_path_debug='{}/libra_nile_udf{}.so.debug'.format(RESULT_DIR, system_python_version),
        yt_path_versioned='//statbox/resources-versioned/libra_nile_udf{}.so'.format(system_python_version),
        yt_path_link='//statbox/resources/libra_nile_udf{}.so'.format(system_python_version),
    )
    for system_python_version in NILE_UDF_PYTHON_VERSIONS
]


class StatboxLibraCook(sdk2.Task):
    """Сборка libra.so и python udf с либрой и загрузка на YT"""

    class Requirements(sdk2.Task.Requirements):
        # client_tags = Tag.LINUX_PRECISE
        environments = (PipEnvironment('yandex-yt'),)
        cores = 1
        ram = 52 * 1024
        disk_space = 64 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        tested_arcadia_url = sdk2.parameters.Bool(
            "Update from tested user_sessions branch",
            default_value=False
        )
        dry_run = sdk2.parameters.Bool(
            "Dry run, not upload artifacts to yt",
            default_value=False
        )

        with tested_arcadia_url.value[False]:
            reg_general_arcadia_url = sdk2.parameters.ArcadiaUrl(
                'Svn url for release. For example: arcadia:/arc/branches/userdata/<date>/arcadia@<rev>',
                required=True
            )

    def _get_libra_subtask(self, arcadia_url, cook_info):
        subtask_type = sdk2.Task["YA_MAKE_2"]

        # TODO: ttl somehow. or after - set ttl-attrs myself
        params = dict(
            arch='linux',
            checkout_arcadia_from_url=arcadia_url,
            targets=cook_info.source_dir,
            arts=os.path.join(cook_info.source_dir, cook_info.source_artifact),
            result_rt=cook_info.ya_make_resource_type.name,
            build_system='ya',
            build_type='release',
            use_aapi_fuse=True,
            aapi_fallback=True,
            check_return_code=True,
            result_single_file=True,
            strip_binaries=False,
            result_ttl='3'
        )

        if cook_info.system_python_version is not None:
            definition_flags = "-DUSE_SYSTEM_PYTHON={}".format(cook_info.system_python_version)
            if cook_info.system_python_version == '2.7':
                definition_flags += ' -DCFLAGS="-Wno-writable-strings"'
            elif cook_info.system_python_version in ('3.8', '3.9'):
                definition_flags += ' -DCFLAGS="-Wno-deprecated-declarations"'

            params.update(definition_flags=definition_flags)

        subtask = subtask_type(
            self,
            description=cook_info.ya_make_description,
            **params
        )
        return subtask

    def _need_build(self):
        for cook_info in TARGETS_COOK_INFO:
            if getattr(self.Context, cook_info.ya_make_task_id_context_key) == ctm.NotExists:
                return True
        return False

    def build_libra(self, arcadia_url):
        if not self._need_build():
            return

        subtasks = []
        for cook_info in TARGETS_COOK_INFO:
            subtasks.append(self._get_libra_subtask(arcadia_url, cook_info))
        for subtask in subtasks:
            sdk2.Task.server.task[subtask.id].update(
                requirements=dict(disk_space=8 * 1024 ** 2)
            )
            subtask.enqueue()

        for cook_info, subtask in zip(TARGETS_COOK_INFO, subtasks):
            setattr(self.Context, cook_info.ya_make_task_id_context_key, subtask.id)

        raise sdk2.WaitTask(
            subtasks,
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
        )

    def get_branch_from_url(self, arcadia_url):
        return Arcadia.parse_url(arcadia_url).branch

    def get_revision_from_url(self, arcadia_url):
        return Arcadia.parse_url(arcadia_url).revision

    def get_best_resource_for_type(self, arcadia_url, resource_type):
        logging.info('searching for resource type={res_type}, branch={branch}'.format(res_type=str(resource_type), branch=self.get_branch_from_url(arcadia_url)))

        libra_resource_query = sdk2.Resource.find(type=resource_type, state=ctr.State.READY, attrs={'arcadia_branch': self.get_branch_from_url(arcadia_url)}).limit(100)  # noqa
        if not libra_resource_query:
            return None

        libra_resource_items = [x for x in libra_resource_query]
        for item in libra_resource_items:
            logging.info('resource: {res} revision={revision}'.format(res=str(item), revision=getattr(item, 'arcadia_revision', 0)))

        if len(libra_resource_items) == 0:
            return None

        lss = lambda x, y: getattr(x, 'arcadia_revision', 0) > getattr(y, 'arcadia_revision', 0)
        comparator = lambda x, y: -1 if lss(x, y) else 1 if lss(y, x) else 0  # noqa

        libra_resource_items = sorted(libra_resource_items, comparator)
        revision_from_url = self.get_revision_from_url(arcadia_url)
        if len(libra_resource_items) > 0 and getattr(libra_resource_items[0], 'arcadia_revision') >= revision_from_url:
            return libra_resource_items[0]
        return None

    def get_best_arcadia_url(self, arcadia_url):
        best_resource = None
        for resource_type in [StatboxLibraReleaseInfo, StatboxLibra3ReleaseInfo, StatboxLibraNileUdfReleaseInfo]:
            current_resource = self.get_best_resource_for_type(arcadia_url, resource_type)
            if current_resource and (not best_resource or current_resource.arcadia_revision > best_resource.arcadia_revision):
                best_resource = current_resource

        if not best_resource:
            return arcadia_url

        return "arcadia:/arc/branches/{branch_name}/arcadia@{revision}".format(
            branch_name=best_resource.arcadia_branch,
            revision=best_resource.arcadia_revision,
        )

    def on_execute(self):
        import yt.wrapper as yt

        if self.Parameters.tested_arcadia_url:
            arcadia_url = self.get_best_arcadia_url(GetLastGoodReleaseInfo.get_last_success_resource())
        else:
            arcadia_url = self.get_best_arcadia_url(self.Parameters.reg_general_arcadia_url)

        self.build_libra(arcadia_url)
        self.set_info("Seems that all libraries are built")

        for cook_info in TARGETS_COOK_INFO:
            libra_resource = sdk2.Resource.find(
                resource_type=cook_info.ya_make_resource_type,
                task_id=getattr(self.Context, cook_info.ya_make_task_id_context_key),
            ).limit(1).first()

            libra_data = sdk2.ResourceData(libra_resource)

            result_path = cook_info.local_result_path
            result_path_debug = cook_info.local_result_path_debug

            sp.check_call(['mkdir', '-p', RESULT_DIR])
            sp.check_call(['cp', str(libra_data.path), result_path])
            sp.check_call(['chmod', '755', result_path])
            sp.check_call(['cp', result_path, result_path_debug])

            with sdk2.helpers.ProcessLog(self, logger="strip") as pl:
                sp.check_call(['strip', '--strip-debug', '--strip-unneeded', result_path], shell=False, stdout=pl.stdout, stderr=pl.stdout)
                sp.check_call(['strip', '--only-keep-debug', result_path_debug], shell=False, stdout=pl.stdout, stderr=pl.stdout)

            self.set_info("{} was stripped and now it's ready to upload".format(os.path.basename(result_path)))
            resource_kwargs = dict(
                arch=None, arcadia_branch=libra_resource.arcadia_branch, arcadia_revision=libra_resource.arcadia_revision, ttl=3,
            )
            libra_res = sdk2.ResourceData(cook_info.release_resource_type(self, os.path.basename(result_path), result_path, **resource_kwargs))
            libra_debug_res = sdk2.ResourceData(cook_info.release_resource_type(self, os.path.basename(result_path_debug), result_path_debug, **resource_kwargs))
            libra_res.ready()
            libra_debug_res.ready()

        today_date = datetime.date.today().strftime('%Y-%m-%d')
        sandbox_www = server_url()
        yt_attr_built_by_value = '{}/task/{}'.format(sandbox_www, self.id)

        for cook_info in TARGETS_COOK_INFO:
            self.set_info('{} YT path: {}/{}'.format(os.path.basename(cook_info.local_result_path), cook_info.yt_path_versioned, today_date))

        if self.Parameters.dry_run:
            return

        for token, clusters in TOKENS.iteritems():
            yt.config['token'] = sdk2.Vault.data('USERSESSIONSTOOLS', 'robot_ci_sessions_yt_token')
            for cluster in clusters:
                yt.config['proxy']['url'] = '{}.yt.yandex.net'.format(cluster)
                for cook_info in TARGETS_COOK_INFO:
                    yt.smart_upload_file(cook_info.local_result_path, destination='{}/{}'.format(cook_info.yt_path_versioned, today_date), placement_strategy='replace')
                    yt.set('{}/{}/@{}'.format(cook_info.yt_path_versioned, today_date, YT_ATTR_BUILT_BY), yt_attr_built_by_value)
                    yt.smart_upload_file(cook_info.local_result_path_debug, destination='{}/{}.debug'.format(cook_info.yt_path_versioned, today_date), placement_strategy='replace')
                    yt.set('{}/{}.debug/@{}'.format(cook_info.yt_path_versioned, today_date, YT_ATTR_BUILT_BY), yt_attr_built_by_value)
                    yt.link('{}/{}'.format(cook_info.yt_path_versioned, today_date), cook_info.yt_path_link, ignore_existing=False, force=True)
                self.set_info('{}: OK'.format(cluster))
