# coding: utf-8

import logging
import tempfile

import sandbox.common.types.task as ctt
from os import listdir
from os.path import join
from sandbox.common.errors import TaskFailure, TemporaryError
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import svn
from sandbox.projects.common import task_env

from sandbox import sdk2
from ...statinfra.extra import juggler_push

REVISOR_URL = 'arcadia:/arc/trunk/arcadia/statbox/revisor'
CUSTOM_SCRIPT_URL = 'arcadia:/arc/trunk/arcadia/statbox/statinfra/lxc_custom_script.jinja'
PIP_REQUIREMENTS_URL = 'arcadia:/arc/trunk/arcadia/statbox/statinfra/lxc_pip_requirements.txt'
STATENVS_MAP = dict(
    testing='betamrparser',
    production='mrparser'
)
RESOURCE_DESCRIPTION = 'rootfs.squashfs'
UBUNTU_RELEASE = 'precise'
CUSTOM_REPOS_TESTING = '''
deb http://dist.yandex.ru/statbox-common unstable/all/
deb http://dist.yandex.ru/statbox-common testing/all/
deb http://dist.yandex.ru/statbox-common testing/amd64/
deb http://dist.yandex.ru/statbox-common prestable/all/
deb http://dist.yandex.ru/statbox-common prestable/amd64/

deb http://dist.yandex.ru/common testing/all/
deb http://dist.yandex.ru/common testing/amd64/
deb http://dist.yandex.ru/search testing/all/
deb http://dist.yandex.ru/search testing/amd64/
deb http://dist.yandex.ru/common prestable/all/
deb http://dist.yandex.ru/common prestable/amd64/
deb http://dist.yandex.ru/yandex-precise testing/all/
deb http://dist.yandex.ru/yandex-precise testing/amd64/
deb http://dist.yandex.ru/yandex-precise prestable/all/
deb http://dist.yandex.ru/yandex-precise prestable/amd64/
deb http://search-precise.dist.yandex.ru/search-precise unstable/amd64/
deb http://search-precise.dist.yandex.ru/search-precise stable/amd64/
'''
CUSTOM_REPOS_STABLE = '''
deb http://dist.yandex.ru/statbox-common stable/all/
deb http://dist.yandex.ru/statbox-common stable/amd64/

deb http://dist.yandex.ru/common stable/all/
deb http://dist.yandex.ru/common stable/amd64/
deb http://dist.yandex.ru/search stable/all/
deb http://dist.yandex.ru/search stable/amd64/
deb http://dist.yandex.ru/system configs/all/
deb http://dist.yandex.ru/system precise/all/
deb http://dist.yandex.ru/system precise/amd64/
deb http://dist.yandex.ru/yandex-precise stable/all/
deb http://dist.yandex.ru/yandex-precise stable/amd64/

deb http://mirror.yandex.ru/old-ubuntu precise main restricted multiverse universe
deb http://mirror.yandex.ru/old-ubuntu precise-security main restricted multiverse universe
deb http://mirror.yandex.ru/old-ubuntu precise-updates main restricted multiverse universe
'''
BASE_PACKAGES = '''
libblas3gf
libffi-dev
libgfortran3
liblapack3gf
libquadmath0
python-scipy=0.15.1-yandex2
python-numpy=1:1.6.1-6ubuntu1
rsyslog
rsync
yandex-archive-keyring
yandex-internal-root-ca
yandex-search-common-apt
yandex-ubuntu-archive-apt
yandex-environment-{yandex_env}
'''
CUSTOM_PACKAGES = '''
curl
fping=4.0-5
git
libio-socket-ssl-perl=1.89-1
libclone-perl
libnetaddr-ip-perl=4.058+dfsg-2
mailutils
postgresql-client
python-cxoracle
python-dev
python-imaging
python-lxml
python-pip=6.1.1-yandex1
python-six=1.9.0-yandex1
python-software-properties
python-tk
python-virtualenv
r-base
squashfs-tools
tcpdump
'''
UNWANTED_PACKAGES = '''
juggler-client
^config-juggler
^p2p-
yandex-hbf-agent
yandex-hbf-agent-init
'''
DEFAULT_ENV_NAME = 'default'
JUGGLER_HOST = 'statbox-build-production'
JUGGLER_SERVICE = 'statbox-lxc-{env}'


def unknown_author(author, soft=False):
    allowed = ['robot-statinfra']
    if soft:
        allowed.extend([
            'seray',
            'komendart',
            'mvel',
            'dalamar',
        ])
    if author in allowed:
        return False
    return True


class RevisorPackages(object):
    def __init__(self, url, revision):
        self.revisor_path = self._checkout(url, revision)
        self.revisor_base = self._fill(self.revisor_path)

    def _checkout(self, url, revision):
        checkout_path = tempfile.mkdtemp()
        svn.Arcadia.checkout(url, checkout_path, revision=revision)
        return checkout_path

    def _fill(self, checkout_path):
        revisor_base = dict()
        for pkg_env in listdir(checkout_path):
            if any([pkg_env == '.svn', pkg_env == 'ya.make', pkg_env.endswith('.py'), pkg_env.endswith('.swp')]):
                continue
            p_file = join(checkout_path, pkg_env)
            if unknown_author(svn.Arcadia.info(p_file)['author'], soft=True):
                raise TaskFailure('{} author unknown'.format(pkg_env))
            p, e = pkg_env.split('.')
            if p not in revisor_base:
                revisor_base[p] = dict()
            with open(p_file) as f:
                v = f.read()
                revisor_base[p][e] = v.split()[0]
        return revisor_base

    def to_install(self, env):
        d = dict()
        for p in self.revisor_base:
            if DEFAULT_ENV_NAME in self.revisor_base[p]:
                d[p] = self.revisor_base[p][DEFAULT_ENV_NAME]
            elif env in self.revisor_base[p]:
                d[p] = self.revisor_base[p][env]
        return d


class StatboxLoxer(sdk2.Task):
    """Сборка LXC-контейнера для запуска STATINFRA_TASK"""
    name = 'STATBOX_LOXER'

    class Parameters(sdk2.Task.Parameters):
        env = sdk2.parameters.String('Environment name', default='testing')

    class Requirements(task_env.TinyRequirements):
        disk_space = 4096
        environments = (environments.PipEnvironment("Jinja2"),)

    @property
    def _custom_script(self):
        from jinja2 import Template
        if unknown_author(svn.Arcadia.info(PIP_REQUIREMENTS_URL)['author'], soft=True):
            raise TaskFailure('{} unknown author'.format(PIP_REQUIREMENTS_URL))
        pip_requirements = svn.Arcadia.cat(PIP_REQUIREMENTS_URL, revision=self.Context.arcadia_revision)
        apt_install = ' '.join(['{}={}'.format(k, v) for k, v in self.Context.packages_dict.iteritems()])
        if unknown_author(svn.Arcadia.info(CUSTOM_SCRIPT_URL)['author'], soft=True):
            raise TaskFailure('{} unknown author'.format(CUSTOM_SCRIPT_URL))
        custom_script_jinja = Template(svn.Arcadia.cat(CUSTOM_SCRIPT_URL, revision=self.Context.arcadia_revision))
        return custom_script_jinja.render(yandex_env=self.Parameters.env,
                                          statinfra_env=STATENVS_MAP[self.Parameters.env],
                                          apt_install=apt_install,
                                          pip_requirements=pip_requirements)

    def on_prepare(self):
        self.Context.arcadia_revision = svn.Arcadia.info('arcadia:/arc/trunk/arcadia')['entry_revision']

    def on_execute(self):
        if self.Parameters.env == 'production':
            resource_type = 'STATINFRA_LXC_CONTAINER'
            custom_repos = CUSTOM_REPOS_STABLE
        else:
            resource_type = 'STATINFRA_LXC_CONTAINER_BETA'
            custom_repos = CUSTOM_REPOS_STABLE + CUSTOM_REPOS_TESTING
        self.Context.packages_dict = RevisorPackages(
            REVISOR_URL, self.Context.arcadia_revision
        ).to_install(self.Parameters.env)
        custom_attrs = dict(
            revision__arcadia=self.Context.arcadia_revision,
            statinfra_env_spec=STATENVS_MAP[self.Parameters.env],
            **self.Context.packages_dict
        )
        push_to_preprod = True if self.Parameters.env == 'testing' else False
        if not self.Context.subtask_id:
            self.Context.subtask_id = 0
        if (self.Context.subtask_id == 0):
            subtask = sdk2.Task['SANDBOX_LXC_IMAGE'](
                self,
                description=self.Parameters.description,
                resource_type=resource_type,
                resource_description=RESOURCE_DESCRIPTION,
                ubuntu_release=UBUNTU_RELEASE,
                test_result_lxc=False,
                set_default=False,
                custom_attrs=custom_attrs,
                push_to_preprod=push_to_preprod,
                custom_image=True,
                install_common=False,
                custom_repos=custom_repos,
                custom_script=self._custom_script,
                base_packages=BASE_PACKAGES.format(yandex_env=self.Parameters.env),
                custom_packages=CUSTOM_PACKAGES,
                unwanted_packages=UNWANTED_PACKAGES,
            )
            subtask.enqueue()
            self.Context.subtask_id = subtask.id
            logging.debug("SANDBOX_LXC_IMAGE task: {}".format(subtask.id))
            raise sdk2.WaitTask(
                [self.Context.subtask_id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
            )
        else:
            if not all(task.status in ctt.Status.Group.SUCCEED for task in self.find(id=self.Context.subtask_id)):
                self.Context.subtask_id = 0
                raise TemporaryError("Build STATINFRA_LXC_CONTAINER failed")

    def on_success(self, prev_status):
        juggler_push(
            host=JUGGLER_HOST,
            service=JUGGLER_SERVICE.format(env=self.Parameters.env),
            status='OK',
            description='',
        )
        super(StatboxLoxer, self).on_success(prev_status)

    def on_failure(self, prev_status):
        juggler_push(
            host=JUGGLER_HOST,
            service=JUGGLER_SERVICE.format(env=self.Parameters.env),
            status='CRIT',
            description='Build STATINFRA_LXC_CONTAINER failed',
        )
        super(StatboxLoxer, self).on_failure(prev_status)

    def on_break(self, prev_status):
        juggler_push(
            host=JUGGLER_HOST,
            service=JUGGLER_SERVICE.format(env=self.Parameters.env),
            status='CRIT',
            description='Build STATINFRA_LXC_CONTAINER finished with exception',
        )
        super(StatboxLoxer, self).on_break(prev_status)
