# -*- coding: utf-8 -*-
import tempfile
import subprocess

from os import environ
from os.path import exists, join
from sandbox.common.errors import TaskFailure
from sandbox.projects.statinfra import STATINFRA_LXC_CONTAINER
from sandbox.sdk2.vcs import svn

from sandbox import sdk2

REPO_MAP = {
    'python-clickhouse-client': 'arcadia:/arc/trunk/arcadia/library/python/clickhouse_client',
    'python-statinfra': 'arcadia:/arc/trunk/arcadia/statbox/python-statinfra',
    'python-statface-client': 'arcadia:/arc/trunk/arcadia/library/python/statface_client',
    'python-step-client': 'arcadia:/arc/trunk/arcadia/statbox/step_client',
    'python-signurl-bindings': 'arcadia:/arc/trunk/arcadia/statbox/bindings/signurl',
}

PYPIRC = '''
[distutils]
index-servers =
    yandex

[yandex]
repository: https://pypi.yandex-team.ru/simple/
username: {username}
password: {password}
'''


class StatboxPopeyeUploader(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 4096

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container(
            'LXC Container',
            resource_type=STATINFRA_LXC_CONTAINER,
            owner='STATINFRA',
            attrs=None,
            required=True,
        )
        package_name = sdk2.parameters.String('Package to upload to pypi', description='name')

    def _get_version(self):
        revisor = svn.Arcadia.checkout('arcadia:/arc/trunk/arcadia/statbox/revisor', tempfile.mkdtemp())
        for e in ['default', 'production']:
            package_file = join(revisor, '{}.{}'.format(self.Parameters.package_name, e))
            if exists(package_file):
                with open(package_file) as f:
                    return f.read()
        raise TaskFailure("Can't determine production package version")

    def on_execute(self):
        self.Context.ver = self._get_version()
        self.set_info('Found {} version'.format(self.Context.ver))
        self.Context.repo = REPO_MAP[self.Parameters.package_name]
        repo_changelog = '{}/debian/changelog'.format(self.Context.repo)
        for line in svn.Arcadia.blame(repo_changelog).splitlines():
            if ' ({}) '.format(self.Context.ver) in line:
                self.Context.rev = line.split()[0]
                self.set_info('Found {} revision'.format(self.Context.rev))
                break
        else:
            raise TaskFailure("Can't determine revision which is in production")
        checkout = svn.Arcadia.checkout(self.Context.repo, tempfile.mkdtemp(), revision=self.Context.rev)
        pypi_user = sdk2.Vault.data('STATINFRA', 'robot-statinfra-pypi-access')
        pypi_pass = sdk2.Vault.data('STATINFRA', 'robot-statinfra-pypi-secret')
        with open('{}/.pypirc'.format(environ['HOME']), 'w') as pypirc:
            pypirc.write(PYPIRC.format(username=pypi_user, password=pypi_pass))
        venv2, venv3 = tempfile.mkdtemp(), tempfile.mkdtemp()
        with sdk2.helpers.ProcessLog(self, logger='prepare_venv2') as log2:
            subprocess.check_call([
                'virtualenv', '--python', '/usr/bin/python', venv2
            ], stdout=log2.stdout, stderr=log2.stderr)
        with sdk2.helpers.ProcessLog(self, logger='prepare_venv3') as log3:
            subprocess.check_call([
                'virtualenv', '--python', '/usr/bin/python3.5', venv3
            ], stdout=log3.stdout, stderr=log3.stderr)
        with sdk2.helpers.ProcessLog(self, logger='py2_build_upload') as log0:
            subprocess.check_call([
                join(venv2, 'bin/python'), 'setup.py', 'sdist', 'bdist_wheel', 'upload', '-r', 'yandex',
            ], cwd=checkout, stdout=log0.stdout, stderr=log0.stderr)
        self.set_info('Successfully uploaded py2 distibution to PyPI')
        with sdk2.helpers.ProcessLog(self, logger='py3_build_upload') as log1:
            if subprocess.call([
                join(venv3, 'bin/python'), 'setup.py', 'bdist_wheel', 'upload', '-r', 'yandex'
            ], cwd=checkout, stdout=log1.stdout, stderr=log1.stderr):
                self.set_info("Can't build&upload py3 distribution")
            else:
                self.set_info('Successfully uploaded py3 distibution to PyPI')
