# coding: utf-8

import logging
import tempfile

import datetime
import requests
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from os import getcwd
from os.path import join
from sandbox.common import share
from sandbox.common import utils as su
from sandbox.projects.porto import BuildPortoLayer
from sandbox.projects.common import task_env
from sandbox.projects.statinfra.extra import STATBOX_PACKAGES_LIST

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process

from sandbox import common
from sandbox import sdk2

from sandbox.projects.porto.common.resource_types import PORTO_LAYER_YT

PORTO_LAYER_RESOURCE_NAME = 'PORTO_LAYER_YT_STATBOX'
PORTO_LAYER_COMPRESSION = 'tar.gz'
JUGGLER_HOST = 'statbox-build-production'
JUGGLER_SERVICE = 'porto-layer'


def juggler_push(host, service, status, description):
    def requests_retry():
        s = requests.Session()
        r = requests.packages.urllib3.util.retry.Retry(total=3, backoff_factor=0.5)
        a = requests.adapters.HTTPAdapter(max_retries=r)
        s.mount('http://', a)
        return s
    reply = requests_retry().post(
        'http://juggler-push.search.yandex.net/events',
        json={
            'source': 'sandbox',
            'events': [
                {
                    'host': host,
                    'service': service,
                    'status': status,
                    'description': description,
                }
            ]
        },
        timeout=10,
    )
    logging.info(
        'Juggler event (service: %s; status: %s) returned %s code: %s',
        service, status, reply.status_code, reply.text,
    )


class StatboxPortoPorno(sdk2.Task):
    """Сборка porto-слоя для запуска statbox'овых задач на YT"""
    name = 'STATBOX_PORTO_PORNO'

    class Requirements(task_env.TinyRequirements):
        disk_space = 10 * 1024  # 10 Gb
        environments = (environments.PipEnvironment("yandex-yt"),)

    class Parameters(sdk2.Task.Parameters):
        def resource_types_with_prefixes(*args):
            return [r for r in sdk2.Resource if any(r.name.startswith(arg) for arg in args)]

        parent_layer = sdk2.parameters.Resource(
            "Parent layer",
            resource_type=resource_types_with_prefixes('PORTO_LAYER')
        )
        script_url = sdk2.parameters.ArcadiaUrl('Setup script URL', required=True)
        script2_url = sdk2.parameters.ArcadiaUrl('Post OS setup script URL', required=False)
        layer_filename = sdk2.parameters.String('Layer filename', required=True)
        clusters = sdk2.parameters.String('Target YT clusters', multiline=True, required=True)
        yt_token_vault_name = sdk2.parameters.String('YT token vault name', required=True)
        debug_build = sdk2.parameters.Bool("Debug build", default=False)

    def _get_yt_client(self, cluster):
        from yt.wrapper import YtClient
        return YtClient(cluster, token=sdk2.Vault.data(self.Parameters.yt_token_vault_name))

    def on_execute(self):
        full_filename = '{}.{}'.format(self.Parameters.layer_filename, PORTO_LAYER_COMPRESSION)
        if not self.Context.build_layer_task_id:
            if self.Parameters.parent_layer is None:
                parent_layer = PORTO_LAYER_YT.find(
                    state=ctr.State.READY,
                    attrs={'name_without_datetime': 'porto_layer_search_ubuntu_precise_app'}
                ).first()
            else:
                parent_layer = self.Parameters.parent_layer
            layer_build_params = {
                BuildPortoLayer.ParentLayer.name: parent_layer,
                BuildPortoLayer.LayerType.name: PORTO_LAYER_RESOURCE_NAME,
                BuildPortoLayer.LayerName.name: self.Parameters.layer_filename,
                BuildPortoLayer.Compress.name: PORTO_LAYER_COMPRESSION,
                BuildPortoLayer.ScriptUrl.name: self.Parameters.script_url,
                BuildPortoLayer.MergeLayers.name: False,
                BuildPortoLayer.DebugBuild.name: self.Parameters.debug_build
            }
            if self.Parameters.script2_url:
                layer_build_params.update(
                    {
                        BuildPortoLayer.Script2Url.name: self.Parameters.script2_url
                    }
                )
            logging.debug("BUILD_PORTO_LAYER build params: %s", str(layer_build_params))
            task_class = sdk2.Task['BUILD_PORTO_LAYER']
            layer_build_task = task_class(
                self,
                description=self.Parameters.description,
                **{
                    key: value.id if isinstance(value, sdk2.Resource) else value
                    for key, value in layer_build_params.iteritems()
                }
            ).enqueue()
            self.Context.build_layer_task_id = layer_build_task.id

            logging.debug("BUILD_PORTO_LAYER build task: %d", layer_build_task.id)

            raise sdk2.WaitTask(
                [self.Context.build_layer_task_id],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
            )
        else:
            if not all(
                task.status in ctt.Status.Group.SUCCEED
                for task in self.find(id=self.Context.build_layer_task_id)
            ):
                raise common.errors.TaskFailure("Build porto layer failed")

            layer_resource = sdk2.Resource.find(
                task_id=self.Context.build_layer_task_id, type=PORTO_LAYER_RESOURCE_NAME
            ).first()
            cwd = getcwd()
            share.skynet_get(layer_resource.skynet_id, cwd)

            packages = dict()
            if run_process(
                [
                    'tar', '--list', '--gzip', '--file', join(cwd, full_filename), STATBOX_PACKAGES_LIST
                ],
                check=False,
            ).returncode == 0:
                tmp = tempfile.mkdtemp()
                run_process(
                    [
                        'tar', '--extract', '--gzip', '--file', join(cwd, full_filename), STATBOX_PACKAGES_LIST,
                    ], work_dir=tmp)
                with open(join(tmp, STATBOX_PACKAGES_LIST)) as pkgs_file:
                    for pkg_line in pkgs_file.readlines():
                        pkg, ver = pkg_line.split()
                        packages[pkg] = ver
            self.set_info('\n'.join(['{}={}'.format(k, v) for k, v in packages.iteritems()]))

            today = datetime.date.isoformat(datetime.date.today())
            yt_path = '//statbox/resources-versioned/{}/{}'.format(full_filename, today)
            for cluster in self.Parameters.clusters.split():
                yt_client = self._get_yt_client(cluster)
                logging.debug('Uploading layer to {}{}'.format(cluster, yt_path))
                with open(full_filename) as layer_file:
                    yt_client.write_file(yt_path, layer_file)
                logging.debug('Setting attributes')
                yt_client.set("{}/@sandbox_resource_id".format(yt_path), layer_resource.id)
                yt_client.set('{}/@packages'.format(yt_path), packages)
                logging.debug('Linking {} to //statbox/resources/{}'.format(full_filename, yt_path))
                yt_client.link(
                    yt_path, '//statbox/resources/{}'.format(full_filename),
                    ignore_existing=False, force=True
                )

    def get_description(self, status):
        task_url = '{}/task/{}'.format(su.server_url(), self.id)
        layer = self.Parameters.layer_filename
        return 'Task: {} Layer: {} Clusters: {} Status: {}' .format(task_url, layer, self.Parameters.clusters, status)

    def get_service(self):
        return '{}-{}'.format(self.Parameters.layer_filename, '-'.join(self.Parameters.clusters.split()))

    def on_success(self, prev_status):
        juggler_push(
            host=JUGGLER_HOST,
            service=self.get_service(),
            status='OK',
            description=self.get_description('Success')
        )
        super(StatboxPortoPorno, self).on_success(prev_status)

    def on_failure(self, prev_status):
        juggler_push(
            host=JUGGLER_HOST,
            service=self.get_service(),
            status='CRIT',
            description=self.get_description('Failure')
        )
        super(StatboxPortoPorno, self).on_failure(prev_status)
