# import logging

import json
import logging
import os
from os.path import join
import shlex
import tempfile
import time

from sandbox.common.types import misc
from sandbox.projects.Statbox import STATBOX_RELOOSER_LXC
from sandbox.projects.common.arcadia.sdk import mount_arc_path
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.ssh import Key as sshKey

from sandbox import sdk2

REGISTRY = 'registry.yandex.net'
DOCKER_CONFIG_FILE = '/etc/docker/daemon.json'

LOGGER = logging.getLogger(__name__)


class mount_git:
    def __init__(self, repo, item):
        self.repo = repo
        self.item = item

    def __enter__(self):
        workdir = tempfile.mkdtemp()
        subprocess.check_call(['git', 'clone', '--recurse-submodules', self.repo, workdir])
        subprocess.check_call(['git', 'checkout', self.item, '--'], cwd=workdir)
        return workdir

    def __exit__(self, exc_type, exc_val, exc_tb):
        pass


class StatboxRelooserExperimental(sdk2.Task):
    """ A full-copy of the StatboxRelooser for experimental modifications """

    class Requirements(sdk2.Task.Requirements):

        # client_tags = client.Tag.LINUX_XENIAL
        dns = misc.DnsType.DNS64

        cores = 1
        ram = 8192
        # disk_space = 50 * 1024

        # class Caches(sdk2.Requirements.Caches):
        #     pass

    class Parameters(sdk2.Task.Parameters):

        Container = sdk2.parameters.Container('LXC Container', resource_type=STATBOX_RELOOSER_LXC, platform='linux_ubuntu_18.04_bionic', required=True)
        GitRepository = sdk2.parameters.String('Repository with Dockerfile (arcadia, arc, git)', required=True)
        GitCheckoutItem = sdk2.parameters.String('Branch or commit or tag or revision', default='master')
        ReleaserWorkflow = sdk2.parameters.String('Releaser workflow or command(s)', multiline=True, required=True)
        GitCloneVaultName = sdk2.parameters.String('Git clone ssh key (vault name)', required=True)
        GitCloneVaultOwner = sdk2.parameters.String('Git clone ssh key (vault owner)', required=True)
        GitUserName = sdk2.parameters.String('Git commit user.name')
        GitUserEmail = sdk2.parameters.String('Git commit user.email')
        RegistryUsername = sdk2.parameters.String('Registry username')
        RegistryVaultName = sdk2.parameters.String('Registry password (vault name)')
        RegistryVaultOwner = sdk2.parameters.String('Registry password (vault owner)')
        OAuthVaultName = sdk2.parameters.String('Qloud OAuth token (vault name)')
        OAuthVaultOwner = sdk2.parameters.String('Qloud OAuth token (vault owner)')
        RamdriveSize = sdk2.parameters.Integer(
            'Create a RAM drive of specified size (in GiB) for dockerd',
            default=None,
        )

    def _make(self, workdir, target):
        with sdk2.helpers.ProcessLog(self, logger='make.{}'.format(target)) as log0:
            dry_run_error = subprocess.call(
                ['make', '--dry-run', target], cwd=workdir, stdout=log0.stdout, stderr=log0.stderr)
            if not dry_run_error:
                subprocess.check_call(['make', target], cwd=workdir, stdout=log0.stdout, stderr=log0.stderr)

    def on_enqueue(self, **kwargs):
        ramdrive_size = self.Parameters.RamdriveSize
        if ramdrive_size:
            self.Requirements.ramdrive = misc.RamDrive(misc.RamDriveType.TMPFS, int(ramdrive_size * 1024), None)
        super(StatboxRelooserExperimental, self).on_enqueue(**kwargs)

    def _repo_mount(self, repo, item):
        if repo.startswith('arcadia:/'):
            if isinstance(item, int):
                repo = '{}@{}'.format(repo, item)
            return mount_arc_path(repo)
        elif repo.startswith('arc:/'):
            if self.Parameters.GitUserName:
                os.environ['YA_USER'] = self.Parameters.GitUserName
                with open(join(os.getenv('HOME'), '.arcconfig'), 'w') as f:
                    f.write('[user]\n')
                    f.write('  name = {}\n'.format(self.Parameters.GitUserName))
            return mount_arc_path('arcadia-arc:/#{}'.format(item))
        else:
            with sdk2.helpers.ProcessLog(self, logger='git.config') as log3:
                if self.Parameters.GitUserName:
                    subprocess.check_call(['git', 'config', '--global', 'user.name', self.Parameters.GitUserName],
                                          stdout=log3.stdout, stderr=log3.stderr)
                if self.Parameters.GitUserEmail:
                    subprocess.check_call(['git', 'config', '--global', 'user.email', self.Parameters.GitUserEmail],
                                          stdout=log3.stdout, stderr=log3.stderr)
            return mount_git(repo, item)

    def _docker_to_ramdrive(self):
        ramdrive = self.ramdrive
        LOGGER.info("Configuring docker to use ramdrive %r", ramdrive)

        with open(DOCKER_CONFIG_FILE, 'r') as fobj:
            docker_config = json.load(fobj)

        if self.ramdrive:
            docker_config['data-root'] = '{}/docker'.format(ramdrive.path)
        else:
            docker_config.pop('data-root', None)
        with open(DOCKER_CONFIG_FILE, 'w') as fobj:
            json.dump(docker_config, fobj, indent=4)

        LOGGER.info('docker config at %r edited to: %s', DOCKER_CONFIG_FILE, json.dumps(docker_config))

        with sdk2.helpers.ProcessLog(self, logger='docker-restart') as log0:
            subprocess.check_call(['sudo', '/etc/init.d/docker', 'restart'], stdout=log0.stdout, stderr=log0.stderr)
            # Wait for docker:
            timelimit = time.time() + 15.0
            while True:
                try:
                    subprocess.check_call(['docker', 'info'], stdout=log0.stdout, stderr=log0.stderr)
                except Exception:
                    if time.time() > timelimit:
                        raise
                    LOGGER.info("Waiting for docker to come up...")
                    time.sleep(0.8)
                else:
                    break
            subprocess.check_call(['docker', 'info'], stdout=log0.stdout, stderr=log0.stderr)

        LOGGER.info("docker ramdrive config is done.")

    def on_execute(self):
        self._docker_to_ramdrive()

        # docker login
        with sdk2.helpers.ProcessLog(self, logger='docker.login') as log2:
            if self.Parameters.RegistryVaultOwner and self.Parameters.RegistryVaultName and self.Parameters.RegistryUsername:
                registry_pass = sdk2.Vault.data(self.Parameters.RegistryVaultOwner, self.Parameters.RegistryVaultName)
                subprocess.check_call(['docker', 'login', '-p', registry_pass, '-u', self.Parameters.RegistryUsername, REGISTRY], stdout=log2.stdout, stderr=log2.stderr)

        # run releaser with commands
        with sshKey(self, self.Parameters.GitCloneVaultOwner, self.Parameters.GitCloneVaultName):
            with self._repo_mount(self.Parameters.GitRepository, self.Parameters.GitCheckoutItem) as wd:
                workdir = join(wd, self.Parameters.GitRepository[5:]) if self.Parameters.GitRepository.startswith('arc:/') else wd
                with sdk2.helpers.ProcessLog(self, logger='releaser') as log1:
                    # set oauth_token
                    if self.Parameters.OAuthVaultName and self.Parameters.OAuthVaultOwner:
                        os.environ['RELEASER_OAUTH_TOKEN'] = sdk2.Vault.data(self.Parameters.OAuthVaultOwner, self.Parameters.OAuthVaultName)
                    # make release.hjson primary
                    if os.path.exists(join(workdir, 'release.hjson')) and not os.path.exists(join(workdir, '.release.hjson')):
                        os.environ['RELEASER_PROJECT_CONFIG'] = './release.hjson'
                    self._make(workdir, 'prereleaser')
                    if self.Parameters.ReleaserWorkflow:
                        if '\n' in self.Parameters.ReleaserWorkflow:
                            lines = self.Parameters.ReleaserWorkflow.splitlines()
                        else:
                            lines = [self.Parameters.ReleaserWorkflow]
                        for line in lines:
                            if line.startswith('> '):
                                cmd = shlex.split(line[2:])
                            else:
                                cmd = ['releaser'] + shlex.split(line)
                            env = os.environ.copy()
                            if self.Parameters.GitRepository.startswith('arc:/'):
                                if self.Parameters.GitUserName:
                                    env.update({'LOGNAME': self.Parameters.GitUserName})
                            subprocess.check_call(cmd, cwd=workdir, env=env, stdout=log1.stdout, stderr=log1.stderr)
                    self._make(workdir, 'postreleaser')

            # subprocess.check_call(['cat', 'changelog.md'], cwd=WORKDIR, stdout=log1.stdout, stderr=log1.stdout)
        # logging.info('{} {}'.format(type(self.Parameters.custom_workflow), dir(self.Parameters.custom_workflow)))
