# -*- coding: utf-8 -*-
import logging
import tarfile
import tempfile

import time
from os.path import join, exists
from sandbox.common.utils import server_url
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.ssh import Key

from sandbox import sdk2
from sandbox.common import errors
from .. import STATBOX_REVISOR_CHECKOUT
from ..StatboxDebuilder import Dist
from sandbox.projects.common import task_env


def clean_changelog(changelog):
    clean_changelog = list()
    for line in '\n'.join(changelog).split('\n'):
        if line == 'Note: mandatory check (NEED_CHECK) was skipped':
            continue
        else:
            clean_changelog.append(line)
        # if line.startswith('[ '):
        #     clean_changelog.append(line)
        # if line.startswith(' * '):
        #     clean_changelog.append(line)
    return '\n'.join(clean_changelog)


class StatboxTicketToStable(sdk2.Task):

    class Requirements(task_env.TinyRequirements):
        disk_space = 4096

    class Parameters(sdk2.Task.Parameters):
        to_stable = sdk2.parameters.String(
            'Packages and optionally versions',
            description='name=version | name',
            multiline=True,
        )

    def on_execute(self):
        revisor = svn.Arcadia.checkout('arcadia:/arc/trunk/arcadia/statbox/revisor', tempfile.mkdtemp())

        if '=' in self.Parameters.to_stable:
            packages = {p.split('=')[0]: p.split('=')[1] for p in self.Parameters.to_stable.split()}
        else:
            packages = dict()
            for p in self.Parameters.to_stable.split():
                testing_path = join(revisor, '{}.testing'.format(p))
                if exists(testing_path):
                    packages[p] = open(testing_path).read().split()[0]
                else:
                    raise errors.TaskError("{} testing branch is not under revisor control".format(p))
        self.Context.next_production = packages

        changelog = list()
        production = dict()
        for p in packages:
            production_path = join(revisor, '{}.production'.format(p))
            testing_path = join(revisor, '{}.testing'.format(p))
            if exists(production_path):
                production[p] = open(production_path).read().split()[0]
                start_changelog, stop_changelog = False, False
                _changelog = list()
                for log_entry in svn.Arcadia.log(testing_path):
                    if "{} {}".format(p, packages[p].rsplit('.', 1)[0]) in log_entry['msg']:
                        start_changelog = True
                    if "{} {}".format(p, production[p].rsplit('.', 1)[0]) in log_entry['msg']:
                        stop_changelog = True
                    if start_changelog and not stop_changelog:
                        _changelog.append(log_entry['msg'])
                    if start_changelog and stop_changelog:
                        break
                if not start_changelog:
                    _changelog.append("{}'s changelog wasn't recognized".format(p))
                    self.set_info('Rolling {} from {} to {} without changelog'.format(p, production[p], packages[p]))
                if start_changelog and not stop_changelog:
                    if len(_changelog) > 3:
                        _changelog = _changelog[:3]
                        _changelog.append('[ ... ]\n * ...')
                changelog.append('\n'.join(_changelog))
            else:
                raise errors.TaskError("{} production branch is not under revisor control".format(p))
        self.Context.current_production = production
        self.Context.changelog = clean_changelog(changelog)
        self.set_info(self.Context.changelog)

        with tarfile.open('revisor.tar.gz', 'w:gz') as t:
            t.add(revisor, 'revisor')
        sdk2.ResourceData(STATBOX_REVISOR_CHECKOUT(self, 'revisor.tar.gz', 'revisor.tar.gz')).ready()

    def on_release(self, additional_parameters):
        # в on_release дмувнуть и закоммитить в ревизор
        for p in self.Context.next_production:
            logging.info('{} {} will dmove'.format(p, self.Context.next_production[p]))
            self._dmove(p, self.Context.next_production[p], 'stable')
        with Key(self, 'STATINFRA', 'robot-statinfra-ssh'):
            revisor = svn.Arcadia.checkout('arcadia:/arc/trunk/arcadia/statbox/revisor', tempfile.mkdtemp())
            for p in self.Context.next_production:
                with open(join(revisor, '{}.production'.format(p)), 'w') as f:
                    f.write(self.Context.next_production[p])
            commit_message = '{}/task/{}'.format(server_url(), self.id)
            commit_message += '\n'
            # TODO: брать список пакетов из словаря next_production
            commit_message += '\n'.join(self.Parameters.to_stable.split()).replace('=', ' ')
            commit_message += '\n\n'
            commit_message += self.Context.changelog
            svn.Arcadia.commit(revisor, commit_message, user='robot-statinfra')

    # TODO: вынести в либу и заиспользовать в STATBOX_DEBUILDER тоже
    def _dmove(self, package, version, tobranch):
        with Key(self, 'STATINFRA', 'robot-statinfra-ssh'):
            frombranch = 'unstable'
            for _ in range(12):
                j = Dist().find_package(package, version)
                if j:
                    frombranch = j[0]['environment']
                    repo = j[0]['repository']
                    break
                else:
                    time.sleep(5)
            if frombranch == tobranch:
                logging.info('Package(s) already in {}'.format(tobranch))
                return
            dmove_cmd = ['ssh',
                         'robot-statinfra@dupload.dist.yandex.ru',
                         'sudo',
                         'dmove',
                         repo,
                         tobranch,
                         package,
                         version,
                         frombranch]
            for _ in range(12):
                _dmove = run_process(dmove_cmd, log_prefix='dmove_{}'.format(tobranch), check=False)
                if _dmove.returncode == 0:
                    logging.info('Package(s) dmoved to {} (from {})'.format(tobranch, frombranch))
                    return
                else:
                    time.sleep(5)
            else:
                raise errors.TaskError("Package(s) weren't dmoved")
