import os
import shutil
import requests

from sandbox import sdk2
from sandbox.sdk2.vcs import git
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.ssh import Key
# from sandbox.common.types import client
from sandbox.common.types import misc
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import server_url

from sandbox.projects.Statbox import STATBOX_TOXIC_LXC


TOX_OUT = 'tox_out'
MAKEFILES = ['Sandbox.Makefile', 'Sandbox.makefile', 'sandbox.makefile']


class StatboxToxic(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        # client_tags = client.Tag.LINUX_XENIAL
        privileged = True
        dns = misc.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container('LXC Container', resource_type=STATBOX_TOXIC_LXC, platform='linux_ubuntu_16.04_xenial', required=True)
        apt_default_release = sdk2.parameters.String('Default release to install packages from (APT::Default-Release)',
                                                     default='stable')
        satisfy_depends = sdk2.parameters.Bool('Satisfy debian/control dependencies', default=False)
        git_clone_ssh_key = sdk2.parameters.String('git clone ssh key (vault name)', default='robot-statinfra-ssh-key')
        git_clone_ssh_owner = sdk2.parameters.String('git clone ssh key (vault owner)', default='STATINFRA')

    def _github_status(self, status, description):
        status_url = self.Context.githubEvent['payload']['repository']['statuses_url'].format(
            sha=self.Context.githubEvent['payload']['pull_request']['head']['sha']).replace(
            'https://github.yandex-team.ru', 'https://github-token-proxy.stat.yandex.net')
        # headers = {
        #     'Authorization': 'token {token}'.format(token=sdk2.Vault.data('STATINFRA', 'stattester_github_token'))}
        body = {'state': status,
                'target_url': '{}/task/{}'.format(server_url(), self.id),
                'description': description,
                'context': 'SandboxToxic'}
        return requests.post(status_url, json=body)

    def _make(self, target):
        with sdk2.helpers.ProcessLog(self, logger='make.{}'.format(target)) as log0:
            dry_run_error = subprocess.call(
                ['make', '--makefile={}'.format(self.Context.makefile), '--dry-run', target], cwd=self.Context.workdir,
                stdout=log0.stdout, stderr=log0.stdout)
            if not dry_run_error:
                subprocess.check_call(['make', '--makefile={}'.format(self.Context.makefile), target],
                                      cwd=self.Context.workdir, stdout=log0.stdout, stderr=log0.stdout)

    def on_enqueue(self):
        self._github_status('pending', 'Just a moment, please!!')

    def on_prepare(self):
        self.Context.workdir = self.Context.githubEvent['payload']['repository']['name']
        with Key(self, self.Parameters.git_clone_ssh_owner, self.Parameters.git_clone_ssh_key):
            repo = git.Git(self.Context.githubEvent['payload']['pull_request']['head']['repo']['ssh_url'])
            repo.clone(self.Context.workdir, self.Context.githubEvent['payload']['pull_request']['head']['ref'])
        self.Context.makefile = None
        for i in MAKEFILES:  # type: str
            if not os.path.exists('{}/{}'.format(self.Context.workdir, i)):
                continue
            if self.Context.makefile:
                raise TaskFailure('There are several sandbox.makefiles')
            self.Context.makefile = i

    def on_execute(self):
        # make -f sandbox.makefile predepends
        if self.Context.makefile:
            self._make('predepends')

        # prepare stuff
        with sdk2.helpers.ProcessLog(self, logger='preparestuff') as log3:
            # add ppa.list
            if os.path.exists('{}/{}'.format(self.Context.workdir, 'ppa.list')):
                subprocess.check_call('cat ppa.list | xargs --max-args=1 add-apt-repository --yes',
                                      cwd=self.Context.workdir, shell=True, stdout=log3.stdout, stderr=log3.stdout)

            # apt keys
            if os.path.exists('{}/{}'.format(self.Context.workdir, 'apt-keys.list')):
                subprocess.check_call(
                    'cat apt-keys.list | xargs --max-args=1 apt-key adv --keyserver keyserver.ubuntu.com --recv-keys',
                    cwd=self.Context.workdir, shell=True, stdout=log3.stdout, stderr=log3.stdout)

        # add sources.list
        if os.path.exists('{}/{}'.format(self.Context.workdir, 'sources.list')):
            shutil.copyfile('{}/{}'.format(self.Context.workdir, 'sources.list'),
                            '/etc/apt/sources.list.d/statbox-toxic.list')

        # satisfying depends
        if self.Parameters.satisfy_depends:
            # import sys
            # sys.path.append('/usr/local/lib/python2.7/dist-packages')
            # from debian import deb822

            with sdk2.helpers.ProcessLog(self, logger='satisfydepends') as log2:
                if self.Parameters.apt_default_release:
                    with open('/etc/apt/apt.conf', 'a') as apt_conf:
                        apt_conf.write('APT::Default-Release "{}";\n'.format(self.Parameters.apt_default_release))
                subprocess.check_call(['apt-get', 'update'], stdout=log2.stdout, stderr=log2.stdout)

                # build_depends = deb822.Deb822(open('{}/debian/control'.format(self.Context.workdir)), ['Build-Depends'])
                # depends = deb822.Deb822({'Package': 'statbox-toxic-deps', 'Depends': build_depends['Build-Depends']})
                # with open('statbox-toxic-deps-control', 'w') as depends_file:
                #     depends.dump(depends_file)
                # subprocess.check_call(['equivs-build', 'statbox-toxic-deps-control'], stdout=log2.stdout, stderr=log2.stdout)
                # subprocess.check_call(['dpkg', '--force-depends', '--install', 'statbox-toxic-deps_1.0_all.deb'], stdout=log2.stdout, stderr=log2.stdout)
                # subprocess.check_call(['apt-get', '--fix-broken', '--yes', '--verbose-versions', '--no-remove', 'install'], stdout=log2.stdout, stderr=log2.stdout)
                subprocess.check_call(['/usr/lib/pbuilder/pbuilder-satisfydepends-classic'], cwd=self.Context.workdir, stdout=log2.stdout, stderr=log2.stdout)

        # make -f sandbox.makefile pretest
        if self.Context.makefile:
            self._make('pretest')

        # tox -e sandbox
        with open(TOX_OUT, 'w') as tox_out:
            with sdk2.helpers.ProcessLog(self, logger="tox") as log1:
                run_tox_error = subprocess.call(['tox', '-v', '-e', 'sandbox'], cwd=self.Context.workdir, stdout=tox_out,
                                                stderr=log1.stderr)

        # make -f sandbox.makefile posttest
        if self.Context.makefile:
            self._make('posttest')

        with open(TOX_OUT) as tox_out:
            if run_tox_error:
                self._github_status('failure', 'Fail!!')
                raise TaskFailure('\n{}'.format(tox_out.read()))
            else:
                self._github_status('success', 'Success')
                self.set_info(tox_out.read())
