import logging

import requests

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox import common

from sandbox.projects.Statbox import STATFACE_TASK_RUNNER_LXC


REGISTRY = 'registry.yandex.net'


LOGGER = logging.getLogger(__name__)


class StatfaceTaskRunner(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):

        # https://wiki.yandex-team.ru/sandbox/cookbook/#cores1multislot
        privileged = False
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

        # dns = ctm.DnsType.DNS64
        disk_space = 50 * 1024

    class Parameters(sdk2.Task.Parameters):

        Container = sdk2.parameters.Container(
            'LXC Container',
            resource_type=STATFACE_TASK_RUNNER_LXC,
            platform='linux_ubuntu_18.04_bionic',
            default=870270439)

        Component = sdk2.parameters.String('Qloud component address', required=True)
        Command = sdk2.parameters.String(
            'Command to execute in the container',
            required=True, multiline=True)

        OAuthVault = sdk2.parameters.String(
            'Qloud OAuth token (vault owner::name)',
            default='STATFACE::robot-statface-api-qloud')
        SecfilesKeyVault = sdk2.parameters.String(
            'Secfiles key (vault owner::name)',
            default='STATFACE::statface-api-secfiles-key')
        RegistryAuth = sdk2.parameters.String(
            'Registry username::vault_owner::valut_name',
            default='robot-statface-api::STATFACE::robot-statface-api-qloud')

    @staticmethod
    def _get_vault_data(param_value):
        owner, name = param_value.split('::', 1)
        return sdk2.Vault.data(owner, name)

    def on_execute(self):
        registry_username, registry_vault = self.Parameters.RegistryAuth.split('::', 1)
        registry_pass = self._get_vault_data(registry_vault)

        secfiles_key = self._get_vault_data(self.Parameters.SecfilesKeyVault)
        oauth_token = self._get_vault_data(self.Parameters.OAuthVault)

        with sdk2.helpers.ProcessLog(self, logger='docker.login') as logger:
            cmd = ('docker', 'login', '--username', registry_username, '--password-stdin', REGISTRY)
            proc = subprocess.Popen(
                cmd,
                stdin=subprocess.PIPE,
                stdout=logger.stdout,
                stderr=logger.stderr)
            # Pieces of `subprocess.check_output`
            proc.communicate(registry_pass)
            retcode = proc.poll()
            if retcode:
                raise subprocess.CalledProcessError(retcode, cmd)

        component_path = self.Parameters.Component
        component_info = self._get_component_info(component_path, oauth_token)

        call_env = dict(component_info['call_env'])
        image_name = component_info['image_name']

        call_env.update(
            SECFILES_KEY=secfiles_key,
            STATFACE_TASK_RUNNER_COMPONENT=component_path,
            CONTEXT_URL=common.utils.get_task_link(self.id),
        )
        # TODO: log env_state['comment'], image_name

        with sdk2.helpers.ProcessLog(self, logger='image_pull') as logger:
            subprocess.check_call(
                ('docker', 'pull', image_name),
                stdout=logger.stdout, stderr=logger.stderr)

        # run releaser with commands
        with sdk2.helpers.ProcessLog(self, logger='main') as logger:
            cmd_envs = tuple(  # `('-e', 'SECFILES_KEY', '-e', 'YENV_TYPE', ...)`
                arg
                for key in call_env
                for arg in ('-e', key))
            cmd = (
                'docker', 'run',
                # Reduce the complexity of this whole situation:
                '--privileged', '--network=host',
                # Do not remove for debuggability that should cost nothing.
                # '--rm',
            ) + cmd_envs + (
                image_name,
                'sh', '-c', self.Parameters.Command,
            )
            subprocess.check_call(
                cmd,
                env=call_env,
                stdout=logger.stdout, stderr=logger.stderr)

    @staticmethod
    def _get_component_info(component_path, oauth_token):
        urlmap = {
            'statbox.statface-api.beta.sf4celery': 'https://api.stat-beta.yandex-team.ru/_v4/version/',
            'statbox.statface-api.prod.sf4celery': 'https://api.stat.yandex-team.ru/_v4/version/',
            'statbox.statface-api.beta.sf3celery': 'https://api.stat-beta.yandex-team.ru/_v3/version/',  # TODO
            'statbox.statface-api.prod.sf3celery': 'https://api.stat.yandex-team.ru/_v3/version/',  # TODO
        }
        url = urlmap[component_path]
        resp = requests.get(url, headers={'Authorization': 'OAuth {}'.format(oauth_token)})
        resp.raise_for_status()
        resp_data = resp.json()
        LOGGER.info("Component info: %r -> %r -> %r", component_path, url, resp_data)
        return dict(
            image_name=resp_data['registry'],
            call_env=resp_data['env'],
            env_state=resp_data,
        )
