# -*- coding: utf-8 -*-
import logging
import os
import time
from sandbox.projects.common import file_utils as fu
import sandbox.common.types.misc as ctm
from sandbox import sdk2, common
from sandbox.sandboxsdk import environments
from sandbox.projects.common.vcs.arc import Arc
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types.client import Tag


ENV_VARS = {
    'tunneler_hosts': '["tun.si.yandex-team.ru"]',
    'tunneler_version': '2',
    'tunneler_cloud': 'yp',
    'tunneler_user': 'robot-frontend',
}


class StationuiBuildBeta(sdk2.Task):
    """ An empty task, which does nothing except of greets you. """

    beta_process = None

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.NodeJS('10.20.1'),
            environments.GCCEnvironment('5.3.0'),
        )

        # DNS64 is required to be able to connect to npm.yandex-team.ru
        dns = ctm.DnsType.DNS64

        cores = 4  # 4 cores or less
        ram = 4096  # 8GiB or less
        client_tags = Tag.LINUX_BIONIC

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 3600

        # custom parameters
        branch = sdk2.parameters.String(
            'Ветка с которой будет запущена бета',
            description='Примеры: trunk, users/<username>/branch',
            default='trunk', required=True)

        commit = sdk2.parameters.String(
            'Хэш коммита в arc',
            description='Пример: 4a9dbeabf69391c9ee3ecce89580e43ba315e525',
            default='')

        beta_template = sdk2.parameters.String("Шаблон беты. Дата центр, id хоста, порт",
                                               default="https://{0}-{1}-{2}-yp.tun.si.yandex.ru", required=True)
        additional_env_vars = sdk2.parameters.Dict('Переменные окружения', default=ENV_VARS)

        key_owner = sdk2.parameters.String("Владелец секрета в vault",
                                           default="FRONTEND", required=True)
        key_name = sdk2.parameters.String("Название секрета в vault",
                                          default="env.ROBOT_FRONTEND_SSH_KEY", required=True)

        with sdk2.parameters.Output():
            beta_address = sdk2.parameters.String("Адрес беты")
            result_commit = sdk2.parameters.String("Хэш коммита, на котором фактически отработала сборка")
            record_config = sdk2.parameters.String("Конфиг записи wpr", multiline=True)

    def on_execute(self):
        logging.debug('Initing environ')

        yandex_int_resources_path = str(self.path('.yandex-int'))
        os.environ['YANDEX_INT_RESOURCES_PATH'] = yandex_int_resources_path
        os.environ['SANDBOX_AUTH_TOKEN'] = str(self.agentr.token)

        for key, value in self.Parameters.additional_env_vars.iteritems():
            os.environ[key] = value

        logging.debug('Inited environ: {}'.format(os.environ))

        arc = Arc()
        with arc.mount_path(None, None, fetch_all=True) as arcadia_src_dir:
            self.Parameters.result_commit = self.checkout_and_get_result_commit(arc, arcadia_src_dir)
            logging.info("result_commit: %s", self.Parameters.result_commit)

            logging.info(arcadia_src_dir)
            station_ui_path = 'frontend/services/station-ui'
            station_ui_dir = os.path.join(arcadia_src_dir, station_ui_path)
            logging.info("Адрес до проекта station-ui")
            logging.info(station_ui_dir)

            self.read_bench_config_file(station_ui_dir)

            logger = logging.getLogger("StationuiBuildBetaNpm")

            with sdk2.ssh.Key(self, self.Parameters.key_owner, self.Parameters.key_name):
                with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
                    sp.check_call(
                        ['node', '--version'],
                        cwd=station_ui_dir,
                        stdout=pl.stdout,
                        stderr=pl.stderr,
                    )
                    sp.check_call(
                        ['npm', '--version'],
                        cwd=station_ui_dir,
                        stdout=pl.stdout,
                        stderr=pl.stderr,
                    )
                    sp.check_call(
                        ['npm', 'run'],
                        cwd=station_ui_dir,
                        stdout=pl.stdout,
                        stderr=pl.stderr,
                    )
                    sp.check_call(
                        ['npm', 'ci'],
                        cwd=station_ui_dir,
                        stdout=pl.stdout,
                        stderr=pl.stderr,
                    )
                    self.beta_process = sp.Popen(
                        ['npm', 'run', 'start:public:bench:play'],
                        cwd=station_ui_dir,
                        stdout=pl.stdout,
                        stderr=pl.stderr,
                    )

                    logging.info("Получение пути до беты")
                    tun_cache_path = os.path.join(yandex_int_resources_path, '.tun-cache.json')

                    while not self.check_beta_file(tun_cache_path):
                        time.sleep(5)

                    beta = self.read_beta_file(tun_cache_path)

                    logging.info("Запись beta_address")
                    self.Parameters.beta_address = beta

                    logging.info("Ожидание завершение процесса беты")
                    self.beta_process.wait()
                    logging.info("Ожидание завершено")

    def on_terminate(self):
        if self.beta_process:
            logging.info("Остановка процесса беты")
            self.beta_process.terminate()
            logging.info("Процесс беты завершён")

    def check_beta_file(self, beta_file_path):
        logging.info("Проверка файла: " + beta_file_path)

        return os.path.isfile(beta_file_path)

    def read_beta_file(self, beta_file_path):
        logging.info("Чтение файла: " + beta_file_path)
        content = fu.json_load(beta_file_path)

        logging.info("\n%s\n" % content)

        host_splinted = str(content.values()[0]['instanceAddress']['host']).split('.')

        host_id = host_splinted[0]
        dc = host_splinted[1]
        port = content.values()[0]['remotePort']

        beta = self.Parameters.beta_template.format(dc, host_id, port)

        logging.info("Считанный url беты: " + beta)

        return beta

    def read_bench_config_file(self, station_ui_dir):
        logging.info("Получение пути до беты")
        bench_config_file = os.path.join(station_ui_dir, '.config/bench.yaml')

        logging.info("Чтение файла: " + bench_config_file)
        config = fu.read_file(bench_config_file)

        logging.info("\n%s\n" % config)

        logging.info("Запись record_config")
        self.Parameters.record_config = config

    def checkout_and_get_result_commit(self, arc_client, arcadia_src_dir):
        logging.info("Переключение на ветку: %s", self.Parameters.branch)
        arc_client.checkout(
            arcadia_src_dir,
            branch=self.Parameters.branch,
        )
        if not self.Parameters.commit:
            return arc_client.log(
                arcadia_src_dir,
                start_commit="HEAD",
                max_count=1,
                as_dict=True
            )[0]['commit']
        else:
            # Проверка существования коммита
            merge_base = arc_client.get_merge_base(
                arcadia_src_dir,
                base_branch=self.Parameters.commit,
                branch=self.Parameters.branch,
            )
            if merge_base == self.Parameters.commit:
                logging.info("Переключение на коммит: %s", self.Parameters.commit)
                arc_client.checkout(
                    arcadia_src_dir,
                    branch=self.Parameters.commit,
                )
                return self.Parameters.commit
            else:
                raise common.errors.TaskFailure(
                    'Не удалось найти коммит {} в'
                    ' ветке {}'.format(
                        self.Parameters.commit,
                        self.Parameters.branch)
                )
