# -*- coding: utf-8 -*-
import logging
import sandbox.common.types.task as ctt

from sandbox.projects.StationUI.BuildBeta import StationuiBuildBeta
from sandbox.projects.StationUI.WriteWPR import StationuiWriteWPR
from sandbox.projects.StationUI.PlayWPR import StationuiPlayWPR
from sandbox.projects.browser.perf.record_wpr_archive import WprArchive
from sandbox import sdk2, common


class StationuiRunBenchmarks(sdk2.Task):
    """ Запуск бенчмарка производительности вёрстки на станции """

    class Requirements(sdk2.Task.Requirements):
        cores = 2  # 4 cores or less
        ram = 4096  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        run_beta = sdk2.parameters.Bool("Необходимо ли поднимать бету", default=True)
        with run_beta.value[True]:
            branch = sdk2.parameters.String(
                'Ветка с которой будет запущена бета',
                description='Примеры: trunk, users/<username>/branch',
                default='trunk', required=True)

            commit = sdk2.parameters.String(
                'Хэш коммита в arc',
                description='Пример: 4a9dbeabf69391c9ee3ecce89580e43ba315e525',
                default='')
            record_config_override = sdk2.parameters.String("Переопределение конфига записи wpr", multiline=True)
        with run_beta.value[False]:
            record_config = sdk2.parameters.String("Конфиг записи wpr", multiline=True, required=True)

        run_wpr_play = sdk2.parameters.Bool("Необходимо ли воспроизводить wpr", default=False)
        with run_wpr_play.value[True]:
            wpr_play_repeat = sdk2.parameters.Integer("Количество повторений воспроизведения", default=5)

        with sdk2.parameters.Output():
            result_commit = sdk2.parameters.String("Хэш коммита, на котором фактически отработала сборка")
            wpr_archive = sdk2.parameters.Resource("wpr_archive", required=True)

    def on_execute(self):
        if self.Parameters.run_beta:
            with self.memoize_stage.run_build_beta:
                logging.info("Запуск бенчмарков на ветке: %s", self.Parameters.branch)

                logging.info("Запуск задачи поднятия беты")
                task_build_beta = StationuiBuildBeta(self,
                                                     branch=self.Parameters.branch,
                                                     commit=self.Parameters.commit,
                                                     )
                task_build_beta.enqueue()
                self.Context.build_beta_task_id = task_build_beta.id

                logging.info("Ожидание поднятия беты")
                targets = {task_build_beta.id: 'beta_address'}
                raise sdk2.WaitOutput(targets, wait_all=True, timeout=900)

        with self.memoize_stage.run_write_wpr:
            if self.Parameters.run_beta:
                logging.info("Проверка статуса задачи поднятия беты")
                task_build_beta = self.find(id=self.Context.build_beta_task_id).first()
                self._check_task_status(task_build_beta, ctt.Status.EXECUTING)

                self.Parameters.result_commit = task_build_beta.Parameters.result_commit

                record_config = self.Parameters.record_config_override
                if not record_config:
                    logging.info("Берём record_config из задачи поднятия беты")
                    record_config = task_build_beta.Parameters.record_config

                beta_address = task_build_beta.Parameters.beta_address
                logging.info("Дождались бету: %s", beta_address)
                record_config = record_config.replace('https://yandex.ru', beta_address)
            else:
                record_config = self.Parameters.record_config

            logging.info("Запуск задачи записи wpr")
            task_write_wpr = StationuiWriteWPR(self,
                                               record_config=record_config,
                                               snapshot_id=self.Parameters.result_commit,
                                               )
            task_write_wpr.enqueue()
            self.Context.write_wpr_task_id = task_write_wpr.id

            logging.info("Ожидание завершения задачи записи wpr")
            raise sdk2.WaitTask(task_write_wpr, ctt.Status.Group.FINISH)

        with self.memoize_stage.wait_wpr_and_play_wpr:
            if self.Parameters.run_beta:
                logging.info("Завершение задачи поднятия беты")
                task_build_beta = self.find(id=self.Context.build_beta_task_id).first()
                task_build_beta.stop()

            logging.info("Проверка статуса задачи записи wpr")
            task_write_wpr = self.find(id=self.Context.write_wpr_task_id).first()
            self._check_task_status(task_write_wpr, ctt.Status.SUCCESS)
            self.Parameters.wpr_archive = self._get_task_resource(task_write_wpr, WprArchive)

            if self.Parameters.run_wpr_play:
                logging.info("Запуск задачи воспроизведения wpr")
                task_play_wpr = StationuiPlayWPR(self,
                                                 spec_id=task_write_wpr.Parameters.build_id,
                                                 repeat=self.Parameters.wpr_play_repeat,
                                                 )
                task_play_wpr.enqueue()
                self.Context.play_wpr_task_id = task_play_wpr.id

                logging.info("Ожидание завершения задачи воспроизведения wpr")
                raise sdk2.WaitTask(task_play_wpr, ctt.Status.Group.FINISH)

    def _check_task_status(self, task, status):
        if task.status != status:
            raise common.errors.TaskFailure(
                'Задача {} ({}) должна находится в статусе {}, но находится в статусе {}'.format(
                    task.type, task.id, status, task.status
                )
            )

        logging.info(
            'Задача {} ({}) находится в правильном статусе {}'.format(
                task.type, task.id, task.status
            )
        )

    def _get_task_resource(self, task, resource_type):
        logging.info("Получение ресурса {} из задачи {}".format(resource_type, task.id))
        return sdk2.Resource.find(task=task, type=resource_type).first()
