# -*- coding: utf-8 -*-
import logging

from sandbox import common
from sandbox.common.types.task import Status
from sandbox import sdk2
from sandbox.projects.common import decorators
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.browser.util.BrowserWaitPerfBuilds import BrowserWaitPerfBuilds


class StationuiWPR(sdk2.Task):
    """ wpr на станции """

    BROWSER_TEAMCITY = 'teamcity.browser.yandex-team.ru'
    STATION_POOL = 'browser zombie android-yastation'

    SUBTASK_STATUSES_TO_WAIT = list(
        Status.Group.FINISH + Status.Group.BREAK +
        Status.Group.FAIL_ON_ANY_ERROR +
        Status.Group.SCHEDULER_FAILURE
    )

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('teamcity-client', '4.8.2'),
        ]

        cores = 2  # 4 cores or less
        ram = 4096  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    @sdk2.header()
    def on_execute(self):
        with self.memoize_stage.create_children:
            self.log_start()

            build = self.run_build()

            self.Context.build_id = build.id
            self.Context.build_url = build.web_url

            raise self.wait_build(build.id)

        with self.memoize_stage.finish:
            build_id = self.Context.build_id
            self.check_build_status(build_id)

            logging.info("Запись build_id в output")
            self.Parameters.build_id = build_id

            resource = self.get_resource(build_id)
            if resource:
                sdk2.ResourceData(resource).ready()

        return {
            "<h3>Билд wpr</h3>": [
                {
                    "Build id (spec)": str(self.Context.build_id),
                    "URL": "<a href='{0}'>{0}</a>".format(self.Context.build_url),
                }
            ]
        }

    @property
    @common.utils.singleton
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=self.BROWSER_TEAMCITY,
            auth=sdk2.Vault.data(self.Parameters.browser_teamcity_vault))

    def run_build(self):
        logging.info("Запуск билда в тимсити")

        bt = self.teamcity_client.build_types[self.MEASURE_BUILD_TYPE_ID]

        build = bt.run_build(
            pool=self.teamcity_client.pools[self.STATION_POOL],
            properties=self.get_properties(),
        )

        logging.info('build id: {}'.format(build.id))
        logging.info('build url: {}'.format(build.web_url))

        return build

    def wait_build(self, build_id):
        logging.info('Ожидание завершение билда')

        builds = 'tc://{}/{}'.format(self.BROWSER_TEAMCITY, build_id)
        logging.info(builds)

        measure_wait_task = BrowserWaitPerfBuilds(self, builds=builds, sleep_time=1)

        raise sdk2.WaitTask(measure_wait_task.enqueue(),
                            self.SUBTASK_STATUSES_TO_WAIT)

    def check_build_status(self, build_id):
        build = self.teamcity_client.builds[build_id]

        logging.info(
            'Билд {} находится в статусе {} - {}'.format(
                build.id, build.state, build.status
            )
        )

        if build.status != 'SUCCESS':
            raise common.errors.TaskFailure('Билд завершён с ошибкой')

    def log_start(self):
        logging.info("Запись wpr")
        logging.info("record_config\n" + str(self.Parameters.record_config))

    def get_properties(self):
        return {
            'product': self.MEASURE_PRODUCT,
            'record_config': self.Parameters.record_config
        }

    @decorators.retries(5, delay=1, backoff=1, max_delay=1, exceptions=(ValueError,))
    def download_results(self, build_id, path, dst):
        logging.info("Загрузка результатов из Teamcity ({}): {} в {}"
                     .format(build_id, path, dst))
        build = self.teamcity_client.builds[build_id]
        build.download_artifact(path=path, dst=dst)

    def get_resource(self, build_id):
        return None
