# coding: utf-8

import errno
import json
import logging
import os
import os.path
import yaml

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.common.utils import get_task_link
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.resource_types import TASK_CUSTOM_LOGS
from sandbox.projects.sandbox import LXC_CONTAINER
from sandbox.projects.sandbox_ci.utils.github import GitHubApi, GitHubStatus
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import Resource
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.juggler import jclient


def mkdir_p(path):
    try:
        os.makedirs(path)
    except (OSError, IOError) as e:
        if e.errno != errno.EEXIST:
            raise


def rm_f(path):
    try:
        os.remove(path)
    except (OSError, IOError) as e:
        if e.errno != errno.ENOENT:
            raise


class RichTextTaskFailure(TaskFailure):
    def __init__(self, message, rich_addition):
        super(RichTextTaskFailure, self).__init__(message)
        self.rich_addition = rich_addition

    def get_task_info(self):
        return self.rich_addition


class RunNginxStrmValidator(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Apply Strm's salt config and test nginx configuration
    """
    name_for_humans = 'Apply salt config and test nginx configuration'

    class Requirements(sdk2.Task.Requirements):
        privileged = True
        client_tags = (ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL | ctc.Tag.Group.LINUX)
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('pyyaml', use_wheel=True),
            PipEnvironment('yodax', use_wheel=True),
        ]
        disk_space = 3 * 1024

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 1
        # common parameters
        kill_timeout = 60 * 15
        # custom parameters
        suspend_on_fail = sdk2.parameters.Bool('Suspend task on fail', default=False)
        make_logs_tarball = sdk2.parameters.Bool('Create resource with /var/log tarball', default=False)
        make_config_tarball = sdk2.parameters.Bool('Create resource with /etc/nginx tarball', default=False)
        with make_config_tarball.value[True]:
            # FIXME: wtf? why this list exists at all?
            strm_resources = [
                'STRM_STREAM_DEFAULT_CONFIGS',
                'STRM_PLAYLIST_CACHE_CONFIGS',
                'STRM_STREAM_INTERNAL_EDGE_CONFIGS',
                'STRM_STREAM_COLD_EDGE_CONFIGS',
            ]
            resource_type_name = sdk2.parameters.String(
                'Type of generated resource',
                choices=[(resource, resource) for resource in strm_resources],
                default='',
                required=False
            )
            with sdk2.parameters.String('Auto release to Nanny') as auto_release:
                auto_release.values[""] = "-"
                for status in [ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.UNSTABLE, ctt.ReleaseStatus.PRESTABLE, ctt.ReleaseStatus.STABLE]:
                    auto_release.values[status] = status

        debug = sdk2.parameters.Bool("Debug logging")

        github_review = sdk2.parameters.Bool('Report status as github review', default=True)
        with github_review.value[True]:
            github_owner = sdk2.parameters.String('github owner (DRONE_REPO_NAMESPACE)')
            github_repo = sdk2.parameters.String('github repo name (DRONE_REPO_NAME)')
            github_sha = sdk2.parameters.String('github commit sha (DRONE_COMMIT_SHA)')

        salt_config_repo = sdk2.parameters.String(
            'Git url for salt config', default='https://github.yandex-team.ru/strm/strm-salt-configs.git'
        )
        salt_minion_id = sdk2.parameters.String('Salt minion id', required=False)
        # To get grains do salt-call --local grains.get conductor --out yaml
        salt_grains = sdk2.parameters.String('Salt minion grains', multiline=True)
        salt_grains_file = sdk2.parameters.String('Path to grains file')
        test_nginx_config = sdk2.parameters.Bool("Test nginx config", default=True)
        test_salt_base_diff = sdk2.parameters.Bool("Test salt base diff", default=True)
        test_yodax_config = sdk2.parameters.Bool("[Sec]analyze nginx config", default=True)
        with test_yodax_config.value[True]:
            yodax_arguments = sdk2.parameters.String('yodax arguments', default="--skips http_splitting")

        custom_salt_common_repo = sdk2.parameters.Bool("Use custom salt common repo")
        with custom_salt_common_repo.value[True]:
            salt_config_source_branch = sdk2.parameters.String('git source branch', default='stable', required=True)
            salt_config_base_branch = sdk2.parameters.String('git base branch', default='prestable', required=True)
            salt_common_repo = sdk2.parameters.String(
                'URL for salt common ', default='https://github.yandex-team.ru/salt-media/common.git'
            )
            salt_common_source_branch = sdk2.parameters.String('git branch', default='master', required=True)

        # LXC_CONTAINER  https://sandbox.yandex-team.ru/resource/812814102/view
        _container = sdk2.parameters.Container(
            "Container",
            required=True,
            resource_type=LXC_CONTAINER
        )

    def get_salt_minion_id(self):
        return self.Parameters.salt_minion_id or 'run-nginx-strm-validator.strm.yandex.net'

    def set_github_status(self, status, description):
        if self.Parameters.github_sha:
            token = sdk2.Vault.data(self.owner, 'github_token')
            github = GitHubApi(token=token)
            if description:
                description = description[:140]
            github.create_status(
                owner='strm',
                repo='strm-salt-configs',
                context='ci/sandbox [{}]'.format(self.Parameters.resource_type_name),
                description=description,
                state=status,
                url=get_task_link(self.id),
                sha=self.Parameters.github_sha,
            )

    def run_cmd(self, cmd):
        logging.info("run cmd {}".format(cmd))
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
            except sp.CalledProcessError:
                out = ''
            logging.info(out)

    def bootstrap_external_pillars(self, file_roots):
        logger = 'pillar-fetcher'
        mkdir_p('/tmp/emptydir')
        open('/tmp/emptydir/minion', 'w').close()
        yav_token = sdk2.Vault.data(self.owner, 'yav_oauth_token')
        cms_secret_token = sdk2.Vault.data(self.owner, 'cms_secret_token')
        with sdk2.helpers.ProcessLog(self, logger) as pl:
            cmd = [
                'salt-call',
                '--config-dir', '/tmp/emptydir',
                '--retcode-passthrough',
                '--output-diff',
                '--log-level', 'debug' if self.Parameters.debug else 'critical',
                '--id', self.get_salt_minion_id(),
                '--local',
                '--file-root', file_roots,
                '--pillar-root', '/tmp/emptydir',
                'state.apply', 'units.ext-pillar-fetcher',
                'pillar={}'.format(json.dumps({
                    "pillar-fetcher-cms-secret": cms_secret_token,
                    "pillar-fetcher-yav-token": yav_token,
                }))
            ]
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)

    def make_salt_cmd(self, saltenv):
        # logger = 'salt-{}'.format(saltenv)
        # with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
        return [
            'salt-call',
            '--retcode-passthrough',
            '--output-diff',
            '--log-level', 'debug' if self.Parameters.debug else 'critical',
            '--id', self.get_salt_minion_id(),
            '--local',
            'state.apply',
            'saltenv={}'.format(saltenv),
            'pillarenv={}'.format(saltenv),
        ]

    def test_nginx(self):
        return self.run_test(['nginx', '-t'])

    def test_yodax(self):
        cmd = ['yodax']
        if self.Parameters.debug:
            cmd.append('--debug')
        if self.Parameters.yodax_arguments:
            cmd.extend(self.Parameters.yodax_arguments.split())
        return self.run_test(cmd)

    def test_salt_base(self):
        cmd = self.make_salt_cmd('salt_config_base')
        return self.run_test(
            cmd,
            'salt-base',
            'salt branch {}'.format(self.Parameters.salt_config_base_branch)
        )

    def test_salt_source(self):
        cmd = self.make_salt_cmd('salt_config_source')
        return self.run_test(
            cmd,
            'salt-source',
            'salt commit {}'.format(self.Parameters.github_sha)
        )

    def run_test(self, cmd, logger=None, description=None, raise_on_fail=False):
        logger = logger or os.path.basename(cmd[0])
        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            try:
                sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)
            except Exception as e:
                msg = '{} failed: {}'.format(
                    description or ' '.join(cmd),
                    e,
                )
                logging.exception(msg)
                log_path = '{}.out.log'.format(logger)
                link = os.path.join(self.log_resource.http_proxy, log_path)
                self.set_info("{}<br/>See log <b><a href='{}'>{}</a></b>".format(msg, link, log_path), do_escape=False)
                if raise_on_fail:
                    raise
                else:
                    return e

    def gen_salt_minion(self):
        rm_f('/etc/salt/minion_id')
        with open('/etc/salt/minion', "w+") as f:
            # FIXME: wtf?
            f.write("pillarenv: https://github.com/saltstack/salt/issues/52628\n")
            f.write("saltenv: https://github.com/saltstack/salt/issues/52628\n")
            f.write("file_roots:\n")
            f.write("  salt_config_base:\n")
            f.write("    - salt_config_base/roots\n")
            f.write("    - salt_common_repo/roots\n")
            f.write("  salt_config_source:\n")
            f.write("    - salt_config_source/roots\n")
            f.write("    - salt_common_repo/roots\n")
            f.write("pillar_roots:\n")
            f.write("  salt_config_base:\n")
            f.write("    - salt_config_base/pillar\n")
            f.write("    - /srv/salt/pillar_static_roots/stable/pillar\n")
            f.write("  salt_config_source:\n")
            f.write("    - salt_config_source/pillar\n")
            f.write("    - /srv/salt/pillar_static_roots/stable/pillar\n")

    def load_yaml(self, data):
        # Validate yaml salt grains
        try:
            return yaml.load(data)
        except yaml.YAMLError:
            raise TaskFailure('Failed load yaml salt grains')

    def gen_grains(self):
        grains_path = "/etc/salt/grains"
        grains = {}

        if self.is_auto_release():
            grains.update({'nanny_resource_type': [self.Parameters.resource_type_name.lower()]})

        if self.Parameters.salt_grains_file:
            path = os.path.join('salt_config_source', self.Parameters.salt_grains_file)
            with open(path) as f:
                grains.update(self.load_yaml(f))

        if self.Parameters.salt_grains:
            grains.update(self.load_yaml(self.Parameters.salt_grains))

        with open(grains_path, "w") as f:
            f.write(yaml.dump(grains, default_flow_style=False))

    def prepare_tests(self):
        self.run_cmd('hostname {}'.format(self.get_salt_minion_id()))
        self.run_cmd('service rsyslog start')
        mkdir_p('/run/www')
        self.bootstrap_external_pillars(file_roots='salt_config_base/roots')
        self.gen_salt_minion()
        self.gen_grains()
        self.run_cmd('cat /etc/salt/minion')

    def run_tests(self):
        tests = [self.test_salt_source]

        if self.Parameters.test_nginx_config:
            tests.append(self.test_nginx)
        if self.Parameters.test_yodax_config:
            tests.append(self.test_yodax)

        errors = []
        for test in tests:
            error = test()
            if error:
                errors.append(error)

        if errors:
            raise TaskFailure('{} tests failed'.format(len(errors)))

    def notify_juggler(self, msg, status):
        if self.is_auto_release():
            try:
                jclient.send_events_to_juggler(
                    '{}_{}'.format(
                        self.Parameters.resource_type_name.lower(),
                        self.Parameters.auto_release
                    ),
                    jclient.to_juggler_service_name(self.type.name.lower()),
                    status,
                    msg
                )
            except Exception:
                self.set_info('Failed to send event to juggler')
                logging.error('Failed to send event to juggler', exc_info=True)

    def on_prepare(self):
        # Git clone salt config base repo
        if self.Parameters.github_review:
            self.set_github_status(GitHubStatus.PENDING, 'Task started')

        salt_config_base_git = Git(self.Parameters.salt_config_repo, filter_branches=False)
        salt_config_base_git.clone('salt_config_base', self.Parameters.salt_config_base_branch)
        if self.Parameters.github_sha:
            salt_config_base_git.clone('salt_config_source', commit=self.Parameters.github_sha)
        else:
            salt_config_base_git.clone('salt_config_source', self.Parameters.salt_config_source_branch)

        salt_common_git = Git(self.Parameters.salt_common_repo)
        salt_common_git.clone('salt_common_repo', self.Parameters.salt_common_source_branch)

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        if self.is_auto_release():
            self.on_release(dict(release_status=self.Parameters.auto_release))

    def on_release(self, additional_parameters):
        resources = []

        for resource in sdk2.Resource.find(task=self, state=ctr.State.READY).limit(0):
            if not resource.type.releasable:
                continue
            resources.append(resource)

        release_subject = ', '.join(['{}'.format(r.type.name) for r in resources])
        additional_parameters['release_subject'] = release_subject
        logging.info(str(additional_parameters))
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)

    def on_finish(self, prev_status, status):
        if status == ctt.Status.SUCCESS:
            self.notify_juggler(status, status='OK')
        else:
            self.notify_juggler('{} https://sandbox.yandex-team.ru/task/{}/view'.format(status, self.id), status='WARN')

    def create_nginx_configs_resource(self, resource_type_name):
        self.run_cmd('tar -zcvf nginx_conf.tar.gz /etc/nginx')
        result = sdk2.ResourceData(Resource[resource_type_name](
            self, 'nginx configs', 'nginx_conf.tar.gz')
        )
        result.ready()

    def is_auto_release(self):
        return self.Parameters.auto_release and self.Parameters.make_config_tarball and self.Parameters.resource_type_name

    def on_execute(self):
        try:
            self.prepare_tests()
            if self.Parameters.make_config_tarball and self.Parameters.resource_type_name and \
                    self.Parameters.resource_type_name not in sdk2.Resource:
                raise TaskFailure('Invalid value for \"{}\" input parameter. Unknown resource type: {}'.format(
                    self.Parameters.resource_type_name.description,
                    self.Parameters.resource_type_name
                ))

            self.run_tests()
            if self.Parameters.github_review:
                self.set_github_status(GitHubStatus.SUCCESS, 'Ok')
            if self.Parameters.make_config_tarball:
                self.create_nginx_configs_resource(self.Parameters.resource_type_name)
            if self.Parameters.test_salt_base_diff:
                self.test_salt_base()

        except Exception as e:
            if self.Parameters.github_review:
                self.set_github_status(GitHubStatus.FAILURE, str(e))
            if self.Parameters.make_logs_tarball:
                self.run_cmd('tar --exclude /var/log/sandbox -zcvf var-log.tar.gz /var/log')
                result = sdk2.ResourceData(TASK_CUSTOM_LOGS(self, 'var logs', 'var-log.tar.gz'))
                result.ready()
            if self.Parameters.make_config_tarball:
                self.create_nginx_configs_resource('OTHER_RESOURCE')
            if self.Parameters.suspend_on_fail:
                self.suspend()
            raise
