# coding: utf-8

import errno
import json
import logging
import os
import os.path
import tarfile

import yaml

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task
from sandbox.projects.resource_types import TASK_CUSTOM_LOGS
from sandbox.sdk2.helpers import subprocess as sp


SALTENV = 'salt-env-for-task'


def mkdir_p(path):
    try:
        os.makedirs(path)
    except (OSError, IOError) as e:
        if e.errno != errno.EEXIST:
            raise


def rm_f(path):
    try:
        os.remove(path)
    except (OSError, IOError) as e:
        if e.errno != errno.ENOENT:
            raise


class RichTextTaskFailure(TaskFailure):
    def __init__(self, message, rich_addition):
        super(RichTextTaskFailure, self).__init__(message)
        self.rich_addition = rich_addition

    def get_task_info(self):
        return self.rich_addition


class RunNginxStrmValidatorFromPackage(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Apply Strm's salt config and test nginx configuration
    """

    name_for_humans = 'Apply salt config and test nginx configuration'

    class Requirements(sdk2.Task.Requirements):
        container_resource = 2959964809  # TODO: Make own resource type for container and get latest in task
        privileged = True
        client_tags = ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL | ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64
        disk_space = 3 * 1024

    class Parameters(sdk2.Task.Parameters):
        _ = binary_task.LastBinaryReleaseParameters()
        max_restarts = 1
        # common parameters
        kill_timeout = 60 * 15
        # custom parameters
        suspend_on_fail = sdk2.parameters.Bool('Suspend task on fail', default=False)
        make_logs_tarball = sdk2.parameters.Bool('Create resource with /var/log tarball', default=False)
        make_config_tarball = sdk2.parameters.Bool('Create resource with /etc/nginx tarball', default=False)
        with make_config_tarball.value[True]:
            # FIXME: wtf? why this list exists at all?
            strm_resources = [
                'STRM_STREAM_DEFAULT_CONFIGS',
                'STRM_PLAYLIST_CACHE_CONFIGS',
                'STRM_STREAM_INTERNAL_EDGE_CONFIGS',
                'STRM_STREAM_COLD_EDGE_CONFIGS',
                'OTHER_RESOURCE',
            ]
            resource_type_name = sdk2.parameters.String(
                'Type of generated resource',
                choices=[(resource, resource) for resource in strm_resources],
                default='OTHER_RESOURCE',
                required=False,
            )

        debug = sdk2.parameters.Bool("Debug logging")

        # FIXME: soo coupled with strm
        external_pillars_path = sdk2.parameters.String(
            'Path to bootsrap pillars',
            default='/srv/salt/pillar_static_roots/stable/pillar',
        )
        do_bootstrap_pillars = sdk2.parameters.Bool('Bootstrap pillars', default=True)
        # with do_bootstrap_pillars.value[True]:
        #     cms_secret_token = sdk2.parameters.YavSecret('cms secret', default="sec-01d7q8ts2hx4gtjzk2jtpawpzr#trns-mgr-cms-api-drm-keys")
        #     yav_token = sdk2.parameters.YavSecret('yav token', default="sec-01cw6ce3rd7mahtbb89ygxmtf9#oauth_yav")

        salt_states_resource = sdk2.parameters.Resource('Resource with salt states', required=True)
        salt_minion_id = sdk2.parameters.String('Salt minion id', required=False)
        # To get grains do salt-call --local grains.get conductor --out yaml
        salt_grains = sdk2.parameters.String('Salt minion grains', multiline=True)
        salt_grains_file = sdk2.parameters.String('Path to grains file')
        salt_base_dir = sdk2.parameters.String('Salt base dir', default='/salt/srv')
        salt_file_roots_paths = sdk2.parameters.List('Paths to roots in given package', required=True)
        salt_pillar_roots_paths = sdk2.parameters.List('Paths to pillars in given package', required=True)
        test_nginx_config = sdk2.parameters.Bool("Test nginx config", default=True)
        test_yodax_config = sdk2.parameters.Bool("[Sec]analyze nginx config", default=True)
        with test_yodax_config.value[True]:
            yodax_resource = sdk2.parameters.Resource('Resource with yodax binary', default=2939057630)
            yodax_arguments = sdk2.parameters.String('yodax arguments', default="--skips http_splitting")

    def get_salt_minion_id(self):
        return self.Parameters.salt_minion_id or 'run-nginx-strm-validator.strm.yandex.net'

    def run_cmd(self, cmd):
        logging.info("run cmd {}".format(cmd))
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
            except sp.CalledProcessError:
                out = ''
            logging.info(out)

    # strm specific
    def bootstrap_external_pillars(self):
        logger = 'pillar-fetcher'
        mkdir_p('/tmp/emptydir')
        open('/tmp/emptydir/minion', 'w').close()
        # yav_token = self.Parameters.yav_token.data()[self.Parameters.yav_token.default_key]
        # cms_secret_token = self.Parameters.cms_secret_token.data()[self.Parameters.cms_secret_token.default_key]
        yav_token = sdk2.Vault.data(self.owner, 'yav_oauth_token')
        cms_secret_token = sdk2.Vault.data(self.owner, 'cms_secret_token')
        with sdk2.helpers.ProcessLog(self, logger) as pl:
            cmd = [
                'salt-call',
                '--retcode-passthrough',
                '--output-diff',
                '--log-level',
                'debug' if self.Parameters.debug else 'critical',
                '--id',
                self.get_salt_minion_id(),
                '--local',
                '--pillar-root',
                '/tmp/emptydir',
                'state.apply',
                'units.ext-pillar-fetcher',
                'saltenv={}'.format(SALTENV),
                'pillar={}'.format(
                    json.dumps(
                        {
                            "pillar-fetcher-cms-secret": cms_secret_token,
                            "pillar-fetcher-yav-token": yav_token,
                        }
                    )
                ),
            ]
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)

    def make_salt_cmd(self, saltenv):
        return [
            'salt-call',
            '--retcode-passthrough',
            '--output-diff',
            '--log-level',
            'debug' if self.Parameters.debug else 'critical',
            '--id',
            self.get_salt_minion_id(),
            '--local',
            'state.apply',
            'saltenv={}'.format(saltenv),
            'pillarenv={}'.format(saltenv),
        ]

    def test_nginx(self):
        return self.run_test(['nginx', '-t'])

    def test_yodax(self):
        yodax_binary = sdk2.ResourceData(self.Parameters.yodax_resource)
        cmd = [str(yodax_binary.path)]
        if self.Parameters.debug:
            cmd.append('--debug')
        if self.Parameters.yodax_arguments:
            cmd.extend(self.Parameters.yodax_arguments.split())
        return self.run_test(cmd)

    def test_salt(self):
        cmd = self.make_salt_cmd(SALTENV)
        return self.run_test(
            cmd,
            SALTENV,
            'salt test',
        )

    def run_test(self, cmd, logger=None, description=None, raise_on_fail=False):
        logger = logger or os.path.basename(cmd[0])
        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            try:
                sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)
            except Exception as e:
                msg = '{} failed: {}'.format(
                    description or ' '.join(cmd),
                    e,
                )
                logging.exception(msg)
                log_path = '{}.out.log'.format(logger)
                link = os.path.join(self.log_resource.http_proxy, log_path)
                self.set_info("{}<br/>See log <b><a href='{}'>{}</a></b>".format(msg, link, log_path), do_escape=False)
                if raise_on_fail:
                    raise
                else:
                    return e

    def gen_salt_minion(self):
        rm_f('/etc/salt/minion_id')
        roots = [os.path.join(self.Parameters.salt_base_dir, p) for p in self.Parameters.salt_file_roots_paths]
        pillars = [os.path.join(self.Parameters.salt_base_dir, p) for p in self.Parameters.salt_pillar_roots_paths]
        pillars.append(self.Parameters.external_pillars_path)

        config = {
            'file_client': 'local',
            'pillarenv': 'https://github.com/saltstack/salt/issues/52628',
            'saltenv': 'https://github.com/saltstack/salt/issues/52628',
            'file_roots': {
                SALTENV: roots,
            },
            'pillar_roots': {
                SALTENV: pillars,
            },
            'arcadia_builtin_grains_blacklist': [
                'conductor',
                'cauth',
            ],
        }
        self.save_yaml(config, '/etc/salt/minion')

    def save_yaml(self, data, path):
        with open(path, 'w+') as f:
            yaml.safe_dump(data, f)

    def load_yaml(self, data):
        # Validate yaml salt grains
        try:
            return yaml.safe_load(data)
        except yaml.YAMLError:
            raise TaskFailure('Failed load yaml salt grains')

    def gen_grains(self):
        grains_path = "/etc/salt/grains"
        grains = {'nanny_resource_type': self.Parameters.resource_type_name.lower()}

        if self.Parameters.salt_grains_file:
            path = os.path.join(self.Parameters.salt_base_dir, self.Parameters.salt_grains_file)
            with open(path) as f:
                grains.update(self.load_yaml(f))

        if self.Parameters.salt_grains:
            grains.update(self.load_yaml(self.Parameters.salt_grains))

        self.save_yaml(grains, grains_path)

    def prepare_tests(self):
        self.run_cmd('hostname {}'.format(self.get_salt_minion_id()))
        self.run_cmd('service rsyslog start')
        mkdir_p('/run/www')
        self.gen_salt_minion()
        self.gen_grains()
        if self.Parameters.do_bootstrap_pillars:
            self.bootstrap_external_pillars()
        self.run_cmd('cat /etc/salt/minion')

    def run_tests(self):
        tests = [self.test_salt]

        if self.Parameters.test_nginx_config:
            tests.append(self.test_nginx)
        if self.Parameters.test_yodax_config:
            tests.append(self.test_yodax)

        errors = []
        for test in tests:
            error = test()
            if error:
                errors.append(error)

        if errors:
            raise TaskFailure('{} tests failed'.format(len(errors)))

    def create_nginx_configs_resource(self, resource_type_name):
        self.run_cmd('tar -zcvf nginx_conf.tar.gz /etc/nginx')
        result = sdk2.ResourceData(sdk2.Resource[resource_type_name](self, 'nginx configs', 'nginx_conf.tar.gz'))
        result.ready()

    def unpack_resource(self, resource, dst):
        # todo: check if its actually tar.gz
        resource_data = sdk2.ResourceData(resource)
        with tarfile.open(str(resource_data.path), mode='r:gz') as tar:
            tar.extractall(dst)

    def on_execute(self):
        try:
            self.unpack_resource(self.Parameters.salt_states_resource, self.Parameters.salt_base_dir)
            self.prepare_tests()

            self.run_tests()
            if self.Parameters.make_config_tarball:
                self.create_nginx_configs_resource(self.Parameters.resource_type_name)

        except Exception as e:
            logging.exception('Something broken: %s', vars(e))
            if self.Parameters.make_logs_tarball:
                self.run_cmd('tar --exclude /var/log/sandbox -zcvf var-log.tar.gz /var/log')
                result = sdk2.ResourceData(TASK_CUSTOM_LOGS(self, 'var logs', 'var-log.tar.gz'))
                result.ready()
            if self.Parameters.make_config_tarball:
                self.create_nginx_configs_resource('OTHER_RESOURCE')
            if self.Parameters.suspend_on_fail:
                self.suspend()
            raise
