# coding=utf-8

import logging
import urllib
from collections import namedtuple
from itertools import chain
from typing import Dict, Iterable, List, Optional, Tuple, Union

import requests

from sandbox import sdk2
from sandbox.projects.Strm.StrmBuildClickhouseDictionary.common import StrmBuildClickhouseDictionaryBase

logger = logging.getLogger()

ABCService = namedtuple("ABCService", ("id", "slug", "name_ru", "name_en"))

ABC_SERVICES_FIELDS = ("id", "name", "slug")

SPECIAL_SERVICES = (
    ABCService(-1, "empty", "Неразмеченный траффик", "Untagged traffic"),
    ABCService(-2, "other", "Прочее", "Other"),
)


def _get_services_page(session, url):
    # type: (requests.Session, str) -> Tuple[Optional[str], List[Dict[str, Union[str, int, Dict[str, str]]]]]
    req = session.get(url)
    req.raise_for_status()

    abc_data = req.json()

    return abc_data.get("next"), abc_data.get("results", [])


def get_services(session, url):
    # type: (requests.Session, str) -> Iterable[ABCService]
    page, total_services = 0, 0
    next_url = url
    while next_url:
        page += 1
        logger.info("Getting page #{}".format(page))

        next_url, services = _get_services_page(session, next_url)
        total_services += len(services)

        for service in services:
            yield ABCService(service["id"], service["slug"], service["name"]["ru"], service["name"]["en"])

    logger.info("Total {} services".format(total_services))


class StrmBuildClickhouseDictionaryABCServices(StrmBuildClickhouseDictionaryBase):
    """
    Build Clickhouse dictionary from ABC API / Services and upload it to S3
    """

    class Parameters(StrmBuildClickhouseDictionaryBase.Parameters):
        dictionary_name = StrmBuildClickhouseDictionaryBase.Parameters.dictionary_name(default="abc_services")
        output_name = StrmBuildClickhouseDictionaryBase.Parameters.output_name(default="abc_services.csv")
        source_url = StrmBuildClickhouseDictionaryBase.Parameters.source_url(
            default="https://abc-back.yandex-team.ru/api/v4/services/"
        )

        check_last_modified = None

        with sdk2.parameters.Group("Source request OAuth token"):
            source_oauth_token_secret = sdk2.parameters.YavSecret("OAuth token secret", required=False)
            source_oauth_token_key = sdk2.parameters.String("Key of secret with OAuth token", required=False)

    def get_lines_from_source(self, source_url, if_modified_since):
        session = requests.session()

        if self.Parameters.source_oauth_token_secret:
            oauth_token_secret = self.Parameters.source_oauth_token_secret.data()
            oauth_token = oauth_token_secret[self.Parameters.source_oauth_token_key]
            session.headers["Authorization"] = "OAuth {}".format(oauth_token)

        initial_url_query = urllib.urlencode({"fields": ",".join(ABC_SERVICES_FIELDS)})
        initial_url = "{}?{}".format(source_url, initial_url_query)
        services = get_services(session, initial_url)

        for service in chain(SPECIAL_SERVICES, services):
            yield service

    @staticmethod
    def parse_lines(data_file):
        pass
