import datetime
import logging
import time

from sandbox import sdk2
import sandbox.projects.common.binary_task as binary_task


class StrmVodTop(sdk2.Resource):
    releasable = False
    auto_backup = True
    creation_ts = sdk2.Attributes.Integer('creation ts')


class StrmBuildVodTop(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Prepare resource with STRM VOD top using common VOD URI schemes
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        ext_params = binary_task.LastBinaryReleaseParameters()

        days = sdk2.parameters.Integer(
            'How many days of a week to count (2 by default - yesterday, today will always be included)?',
            default=2,
        )
        maximum_cache_size = sdk2.parameters.Integer(
            'Maximum amount of VOD in bytes to process',
            default=2000*(1024**3),
        )

        vault_owner = sdk2.parameters.String(
            'Owner of vault secrets',
            required=True,
        )
        vault_token = sdk2.parameters.String(
            'Name of YQL token secret',
            required=True,
        )

        cache_zones_url = sdk2.parameters.String(
            'URL for fetching information about cache zones',
            default='http://pinger.strm.yandex.net/api/v1/status/cache',
        )
        cache_zones_url_timeout = sdk2.parameters.Float(
            'Timeout in seconds for fetching information about cache zones',
            default=3,
        )
        cache_zones_url_max_retries = sdk2.parameters.Integer(
            'Max number of retries for fetching information about cache zones',
            default=3,
        )

    def on_execute(self):
        super(StrmBuildVodTop, self).on_execute()
        try:
            from yql.api.v1.client import YqlClient
            from library.python import resource
            import sandbox.projects.Strm.common.yql_helpers as yql_helpers

            token = sdk2.Vault.data(
                self.Parameters.vault_owner,
                self.Parameters.vault_token,
            )

            client = YqlClient(db='hahn', token=token)

            query = resource.find('sandbox/projects/Strm/StrmBuildVodTop/query.sql')
            logging.info('Query body: %s', query)

            query_params = self._get_query_params()
            logging.info('Query params: %s', query_params)

            result_table = yql_helpers.execute_yql_query(
                client,
                query,
                syntax_version=1,
                attach_files=[
                    ('schemes.sql', resource.find('strm/yql/vod_uri/schemes/schemes.sql'))
                ],
                title='STRM VoD Top',
                parameters=query_params,
            )

            result_resource = StrmVodTop(
                self,
                'STRM top VOD content statistics',
                'result',
                creation_ts=int(time.time()),
            )
            result = sdk2.ResourceData(result_resource)

            yql_helpers.dump_yql_table_to_resource(result_table, result_resource)
            result.ready()

        except Exception:
            logging.exception('Failed to build top')
            raise

    def _get_query_params(self):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder
        from sandbox.projects.Strm.common import cache_zones

        start_date = datetime.datetime.today() - datetime.timedelta(days=int(self.Parameters.days))

        locations_vod_cache_sizes = cache_zones.get_locations_vod_cache_sizes(
            cache_zones_url=self.Parameters.cache_zones_url,
            cache_zones_url_timeout=self.Parameters.cache_zones_url_timeout,
            cache_zones_url_max_retries=self.Parameters.cache_zones_url_max_retries,
        )

        maximum_cache_size = self.Parameters.maximum_cache_size
        if locations_vod_cache_sizes:
            maximum_cache_size = min(
                maximum_cache_size,
                max(locations_vod_cache_sizes.itervalues()),
            )

        params = {
            "$start_date": ValueBuilder.make_string(start_date.strftime('%Y-%m-%dT%H-00-00')),
            "$maximum_cache_size": ValueBuilder.make_uint64(maximum_cache_size),
        }

        return ValueBuilder.build_json_map(params)
