# coding=utf8

import gzip

import requests
import six
from sandbox import sdk2

from sandbox.projects.common import task_env


class StrmHexDataDump(sdk2.Resource):
    """
    Dump of HEX metrics and prefixes
    """

    releasable = False


class StrmDumpHexData(sdk2.Task):
    """
    Task to dump metrics and prefixes from HEX as sandbox resource
    """

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        dump_urls = sdk2.parameters.Dict(
            "URLs to dump",
            default={
                "metrics": "https://rdr.tt.yandex.net/api/v3.0.2/metrics",
            },
            required=True,
        )

        resource_ttl = sdk2.parameters.Integer("Resource TTL", default=14)

    def on_execute(self):
        sess = requests.Session()

        resource = StrmHexDataDump(self, "RDR data dump", "dump")
        resource.ttl = self.Parameters.resource_ttl

        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)

        for name, url in six.iteritems(self.Parameters.dump_urls):
            self._dump_url(sess, name, data, url)

    @staticmethod
    def _dump_url(sess, name, data, url):
        # type: (requests.Session, six.string_types, sdk2.ResourceData, six.string_types) -> None

        with sess.get(url, stream=True) as req:
            with data.path.joinpath("{}_headers.txt".format(name)).open("wt") as f:
                f.write(u"Status: {}\n".format(req.status_code))
                for header, value in six.iteritems(req.headers):
                    f.write(u"{}: {}\n".format(header, value))

            with gzip.open(str(data.path.joinpath("{}.json.gz".format(name))), "wb") as f:
                for chunk in req.iter_content(4096):
                    f.write(chunk)
