import io
import json
import logging
import time

from datetime import datetime, timedelta
import requests

import sandbox.projects.common.binary_task as binary_task
from sandbox import sdk2


TRAFFIC_QUERY_YQL_PATH = 'sandbox/projects/Strm/StrmEvaluateKalturaTraffic/query.sql'


class StrmKalturaTraffic(sdk2.Resource):
    releasable = False
    auto_backup = True
    creation_ts = sdk2.Attributes.Integer('creation ts')


class StrmEvaluateKalturaTraffic(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """ Prepare resource with STRM Evaluate Kaltura Traffic using common VOD URI schemes
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        ext_params = binary_task.LastBinaryReleaseParameters()

        hours = sdk2.parameters.Integer(
            'How many hours to cover counting importance',
            default=5,
        )

        vault_owner = sdk2.parameters.String(
            'Owner of vault secrets',
            required=True,
        )
        vault_token = sdk2.parameters.String(
            'Name of YQL token secret',
            required=True,
        )

        stat_report_data_upload_url = sdk2.parameters.String(
            'URL to upload data to stat',
            default='https://upload.stat.yandex-team.ru/_api/report/data',
        )
        stat_report_name = sdk2.parameters.String(
            'Name of the stat report',
            default='Video.All/STRM/KalturaVOD/Traffic',
        )
        stat_report_scale = sdk2.parameters.String(
            'Stat report scale',
            default='hourly',
        )
        stat_token = sdk2.parameters.String(
            'Name of stat token secret',
            required=True,
        )

    def on_execute(self):
        super(StrmEvaluateKalturaTraffic, self).on_execute()
        try:
            from yql.api.v1.client import YqlClient
            from library.python import resource
            import sandbox.projects.Strm.common.yql_helpers as yql_helpers

            token = sdk2.Vault.data(
                self.Parameters.vault_owner,
                self.Parameters.vault_token,
            )
            client = YqlClient(token=token)

            traffic_request_body = resource.find(TRAFFIC_QUERY_YQL_PATH)
            logging.info('Traffic request body: %s', traffic_request_body)
            traffic_request_params = self._traffic_request_params()
            logging.info('Traffic request params: %s', traffic_request_params)

            traffic_result_table = yql_helpers.execute_yql_query(
                client,
                traffic_request_body,
                syntax_version=1,
                attach_files=[
                    ('schemes.sql', resource.find('strm/yql/vod_uri/schemes/schemes.sql')),
                ],
                title='Evaluate Kaltura Traffic',
                parameters=traffic_request_params,
            )
            traffic_table = self._get_table_map(traffic_result_table)

            total_traffic = 0
            total_kaltura_traffic = 0
            for t in traffic_table:
                total_traffic += t['traffic']
                total_kaltura_traffic += t['kaltura_traffic']
            total = {
                'traffic': total_traffic,
                'kaltura_traffic': total_kaltura_traffic,
                'bucket': 'total',
            }
            traffic_table.append(total)

            result_resource = StrmKalturaTraffic(
                self,
                'STRM kaltura share traffic',
                'result',
                creation_ts=int(time.time()),
            )
            result = sdk2.ResourceData(result_resource)
            result_file_name = str(result_resource.path)
            logging.info('Dump results to file: %s', result_file_name)
            with io.open(result_file_name, 'w', encoding='utf-8') as res_file:
                res_file.write(
                    json.dumps(
                        traffic_table,
                        ensure_ascii=False,
                        encoding='utf-8',
                    ).decode('utf-8'),
                )

            result.ready()

            fielddate = datetime.now().strftime('%Y-%m-%d %H:00:00')
            stat_data = [
                {
                    'fielddate': fielddate,
                    'kaltura_share': float(traffic_data['kaltura_traffic']) / float(max(traffic_data['traffic'], 1)),
                    'total_traffic_share': float(traffic_data['traffic']) / float(max(total_traffic, 1)),
                    'kaltura_share_among_total_traffic': float(traffic_data['kaltura_traffic']) / float(max(total_traffic, 1)),
                    'kaltura_share_among_total_kaltura_traffic': float(traffic_data['kaltura_traffic']) / float(max(total_kaltura_traffic, 1)),
                    'bucket': traffic_data['bucket'],
                }
                for traffic_data in traffic_table
            ]
            stat_report_url = '/'.join([
                self.Parameters.stat_report_data_upload_url,
                self.Parameters.stat_report_name,
            ])
            stat_token = sdk2.Vault.data(
                self.Parameters.vault_owner,
                self.Parameters.stat_token,
            )
            self._upload_data_to_stat(
                stat_report_url,
                stat_token,
                stat_data,
                self.Parameters.stat_report_scale,
            )
        except Exception:
            logging.exception('Failed to build stat transcoding vod top')
            raise

    @staticmethod
    def _get_table_map(table):
        columns = [
            column
            for column, _ in table.columns
        ]
        table_mapping = [
            dict(zip(columns, row))
            for row in table.rows
        ]
        return table_mapping

    def _traffic_request_params(self):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        time_back = timedelta(hours=int(self.Parameters.hours))
        start_date = datetime.today() - time_back

        params = {
            "$start_date": ValueBuilder.make_string(start_date.strftime('%Y-%m-%dT%H-00-00')),
        }

        return ValueBuilder.build_json_map(params)

    @staticmethod
    def _upload_data_to_stat(url, token, data, scale):
        response = requests.post(
            url,
            headers={
                'Authorization': 'OAuth {token}'.format(token=token)
            },
            json={
                'data': data,
            },
            params={
                'scale': scale,
            },
        )
        logging.info(
            'Stat data upload operation uuid = %s',
            response.text,
        )
