# -*- coding: utf-8 -*-
import logging
import os
import shutil
import tarfile
import tempfile
from datetime import datetime, timedelta
from multiprocessing.dummy import Pool

import requests
import six
import six.moves
from requests.adapters import HTTPAdapter
from typing import Optional, Tuple
from urllib3.util.retry import Retry

import boto3
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
import sandbox.projects.common.file_utils as fu
from sandbox import sdk2
from sandbox.agentr.errors import ResourceNotAvailable
from sandbox.common.errors import TaskError, TaskFailure
from sandbox.projects.common.juggler import jclient
from sandbox.projects.resource_types import FFMPEG_BUNDLE, AbstractResource
from sandbox.sdk2.helpers import subprocess as sp


class FFMPEG_AD_DB(AbstractResource):
    """
        ffmpeg bundle built in Arcadia
    """
    releasable = False
    any_arch = True
    auto_backup = False
    executable = False
    share = True
    sync_upload_to_mds = True


def notify_juggler(self, msg, status):
    if self.is_auto_release():
        try:
            jclient.send_events_to_juggler(
                '{}_{}'.format(
                    FFMPEG_AD_DB.__name__.lower(),
                    self.Parameters.auto_release
                ),
                jclient.to_juggler_service_name(self.type.name.lower()),
                status,
                msg
            )
        except Exception:
            self.set_info('Failed to send event to juggler')
            logging.error('Failed to send event to juggler', exc_info=True)


def on_finish(self, prev_status, status):
    if status == ctt.Status.SUCCESS:
        self.notify_juggler(status, status='OK')
    else:
        self.notify_juggler('{} https://sandbox.yandex-team.ru/task/{}/view'.format(status, self.id), status='WARN')


def get_resource(res):
    # type: (sdk2.Resource) -> Tuple[sdk2.Resource, Optional[sdk2.ResourceData]]
    try:
        return res, sdk2.ResourceData(res)
    except ResourceNotAvailable:
        return res, None


class StrmFFmpegADCVDBGenerator(sdk2.Task):
    """ An empty task, which does nothing except of greets you. """

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 16384
        client_tags = ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Task.Context):
        childs = []

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 10800

        # custom parameters
        channel_name = sdk2.parameters.String('Channel slug name')
        channels_name = sdk2.parameters.List('Channels name')
        channels_name_from_pult = sdk2.parameters.Bool('Get channels name from pult', default=False)
        with channels_name_from_pult.value[True]:
            channels_pult_endpoint_url = sdk2.parameters.Url(
                'Channels pult endpoint URL', default='https://cfg.strm.yandex.net/api/v1/config/channels'
            )

        number_of_days = sdk2.parameters.Integer('Number of days')
        date = sdk2.parameters.String('Date chunks download')

        with sdk2.parameters.String("ffmpeg bundl released") as ffmpeg_bundl_released:
            ffmpeg_bundl_released.values[ctt.ReleaseStatus.TESTING] = ctt.ReleaseStatus.TESTING
            ffmpeg_bundl_released.values[ctt.ReleaseStatus.PRESTABLE] = ctt.ReleaseStatus.PRESTABLE
            ffmpeg_bundl_released.values[ctt.ReleaseStatus.STABLE] = ffmpeg_bundl_released.Value(
                ctt.ReleaseStatus.STABLE, default=True)

        resolution = sdk2.parameters.String('Channel resolution', default='')
        force = sdk2.parameters.Bool('Force regenerate daily thdb', default=False)

        with sdk2.parameters.Group('duplicates analyser settings') as da_group:
            da_min_ad_thickness = sdk2.parameters.Integer('Minimal number of ad repetitions')
            da_min_flick_duration = sdk2.parameters.Integer('Minimal ad flick duration in seconds')
            da_max_flick_duration = sdk2.parameters.Integer('Maximal ad flick duration in seconds')
            da_min_block_duration = sdk2.parameters.Integer('Minimal ad block duration in seconds')
            da_max_block_duration = sdk2.parameters.Integer('Maximal ad block duration in seconds')

        debug = sdk2.parameters.Bool('For debugging task', default=False)

    def get_db_compatibility_version(self):
        cmd = (
            '{path}/frame_extractor --svnrevision |'
            ' grep -m1 -o -P "database compatibility version [\\d.]+$" |'
            ' tr -d "\n"'
        ).format(path=self.ffmpeg_adcv.path)
        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('cmd')
        ) as pl:
            try:
                out = sp.check_output(
                    cmd, shell=True, stderr=pl.stderr
                )
            except sp.CalledProcessError:
                out = ''
        self.Context.db_version = out.split(' ')[-1] if out else '1'

    def s3_get_marker(self):
        timestamp = int(datetime.strptime(
            self.Parameters.date, '%Y-%m-%d'
        ).strftime('%s'))
        for s in six.moves.range(90):
            key = '{name}/{name}0_{resolution}-{timestamp}000.{ext}'.format(
                ext='mp4',
                name=self.Parameters.channel_name,
                resolution=self.Parameters.resolution,
                timestamp=timestamp+s*10
            )
            try:
                self.s3client.head_object(Bucket='strm', Key=key)
            except:
                pass
            else:
                return key
        return None

    def s3_get_chunks_keys(self):
        marker = self.s3_get_marker()
        if not marker:
            error_descr = 'Cannot find first chunk for date {}'.format(self.Parameters.date)
            fu.write_file(
                '{}/error'.format(self.daily_thdb_resource_data.path),
                error_descr
            )
            self.daily_thdb_resource_data.broken()
            raise TaskFailure(error_descr)
        chunk_keys = []
        prefix = '{name}/{name}0_{resolution}'.format(
            name=self.Parameters.channel_name,
            resolution=self.Parameters.resolution
        )
        date = datetime.strptime(self.Parameters.date, '%Y-%m-%d').date()
        while True:
            bucket_objects = self.s3client.list_objects(
                Bucket=self.s3_bucket_name,
                Prefix=prefix,
                Marker=marker
            )
            if 'Contents' in bucket_objects:
                for obj in bucket_objects['Contents']:
                    if obj['LastModified'].date() == date:
                        chunk_keys.append(obj['Key'])
                    elif obj['LastModified'].date() > date:
                        break
                if bucket_objects['IsTruncated']:
                    if self.Parameters.debug:
                        if len(chunk_keys) == 0:
                            marker = bucket_objects['Contents'][-1]['Key']
                        else:
                            chunk_keys = chunk_keys[0:100]
                            break
                    else:
                        marker = bucket_objects['Contents'][-1]['Key']
                else:
                    break
            else:
                break
        if len(chunk_keys) > 0:
            return chunk_keys
        else:
            self.daily_thdb_resource_data.broken()
            raise TaskFailure(
                'Something went wrong chunk keys is zero'
            )

    def s3_download_chunk(self, key):
        self.s3client.download_file(
            self.s3_bucket_name, key, '{}/{}'.format(self.temp_dir, key)
        )

    @staticmethod
    def make_imparser_config(imparser_config, mode):
        with open(imparser_config, 'wt') as config:
            config.write('ThumbsCount = 0\n')
            config.write('CalcImgSigs = %d\n' % (mode == 'fastmatcher'))
            config.write('CalcExtImageAttributesChunk = 0\n')
            config.write('CalcGeomQuantization = %d\n' % (mode == 'copysearcher'))
            config.write('CalcQuantizedDescriptors = %d\n' % (mode == 'copysearcher'))
            config.write('CalcExtImageAttributesChunk = 0\n')
            config.write('CalcQualityChunk = 0\n')
            config.write('DetectFaces = 0\n')
            config.write('CalcFaceKeyAnnotationChunk = 0\n')
            config.write('CalcHistogramChunk = 0\n')
            config.write('ExtractExifTags = 0\n')
            config.write('RunOldPornoClassifier = 0\n')
            config.write('AnalyseAttributes = 0\n')
            config.write('AnalyseExtThumb = 0\n')
            config.write('CalcColorStatsChunk = 0\n')

    @staticmethod
    def create_requests_session():
        session = requests.Session()
        retry = Retry(total=5, backoff_factor=1, status_forcelist=(500, 502, 503, 504))
        session.mount('http://', HTTPAdapter(max_retries=retry))
        session.mount('https://', HTTPAdapter(max_retries=retry))

        return session

    def on_enqueue(self):
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[ctt.Semaphores.Acquire(name='strm_ffmeg_addb_generation')]
        )

    def get_or_gen_thdb_daily(self, name, datestr):
        res_thdb = list(sdk2.Resource.find(
            FFMPEG_AD_DB, state=[ctr.State.READY, ctr.State.BROKEN],
            attrs=dict(
                date=datestr,
                type='daily',
                channel=name,
                ffmpeg_adcv_db_version=self.Context.db_version
            )
        ).limit(1))
        if not res_thdb:
            res_thdb = list(sdk2.Resource.find(
                FFMPEG_AD_DB, state=[ctr.State.READY, ctr.State.BROKEN],
                attrs=dict(
                    date=datestr,
                    type='daily',
                    channel=name.split('_')[0],
                    ffmpeg_adcv_db_version=self.Context.db_version
                )
            ).limit(1))
        if res_thdb and not (
                len(self.Context.childs) == 0 and
                self.Parameters.force
        ) and res_thdb[0].state == 'READY':
            self.thdb_daily.append(res_thdb[0])
            logging.info(
                'Resources append {}/{}'.format(
                    res_thdb[0].channel,
                    res_thdb[0].date
                )
            )
        elif res_thdb and res_thdb[0].state == 'BROKEN':
            logging.warning(
                'Skip resource for date {}, because it broken'.format(
                    res_thdb[0].date
                )
            )
        else:
            child = StrmFFmpegADCVDBGenerator(
                self,
                owner=self.owner,
                description='Child of StrmFFmpegADCVDBGenerator task {} for channel {} for {}'.format(self.id, name, datestr),
                priority=self.Parameters.priority,
                channels_name=None,
                channels_name_from_pult=False,
                number_of_days=None,
                channel_name=name,
                resolution=self.Parameters.resolution,
                notifications=self.Parameters.notifications,
                ffmpeg_bundl_released=self.Parameters.ffmpeg_bundl_released,
                debug=self.Parameters.debug,
                date=datestr
            )
            self.Context.childs.append(child.enqueue().id)

    def on_execute(self):
        self.ffmpeg_bundl = list(sdk2.Resource.find(
                FFMPEG_BUNDLE,
                state=[ctr.State.READY],
                attrs=dict(released=self.Parameters.ffmpeg_bundl_released)
            ).limit(1))[0]
        self.ffmpeg_adcv = sdk2.ResourceData(self.ffmpeg_bundl)
        logging.info('Resource downloaded')

        self.get_db_compatibility_version()
        logging.info('get_db_compatibility_version: {}'.format(self.Context.db_version))

        self.thdb_daily = []
        self.threads = 8
        self.temp_dir = tempfile.mkdtemp()
        self.Context.childs = []
        pool = Pool(self.threads)

        channels_name = self.Parameters.channels_name
        if self.Parameters.channels_name_from_pult:
            channels_name = []

            session = self.create_requests_session()
            req = session.get(self.Parameters.channels_pult_endpoint_url)
            req.raise_for_status()

            pult_channels = req.json().get('sources', {})

            for pult_channel, channel_config in six.iteritems(pult_channels):
                if (
                    channel_config.get('removed') != '1'
                    and channel_config.get('archived') != '1'
                    and channel_config['options'].get('adcv') == '1'
                    and channel_config['options'].get('adcv2') != '1'
                ):
                    channels_name.append(pult_channel)

        # Создание базы для канала за 1 день
        if self.Parameters.date and self.Parameters.channel_name:
            self.s3_bucket_name = 'strm'
            s3_endpoint_url = 'http://s3.mds.yandex.net'
            s3_secret_key = sdk2.Vault.data(self.owner, 's3_secret_key')
            s3_secret_key_id = sdk2.Vault.data(self.owner, 's3_secret_key_id')
            self.s3client = boto3.client(
                's3', endpoint_url=s3_endpoint_url,
                aws_access_key_id=s3_secret_key_id,
                aws_secret_access_key=s3_secret_key
            )
            logging.info('S3 connected')

            os.makedirs(
                '{}/{}'.format(self.temp_dir, self.Parameters.channel_name)
            )
            self.daily_thdb_resource_data = sdk2.ResourceData(
                FFMPEG_AD_DB(
                    self,
                    'daily thdb',
                    self.Parameters.channel_name,
                    channel=self.Parameters.channel_name,
                    date=self.Parameters.date,
                    type='daily',
                    ffmpeg_bundl_released=self.Parameters.ffmpeg_bundl_released,
                    ffmpeg_adcv_db_version=self.Context.db_version,
                    ffmpeg_adcv_arcadia_revision=self.ffmpeg_bundl.arcadia_revision
                )
            )
            self.daily_thdb_resource_data.path.mkdir(
                0o755, parents=True, exist_ok=True
            )
            pool.map(self.s3_download_chunk, self.s3_get_chunks_keys())

            logging.info('Video chunks downloaded')
            os.makedirs(
                '{}/{}-images'.format(
                    self.daily_thdb_resource_data.path, self.Parameters.date
                )
            )
            cmd_find = (
                'find {tmpdir} -name *.ts -o -name *.mp4 |'
                ' sort > {tmpdir}/{name}.videolist;'
            ).format(
                tmpdir=self.temp_dir, name=self.Parameters.channel_name
            )
            with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('cmd')
            ) as pl:
                pr = sp.Popen(
                    cmd_find, shell=True, stdout=pl.stdout, stderr=sp.STDOUT
                )
                pr.wait()

            cmd_frame_extractor = (
                '{path}/frame_extractor -i "{tmpdir}/{name}.videolist"'
                ' -o "{name}/{date}-images"'
                ' -s "{tmpdir}/{name}.frame-extractor-state"'
            ).format(
                tmpdir=self.temp_dir,
                path=self.ffmpeg_adcv.path,
                date=self.Parameters.date,
                name=self.Parameters.channel_name
            )
            retry_count = 0
            while True:
                retry_count += 1
                if retry_count > 10:
                    raise TaskFailure("frame_extractor retries limit reached")

                with sdk2.helpers.ProcessLog(
                    self, logger=logging.getLogger('cmd')
                ) as pl:
                    pr = sp.Popen(
                        cmd_frame_extractor,
                        shell=True,
                        stdout=pl.stdout,
                        stderr=sp.STDOUT
                    )
                    pr.wait()
                    if pr.returncode == 0 or pr.returncode == 1:
                        break
                    else:
                        logging.error("Unknown frame_extractor return code: %d", pr.returncode)

            logging.info('Frames extracted')

            imparser_config = '{}/imparser.config'.format(self.temp_dir)
            self.make_imparser_config(imparser_config, 'fastmatcher')
            cmd_imparser = (
                '{path}/imparsertest -c "{config}"'
                ' -d "{name}/{date}-images"'
                ' -o "{name}/{date}.thdb" -n {threads} |'
                ' tee {name}/{date}.imparser &&'
                ' tar -cf {name}/{date}-images.tar'
                ' -C {name}/{date}-images . --remove-files'
            ).format(
                path=self.ffmpeg_adcv.path,
                config=imparser_config,
                threads=self.threads,
                date=self.Parameters.date,
                name=self.Parameters.channel_name,
                tmpdir=self.temp_dir
            )
            with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('cmd')
            ) as pl:
                pr = sp.Popen(
                    cmd_imparser, shell=True, stdout=pl.stdout, stderr=sp.STDOUT
                )
                pr.wait()
                if pr.returncode != 0:
                    raise TaskError(
                        'Something went wrong in frames parsing '
                    )
            logging.info('Frames parsed')
            self.daily_thdb_resource_data.ready()

        # Создание базы для канала за несколько дней
        elif len(channels_name) > 0:
            from .playlist import get_min_resolution

            session = self.create_requests_session()

            for name in channels_name:
                try:
                    min_resolution = get_min_resolution(session, name)
                except Exception:
                    logging.error('Problem to get resolution for channel {}'.format(name))
                else:
                    child = StrmFFmpegADCVDBGenerator(
                        self,
                        owner=self.owner,
                        description='Child of StrmFFmpegADCVDBGenerator task {} for channel {}'.format(self.id, name),
                        priority=self.Parameters.priority,
                        channels_name=None,
                        channels_name_from_pult=False,
                        number_of_days=self.Parameters.number_of_days,
                        channel_name=name,
                        resolution=min_resolution,
                        notifications=self.Parameters.notifications,
                        ffmpeg_bundl_released=self.Parameters.ffmpeg_bundl_released,
                        debug=self.Parameters.debug,
                    )
                    child.enqueue()

        # Создание дочерних задач для каждого канала
        elif self.Parameters.channel_name and self.Parameters.number_of_days and self.Parameters.number_of_days > 0:
            self.channel_resolutions = {}
            for day_num in six.moves.range(self.Parameters.number_of_days, 0, -1):
                datestr = (
                    datetime.today().date() - timedelta(days=day_num)
                ).isoformat()
                self.get_or_gen_thdb_daily(self.Parameters.channel_name, datestr)

            if len(self.Context.childs) > 0:
                wait_timeout = self.Parameters.kill_timeout * 2
                raise sdk2.WaitTask(
                    self.Context.childs,
                    ctt.Status.Group.FINISH,
                    timeout=wait_timeout,
                    wait_all=True
                )

            for task_id in self.Context.childs:
                self.thdb_daily.extend(
                    sdk2.Resource.find(
                        FFMPEG_AD_DB,
                        attrs=dict(task_id=task_id)
                    ).limit(1)
                )

            logging.info(
                'Resources found count {}'.format(len(self.thdb_daily))
            )

            resources = pool.map(get_resource, self.thdb_daily)
            logging.info('All resources synchronized')

            if len(self.thdb_daily) > 0:
                with open(
                    '{}/merged_thdb.list'.format(self.temp_dir),
                    'wt'
                ) as merged_thdb_list:
                    for thdb, resource_data in resources:
                        if resource_data is None:
                            logging.warning("Can't get resource #%s, skipping", thdb.id)
                            continue

                        thdb_dir = os.path.join(self.temp_dir, thdb.channel)
                        if not os.path.isdir(thdb_dir):
                            os.mkdir(thdb_dir)
                        merged_thdb_list.write('{}/{}.thdb\n'.format(
                            resource_data.path,
                            thdb.date
                        ))
                        imgs = '{}/{}/{}-images'.format(
                            self.temp_dir, thdb.channel, thdb.date
                        )
                        with tarfile.open(
                            six.text_type(
                                resource_data.path.joinpath(
                                    '{}-images.tar'.format(
                                        thdb.date
                                    )
                                )
                            )
                        ) as tar:
                            if not os.path.exists(imgs):
                                os.mkdir(imgs)
                            tar.extractall(path=imgs)
                        logging.info('Synchronized {}/{}'.format(
                            resource_data.path, thdb.date
                        ))
            else:
                raise TaskFailure('Something went wrong thdb daily resources not found')

            first_datestr = (datetime.today().date() - timedelta(days=1)).isoformat()
            cmd_merge_thdbs = (
                '{path}/merge_thdbs -i merged_thdb.list'
                ' -o merged.thdb'
            ).format(path=self.ffmpeg_adcv.path)

            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
                pr = sp.Popen(cmd_merge_thdbs, shell=True, cwd=self.temp_dir, stdout=pl.stdout, stderr=sp.STDOUT)
                pr.wait()
                if pr.returncode != 0:
                    raise TaskFailure('Something went wrong in merge thdbs')

            logging.info('thdb merged')
            cmd_fm_loaddata = (
                '{path}/fastmatcher3 loaddata'
                ' --thdb merged.thdb'
                ' --target-state merged.state'
                ' --threads {threads}'
            ).format(
                dir=self.temp_dir,
                path=self.ffmpeg_adcv.path,
                threads=self.threads
            )
            with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('cmd')
            ) as pl:
                pr = sp.Popen(
                    cmd_fm_loaddata,
                    shell=True,
                    cwd=self.temp_dir,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT
                )
                pr.wait()
                if pr.returncode != 0:
                    raise TaskFailure(
                        'Something went wrong in fastmatcher loaddata'
                    )
            logging.info('Data loadded in fastmatcher')

            cmd_fm_makeindex = (
                '{path}/fastmatcher3 makeindex'
                ' --target-state merged.state'
                ' --threads {threads}'
            ).format(
                dir=self.temp_dir,
                path=self.ffmpeg_adcv.path,
                threads=self.threads
            )
            with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('cmd')
            ) as pl:
                pr = sp.Popen(
                    cmd_fm_makeindex, shell=True,
                    cwd=self.temp_dir,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT
                )
                pr.wait()
                if pr.returncode != 0:
                    raise TaskFailure(
                        'Something went wrong in fastmatcher makeindex'
                    )
            logging.info('Index made in fastmatcher')

            cmd_fm_export_crc2group = (
                '{path}/fastmatcher3 export_crc2group'
                ' --target-state merged.state'
                ' --text-out merged.bundles'
            ).format(
                dir=self.temp_dir,
                path=self.ffmpeg_adcv.path
            )
            with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('cmd')
            ) as pl:
                pr = sp.Popen(
                    cmd_fm_export_crc2group,
                    shell=True,
                    cwd=self.temp_dir,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT
                )
                pr.wait()
                if pr.returncode != 0:
                    raise TaskFailure(
                        'Something went wrong in fastmatcher export_crc2group'
                    )
            logging.info('exported crc2group in fastmatcher')

            report_resource_data = sdk2.ResourceData(FFMPEG_AD_DB(
                self,
                'report of {}'.format(self.Parameters.channel_name),
                'report',
                channel=self.Parameters.channel_name,
                date=first_datestr,
                type='report',
                ffmpeg_bundl_released=self.Parameters.ffmpeg_bundl_released,
                ffmpeg_adcv_db_version=self.Context.db_version,
                ffmpeg_adcv_arcadia_revision=self.ffmpeg_bundl.arcadia_revision
            ))
            report_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
            da_config = []
            if self.Parameters.da_min_ad_thickness > 0:
                da_config.append(('--min_ad_thickness {}'.format(self.Parameters.da_min_ad_thickness)))
            if self.Parameters.da_min_flick_duration > 0:
                da_config.append(('--min_flick_duration {}'.format(self.Parameters.da_min_flick_duration)))
            if self.Parameters.da_max_flick_duration > 0:
                da_config.append(('--max_flick_duration {}'.format(self.Parameters.da_max_flick_duration)))
            if self.Parameters.da_min_block_duration > 0:
                da_config.append(('--min_block_duration {}'.format(self.Parameters.da_min_block_duration)))
            if self.Parameters.da_max_block_duration > 0:
                da_config.append(('--max_block_duration {}'.format(self.Parameters.da_max_block_duration)))
            cmd_duplicates_analyser = (
                '{path}/duplicates_analyser'
                ' {config}'
                ' --list merged.imparser'
                ' --bundles merged.bundles'
                ' --outputdir {cwd} --report'
            ).format(
                path=self.ffmpeg_adcv.path, config=' '.join(da_config), cwd=report_resource_data.path
            )
            with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('cmd')
            ) as pl:
                pr = sp.Popen(
                    cmd_duplicates_analyser, shell=True,
                    cwd=self.temp_dir, stdout=pl.stdout, stderr=sp.STDOUT
                )
                pr.wait()
                if pr.returncode != 0:
                    raise TaskFailure(
                        'Something went wrong in analyser duplicates'
                    )
            logging.info('Duplicates analysed')
            channel_name = self.Parameters.channel_name
            merged_thdb_resource_data = sdk2.ResourceData(FFMPEG_AD_DB(
                self,
                'merged thdb',
                self.Parameters.channel_name,
                channel=channel_name,
                date=first_datestr,
                type='merged',
                ffmpeg_bundl_released=self.Parameters.ffmpeg_bundl_released,
                ffmpeg_adcv_db_version=self.Context.db_version,
                ffmpeg_adcv_arcadia_revision=self.ffmpeg_bundl.arcadia_revision
            ))
            merged_thdb_resource_data.path.mkdir(
                0o755, parents=True, exist_ok=True
            )
            imparser_config = '{}/{}.imparser_config'.format(
                self.temp_dir, self.Parameters.channel_name
            )
            self.make_imparser_config(imparser_config, 'copysearcher')
            cmd_imparser = (
                '{path}/imparsertest -c "{config}"'
                ' -d "{report_path}/{name}-ad-images"'
                ' -o "{cwd}/{date}-ad.thdb"'
                ' -n {threads} | tee {cwd}/{date}-ad.imparser'
            ).format(
                config=imparser_config,
                report_path=report_resource_data.path,
                cwd=merged_thdb_resource_data.path,
                date=first_datestr,
                path=self.ffmpeg_adcv.path, name=self.Parameters.channel_name, threads=self.threads
            )
            with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('cmd')
            ) as pl:
                pr = sp.Popen(
                    cmd_imparser, shell=True, cwd=self.temp_dir,
                    stdout=pl.stdout, stderr=sp.STDOUT
                )
                pr.wait()
                if pr.returncode != 0:
                    raise TaskFailure(
                        'Something went wrong in frames ad parsing '
                    )
            logging.info('Frames {}-ad parsed'.format(self.Parameters.channel_name))
            merged_thdb_resource_data.ready()
            logging.info('Archive all directory for report resource')
            for directory in os.listdir(str(report_resource_data.path)):
                path = '/'.join([str(report_resource_data.path), directory])
                if os.path.isdir(path):
                    logging.info('Archiving directory {}'.format(directory))
                    with tarfile.open(path + '.tar.gz', 'w:gz') as tar:
                        tar.add(path, arcname=os.path.basename(directory))
                    shutil.rmtree(path)
                    logging.info('Done')
            report_resource_data.ready()
