import logging
import tempfile

from sandbox import sdk2

from sandbox.projects.Strm.StrmGenerateStubs.config import get_stubs_config, StubMetaExtractor
from sandbox.projects.Strm.common.s3_client import get_s3_client
from sandbox.projects.Strm.common.resource import get_ffmpeg


logger = logging.getLogger(__name__)


def transcode_and_save_stub(ffmpeg, s3_client, s3_bucket, original_stub_path, stub_config):
    with tempfile.NamedTemporaryFile() as transcoded_stub:
        transcoded_stub_path = transcoded_stub.name
        ffmpeg.transcode_video(original_stub_path, transcoded_stub_path, stub_config.transcoding_options)
        stub_meta_extractor = StubMetaExtractor(ffmpeg)
        try:
            video_meta = stub_meta_extractor.extract_video_meta(transcoded_stub_path)
            audio_meta = stub_meta_extractor.extract_audio_meta(transcoded_stub_path)
        except Exception as e:
            raise RuntimeError('Stub generated for name {path} is not correct: {reason}'.format(
                path=stub_config.name,
                reason=e.message,
            ))

        video_present = stub_config.transcoding_options.filter.video_tracks_number > 0
        audio_present = stub_config.transcoding_options.filter.audio_tracks_number > 0
        bad_meta = (video_present and video_meta != stub_config.meta.video_track) or \
                   (not video_present and video_meta is not None) or \
                   (audio_present and audio_meta != stub_config.meta.audio_track) or \
                   (not audio_present and audio_meta is not None)

        if bad_meta:
            raise RuntimeError('Stub has not expected stub_meta: {stub_meta}'.format(stub_meta=stub_meta))

        try:
            s3_client.upload(s3_bucket, transcoded_stub_path, stub_config.name)
        except Exception as e:
            raise RuntimeError('Failed to upload generated stub to S3: {reason}'.format(reason=e.message))

    return


class StrmGenerateStubs(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):

        s3_credentials_owner = sdk2.parameters.String(
            'S3 credentials owner',
            required=True,
            default='robot-strm-test',
        )

        s3_url = sdk2.parameters.String(
            'S3 url',
            required=True,
            default='http://s3.mdst.yandex.net',
        )

        s3_bucket = sdk2.parameters.String(
            'S3 bucket with stubs',
            required=True,
            default='strm-stub',
        )

        # TODO: Replace by getting original stub name automatically depending on aspect ratio
        s3_original_stub_path = sdk2.parameters.String(
            'S3 original stub path',
            required=True,
            default='stub_figure_default_kaltura_3840x2160.mov',
        )

        s3_transcoding_options_json_path = sdk2.parameters.String(
            'S3 stub transcoding options json path',
            required=True,
            default='stubs_transcoding_options.json'
        )

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(attrs={'name': 'StrmGenerateStubs'}).first()

    def on_execute(self):
        s3_client = get_s3_client(self.Parameters.s3_url, self.Parameters.s3_credentials_owner)
        ffmpeg = get_ffmpeg()
        try:
            stubs_config = get_stubs_config(
                s3_client,
                self.Parameters.s3_bucket,
                self.Parameters.s3_transcoding_options_json_path,
                ffmpeg,
            )
        except Exception as e:
            raise RuntimeError('Failed to get stub transcoding options list: {reason}'.format(reason=e.message))

        with tempfile.NamedTemporaryFile() as local_original_stub:
            local_original_stub_path = local_original_stub.name
            try:
                s3_client.download(self.Parameters.s3_bucket, local_original_stub_path, self.Parameters.s3_original_stub_path)
            except Exception as e:
                raise RuntimeError('Failed to download original stub from S3: {reason}'.format(reason=e.message))
            for stub_config in stubs_config:
                try:
                    transcode_and_save_stub(
                        ffmpeg,
                        s3_client,
                        self.Parameters.s3_bucket,
                        local_original_stub_path,
                        stub_config,
                    )
                except Exception as e:
                    raise RuntimeError('Failed to process stub: {reason}'.format(reason=e.message))

        return
