from collections import namedtuple
import copy
import json
import logging
import tempfile


from sandbox.projects.Strm.common.ffmpeg_wrapper import FilterOptions, EncoderOptions, TranscoderOptions
from sandbox.projects.Strm.common.resolution import Resolution


logger = logging.getLogger(__name__)


class ParameterSet(namedtuple('ParameterSet', ['ps'])):
    def __eq__(self, other):
        if self.ps in other.ps:
            return True
        if other.ps in self.ps:
            return True
        return False


VideoTrackMeta = namedtuple('VideoTrackMeta', [
    'quality',
    'codec',
    'sps',
    'pps',
])


AudioTrackMeta = namedtuple('AudioTrackMeta', [
    'codecs',
    'sample_rate',
    'signal_type',
])


StubMeta = namedtuple('StubMeta', [
    'video_track',
    'audio_track',
])


def get_stub_meta(metadata):
    video_track_meta = VideoTrackMeta(
        quality=str(metadata['quality']),
        codec=str(metadata['video_codec']),
        sps=ParameterSet(metadata['video_params']),
        pps=ParameterSet(metadata['video_params']),
    )

    audio_track_meta = AudioTrackMeta(
        codecs=str(metadata['audio_codecs']),
        sample_rate=int(metadata['sample_rate']),
        signal_type=str(metadata['signal_type']),
    )

    return StubMeta(
        video_track=video_track_meta,
        audio_track=audio_track_meta,
    )


class StubMetaExtractor(namedtuple('StubMetaExtractor', ['ffmpeg'])):

    def extract_video_meta(self, stub_path):
        stub_info = self.ffmpeg.get_video_info(stub_path)

        if len(stub_info.video_tracks) == 0:
            return None
        
        # Stub should have all audio tracks with the same parameters so we can take the first
        stub_video_info = stub_info.video_tracks[0]
        for video_track_info in stub_info.video_tracks:
            if video_track_info != stub_video_info:
                raise RuntimeError('Stub has video tracks with different parameters')
        
        return VideoTrackMeta(
            quality=stub_video_info.resolution.get_quality(),
            codec=stub_video_info.codec,
            sps=ParameterSet(stub_video_info.sps),
            pps=ParameterSet(stub_video_info.pps),
        )
       

    def extract_audio_meta(self, stub_path):
        stub_info = self.ffmpeg.get_video_info(stub_path)

        if len(stub_info.audio_tracks) == 0:
            return None
        
        # Stub should have all audio tracks with the same parameters so we can take the first
        stub_audio_info = stub_info.audio_tracks[0]
        for audio_track_info in stub_info.audio_tracks:
            if audio_track_info != stub_audio_info:
                raise RuntimeError('Stub has audio tracks with different parameters')

        return AudioTrackMeta(
            codecs=audio_track_info.codecs,
            sample_rate=audio_track_info.sample_rate,
            signal_type=audio_track_info.signal_type,
        )


StubConfig = namedtuple('StubConfig', [
    'name',
    'transcoding_options',
    'meta',
])


def get_stubs_config(s3_client, s3_bucket, s3_options_json_path, ffmpeg):
    local_options_json = tempfile.NamedTemporaryFile()
    local_options_json_path = local_options_json.name
    try:
        s3_client.download(s3_bucket, local_options_json_path, s3_options_json_path)
    except Exception as e:
        raise RuntimeError('Failed to download transcoding options json from S3: {reason}'.format(reason=e.message))

    with open(local_options_json_path, 'r') as read_json:
        # Expected json format:
        # {
        #   "default_transcoding_options {
        #       "profile": "baseline",
        #       "fps": 25,
        #       ...
        #   }
        #   "stubs" [
        #       {
        #           "metadata": {
        #               "name": "stub_169_720p_h264_0142c029ffe100146742c029d9005005bb016c800000030080000019078c192401000468cb8cb2_mp4a.40.2_48000_stereo_1.mp4",
        #               "quality": "169_720p",
        #               "video_codec": "h264",
        #               "video_params": "0142c029ffe100146742c029d9005005bb016c800000030080000019078c192401000468cb8cb2",
        #               "audio_codec": "mp4a.40.2",
        #               "sample_rate": 48000,
        #               "signal_type": "stereo"
        #           },
        #           "transcoding_options": {
        #               "resolution": "1280x720",
        #               "framerate": 25,
        #               "video_profile": "baseline",
        #               "level": "4.1"
        #           }
        #       },
        #       ...
        #   ]
        # }
        config = json.load(read_json)
        stubs = config['stubs']

        stubs_config = []
        default_transcoding_options = config.get('default_transcoding_options', {})
        for stub in stubs:
            metadata = stub['metadata']
            stub_name = metadata['name']
            stub_meta = get_stub_meta(metadata)

            transcoding_options = copy.deepcopy(default_transcoding_options)
            transcoding_options.update(stub['transcoding_options'])

            if not transcoding_options.get('resolution'):
                logger.error('Failed to determine resolution for the stub {name}, omitting this stub.'.format(name=stub_name))
                continue

            filter_options = FilterOptions(
                resolution=Resolution.parse_from_name(transcoding_options['resolution']),
                framerate=transcoding_options.get('framerate'),
                pixel_format=transcoding_options.get('pixel_format', 'yuv420p'),
                video_tracks_number=transcoding_options.get('video_tracks_number', 1),
                audio_tracks_number=transcoding_options.get('audio_tracks_number', 1),
            )

            encoder_options = EncoderOptions(
                map_metadata=transcoding_options.get('map_metadata'),
                max_muxing_queue_size=transcoding_options.get('max_muxing_queue_size'),
                stream_duration=transcoding_options.get('stream_duration', 10.0),
                video_codec=transcoding_options.get('video_codec', 'h264'),
                x264_options=transcoding_options.get('x264_options'),
                preset=transcoding_options.get('preset'),
                video_profile=transcoding_options.get('video_profile'),
                level=transcoding_options.get('level'),
                video_bitrate=transcoding_options.get('video_bitrate'),
                video_max_bitrate=transcoding_options.get('video_max_bitrate'),
                video_buffer_size=transcoding_options.get('video_buffer_size'),
                gop_size=transcoding_options.get('gop_size'),
                force_key_frames=transcoding_options.get('force_key_frames'),
                forced_idr=transcoding_options.get('forced_idr'),
                audio_codec=transcoding_options.get('audio_codec', 'aac'),
                sample_rate=transcoding_options.get('sample_rate'),
                audio_bitrate=transcoding_options.get('audio_bitrate'),
                output_format=transcoding_options.get('output_format', 'mp4'),
            )
            transcoding_options = TranscoderOptions(filter=filter_options, encoder=encoder_options)
            stub_config = StubConfig(
                name=stub_name,
                transcoding_options=transcoding_options,
                meta=stub_meta,
            )
            stubs_config.append(stub_config)
        return stubs_config
