# coding: utf-8
import json
import sys
from sandbox import sdk2
import shutil
import logging
import tempfile
import traceback
from multiprocessing.dummy import Pool as ThreadPool
import os
import glob
import time
import requests
import shlex
import math
import operator
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from urllib import unquote_plus
try:
    from urllib.parse import urlparse
    from urllib.parse import parse_qs
except ImportError:
    from urlparse import urlparse
    from urlparse import parse_qs
import re
import xml.etree.ElementTree as XML
from sandbox import common
from sandbox.common.share import skynet_get
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
import sandbox.common.types.notification as ctn
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.common.errors import TemporaryError
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.resource import AbstractResource
from sandbox.projects import (
    resource_types,
    # sandbox
)
from sandbox.projects.Strm import resources as strm_resources

from uuid import UUID

from constants import CONTENT_TYPES as content_types
from constants import DEFAULT_RESOLUTIONS as resolutions
from constants import OTT_RESOLUTIONS as ott_resolutions
from constants import BROWSER_RESOLUTIONS as browser_resolutions
from constants import BANNERSTORAGE_RESOLUTIONS as bannerstorage_resolutions
from constants import EMBED_BASE_169_HEIGHTS as embed_base_169_heights
from constants import EMBED_ASPECT_RATIOS as embed_aspect_ratios
from constants import EMBED_INPUT_RESOLUTIONS as embed_input_resolutions
from constants import QUALITY_CRF as quality_crf
from constants import QUALITY_SCALE as quality_scale
from constants import YOUTUBE_DL_DOMAINS as youtube_dl_domains
from constants import YOUTUBE_DL_DOMAINS_SNI_REQUIRED as youtube_dl_domains_sni_required
from constants.audio import (
    OTT_AUDIO_LANG_DICT as ott_audio_lang_dict,
    OTT_AUDIO_LANG_NAME_DICT as ott_audio_lang_name_dict
)
from constants.subtitles import (
    OTT_SUB_LANG_DICT as ott_sub_lang_dict,
    OTT_SUB_LANG_NAME_DICT as ott_sub_lang_name_dict
)

from drm import playready

from drm.fairplay import Command
from drm.fairplay import MasterPlaylist as PlaylistM3U8
from drm.fairplay import create_subtitles_playlist

from drm.utils import (
    get_audio_lang,
    get_relative_path,
    get_fairplay_keys,
    get_absolute_path,
    file_ends_with,
    parse_rfc6381_codec
)

from drm.client import RTClient

class YOUTUBE_DL_BINARY(AbstractResource):
    """
        youtube-dl binary for youtube videos downloading in FAAS
    """
    releasable = True
    any_arch = False
    auto_backup = True
    executable = True
    releasers = ['toshik', 'eightn', 'hexagonal', 'whitefox', 'rikunov', 'mvtomilov']


class BENTO4_BINARIES(AbstractResource):
    """
        Bento4 binaries for DRM support in FAAS
    """
    releasable = True
    any_arch = False
    auto_backup = True
    executable = True
    ttl = 'inf'
    releasers = ['toshik', 'eightn', 'hexagonal', 'whitefox', 'rikunov']


class StrmVideoConvert(sdk2.Task):
    """ This task runs ffmpeg to convert AD video to multiple resolutions """

    class Context(sdk2.Task.Context):
        childs = []
        ffmpeg_cmds = []

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('m3u8', '0.4.0', use_wheel=True),
            PipEnvironment('boto3', '1.9.202'),
            PipEnvironment('pycountry', '1.10', use_wheel=True)
        ]

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 1
        # custom parameters
        url = sdk2.parameters.Url("Source video url", required=True)
        webhook_payload_url = sdk2.parameters.String("Webhook Payload URL")
        strm_embed_ready = sdk2.parameters.Bool("Strm embed ready", default=False)
        with strm_embed_ready.value[True]:
            with sdk2.parameters.String("Output audio codec name") as ffmpeg_output_audio_codec:
                ffmpeg_output_audio_codec.values.aac = ffmpeg_output_audio_codec.Value('aac', default=True)
                ffmpeg_output_audio_codec.values.libmp3lame = ffmpeg_output_audio_codec.Value('libmp3lame')
                ffmpeg_output_audio_codec.values['aac,libmp3lame'] = ffmpeg_output_audio_codec.Value('aac,libmp3lame')
                # ffmpeg_output_audio_sample_rate = sdk2.parameters.Integer(
                # "Output audio sample rate", default=0, choices=[0, 44100, 48000])
        keep_aspect_ratio = sdk2.parameters.Bool("Keep aspect ratio for original resolution", default=False)
        has_superresed = sdk2.parameters.Bool("Has superresed", default=False)
        reverse_resolutions_sorting = sdk2.parameters.Bool("Reverse resolutions sorting", default=False)
        skip_audio_track_requirements = sdk2.parameters.Bool("Skip audio track requirements", default=False)
        silence = sdk2.parameters.Bool("Add silence if there is no audio stream", default=False)
        _container = sdk2.parameters.Container(
            "Container", default=None, required=False,
            # type=sandbox.LXC_CONTAINER  # FIXME: invalid argument (SANDBOX-6404)
        )
        subtitles = sdk2.parameters.String("Subtitles json")
        nda = sdk2.parameters.Bool("NDA output stream url", default=False)
        outstaff = sdk2.parameters.Bool("Outstaff output stream url", default=False)
        logo = sdk2.parameters.Bool("Lay amediateka logo", default=False)
        sb_parallel_encoding = sdk2.parameters.Bool("Use SB parallel encoding segments", default=False)
        force_duration_calculation = sdk2.parameters.Bool("Manually calculate input duration", default=False)
        with sb_parallel_encoding.value[True]:
            parallel_segment_duration = sdk2.parameters.Integer("segment duration", default=10)
            parallel_parent = sdk2.parameters.Dict("Parent context")
        with sdk2.parameters.String("OTT audio codec name") as ott_audio_codec:
            ott_audio_codec.values.aac = ott_audio_codec.Value('aac', default=True)
            ott_audio_codec.values.ac3 = ott_audio_codec.Value('ac3')
            ott_audio_codec.values.eac3 = ott_audio_codec.Value('eac3')
        with sdk2.parameters.Group('ffpmeg encoding settings') as ffpmeg_group:
            with sdk2.parameters.String("Output format") as ffmpeg_output_format:
                ffmpeg_output_format.values.hls = ffmpeg_output_format.Value('hls', default=True)
                ffmpeg_output_format.values.mp4 = ffmpeg_output_format.Value('mp4')
                ffmpeg_output_format.values.flv = ffmpeg_output_format.Value('flv')
                ffmpeg_output_format.values.webm = ffmpeg_output_format.Value('webm')
                ffmpeg_output_format.values.ogv = ffmpeg_output_format.Value('ogv')
                ffmpeg_output_format.values.mss = ffmpeg_output_format.Value('mss')
                ffmpeg_output_format.values['hls,mp4'] = ffmpeg_output_format.Value('hls,mp4')
                ffmpeg_output_format.values['mp4,webm'] = ffmpeg_output_format.Value('mp4,webm')
                ffmpeg_output_format.values['hls,mp4,webm,ogv'] = ffmpeg_output_format.Value('hls,mp4,webm,ogv')
                ffmpeg_output_format.values['hls,mp4,webm,flv'] = ffmpeg_output_format.Value('hls,mp4,webm,flv')
                ffmpeg_segment_time = sdk2.parameters.Integer("Segment length in seconds", default=10)

            with sdk2.parameters.String("Preset encoding speed to compression ratio") as ffpmeg_encoding_preset:
                ffpmeg_encoding_preset.values.ultrafast = 'ultrafast'
                ffpmeg_encoding_preset.values.superfast = 'superfast'
                ffpmeg_encoding_preset.values.veryfast = 'veryfast'
                ffpmeg_encoding_preset.values.faster = 'faster'
                ffpmeg_encoding_preset.values.fast = 'fast'
                ffpmeg_encoding_preset.values.medium = 'medium'
                ffpmeg_encoding_preset.values.slow = 'slow'
                ffpmeg_encoding_preset.values.slower = 'slower'
                ffpmeg_encoding_preset.values.veryslow = ffpmeg_encoding_preset.Value('veryslow', default=True)

            ffmpeg_two_pass = sdk2.parameters.Bool("Use two-pass", default=False)
            ffmpeg_framerate = sdk2.parameters.Integer("Framerate", default=0)
            ffmpeg_no_padding = sdk2.parameters.Bool("No padding", default=False)
            ffpmeg_first_resolution = sdk2.parameters.Bool("Fast get lower resolution or thumbnails", default=False)
            ffpmeg_thumbnails_scene = sdk2.parameters.Bool("Capturing scene change thumbnails", default=False)
            ffmpeg_timeline_tiles = sdk2.parameters.Bool("Capture timeline preview", default=False)
            ffmpeg_setpts_pts_startpts = sdk2.parameters.Bool("Change timestamps to start from 0", default=False)

            with ffmpeg_timeline_tiles.value[True]:
                ffmpeg_timeline_tiles_dash = sdk2.parameters.Bool("Add tiles info to DASH manifest", default=False)

            with sdk2.parameters.String("Timeline preview version number") as ffmpeg_timeline_tiles_version:
                ffmpeg_timeline_tiles_version.values.v0 = ffmpeg_timeline_tiles_version.Value('v0', default=True)
                ffmpeg_timeline_tiles_version.values.v1 = ffmpeg_timeline_tiles_version.Value('v1')
                ffmpeg_timeline_tiles_version.values.v2 = ffmpeg_timeline_tiles_version.Value('v2')

            with ffpmeg_thumbnails_scene.value[True]:
                ffmpeg_thumbnails_scene_threshold = sdk2.parameters.Float("Indicate a new scene", default=0.5)
            ffpmeg_thumbnails_t = sdk2.parameters.Float("Capturing scene in second", default=0)
            ffpmeg_loudnorm = sdk2.parameters.Bool("Loudness Normalization", default=False)
            ffmpeg_loudnorm = sdk2.parameters.Bool("Loudness Normalization", default=False)
            ffpmeg_capture_start = sdk2.parameters.Float("Capture from second", default=0)
            ffpmeg_capture_stop = sdk2.parameters.Float("Capture to second", default=0)
            ffpmeg_stream_loop = sdk2.parameters.Integer("Number of times input stream shall be looped", default=0)
            ffpmeg_crossfade_duration = sdk2.parameters.Integer("Fade duration videos for seamless loops", default=0)
            ffmpeg_crop_black_area = sdk2.parameters.Bool('Crop black area', default=False)
            ffpmeg_resolutions = sdk2.parameters.String('Custom output resolutions json')
            ffmpeg_fullrange_color = sdk2.parameters.Bool('Full range RGB color', default=False)

        with sdk2.parameters.Group('DRM packaging') as drm_group:
            production = sdk2.parameters.Bool("Production", default=True)
            drm_widevine = sdk2.parameters.Bool("Google Widevine", default=False)
            drm_playready = sdk2.parameters.Bool("Microsoft PlayReady", default=False)
            drm_fairplay = sdk2.parameters.Bool("Apple FairPlay", default=False)
            rt_parallel_encoding = sdk2.parameters.Bool("Use rt for mp4 encoding", default=False)
            package_clear_hls_as_well = sdk2.parameters.Bool("Package clear HLS as well", default=False)
            need_preroll = sdk2.parameters.Bool("Add kinopoisk preroll", default=False)
            with sdk2.parameters.Group('DRM specific parameters. Fill only if at least one drm is checked'):
                ott_content_uuid = sdk2.parameters.String("Content UUID", required=True)
                ott_product = sdk2.parameters.String("Products to create [hd, sd, all] DEPRECATED")
                reuse_old_keys = sdk2.parameters.Bool("If possible use already generated keys DEPRECATED", default=False)
            with drm_widevine.value[True]:
                drm_widevine_header_provider = sdk2.parameters.String("Widevine provider", default="widevine_test")

        with sdk2.parameters.Group('S3 storage settings') as s3_group:
            s3_bucket = sdk2.parameters.String("S3 bucket name", required=True)
            s3_internal_bucket = sdk2.parameters.String("S3 internal bucket name", default='ott-internal')
            s3_key_prefix = sdk2.parameters.String("S3 keys prefix", required=True)
            s3_dir = sdk2.parameters.String("S3 dir inside bucket", required=True)
            s3_testing = sdk2.parameters.Bool("Use testing S3 installation", default=False)
            s3_metadata = sdk2.parameters.Dict("S3 set metadata")
        with sdk2.parameters.Output:
            stream = sdk2.parameters.String("stream", required=False)
            stream_url = sdk2.parameters.String("stream url", required=False)
            has_sound = sdk2.parameters.Bool("has sound", required=False)
            formats = sdk2.parameters.JSON("converted video formats", required=False)
            vast_formats = sdk2.parameters.String("converted video vast formats",
                                             required=False)

            subtitle_package = sdk2.parameters.String("subtitle package json", required=False)

            thumbnail_urls = sdk2.parameters.String('thumbnail urls json', required=False)
            timeline_tiles_urls = sdk2.parameters.String('timeline tiles urls json', required=False)
            drm_package = sdk2.parameters.JSON('drm package json', required=False)
            clear_hls_package = sdk2.parameters.JSON('clear hls package json', required=False)
            fairplay_package = sdk2.parameters.String('fairplay drm package json', required=False)
            md5 = sdk2.parameters.String("md5 source video")
            video_descriptor_id = sdk2.parameters.Integer("Video descriptor id")

            ffprobe_input_result = sdk2.parameters.JSON("ffprobe source video")
            resource_id = sdk2.parameters.Resource("Segment resoource id")

    class DrmSecretFormatter(object):
        __patterns = (
            ('(--encryption-key ).*( --)', r'\1***\2'),
            ('(--key ).*( --)', r'\1***\2')
            )

        def __init__(self, orig_formatter):
            self.orig_formatter = orig_formatter

        def format(self, record):
            msg = self.orig_formatter.format(record)
            for pattern, repl in self.__patterns:
                msg = re.sub(pattern, repl, msg)
            return msg

        def __getattr__(self, attr):
            return getattr(self.orig_formatter, attr)

    class S3File(object):
        def __init__(self, file_name, s3_dir, s3_bucket, base_url):
            self.file_name = file_name
            self.s3_dir = s3_dir
            self.s3_bucket = s3_bucket
            self.bucket_url = os.path.join(base_url, s3_bucket)

        @property
        def relative_path(self):
            return get_relative_path(self.file_name)

        @property
        def s3_path(self):
            return os.path.join(self.s3_dir, self.relative_path)

        @property
        def url(self):
            return os.path.join(self.bucket_url, self.s3_path)

        def __cmp__(self, other):
            if not isinstance(other, self.__class__):
                return NotImplemented
            return cmp(self.file_name, other.file_name)

        def __repr__(self):
            return 'S3File(name={}, relative_path={}, s3_dir={}, s3_bucket={}, s3_path={}, bucket_url={}, url={})'.format(
                self.file_name, self.relative_path, self.s3_dir, self.s3_bucket, self.s3_path, self.bucket_url, self.url
                )

    class TimelineTilesV0(object):
        """
        Image options for timeline preview
        Sprite - image with multiple tiles (f.e. 5x5 tiles)
        Tile -  frame image within sprite
        Screen - separate image in max.resolution for each tile from sprite
        """
        def __init__(self, total_duration, aspect_ratio,
                     temp_dir, key_prefix, add_to_dash,
                     strm_host, s3_url, s3_bucket,
                     sprite_dir, screen_dir,
                     ):

            self.htiles = 5  # horizontal tiles count
            self.vtiles = 5  # vertical tiles count
            self.theight = 90  # tile height
            self.version = 0

            self.sprite_start = 0
            self.screen_start = 0

            self.m3u8_tiles_id = 'com.yandex.video.thumbnail.tiles'
            self.m3u8_uri_id = 'com.yandex.video.thumbnail.uri'
            self.m3u8_duration_id = 'com.yandex.video.thumbnail.duration'
            self.m3u8_resolution_id = 'com.yandex.video.thumbnail.resolution'

            self.strm_host = strm_host
            self.s3_url = s3_url
            self.s3_bucket = s3_bucket
            self.sprite_dir = sprite_dir
            self.screen_dir = screen_dir
            self.key_prefix = key_prefix
            self.add_to_dash = add_to_dash
            self.twidth = int(math.ceil(aspect_ratio * self.theight))  # always round up width to integer pixel number
            self.tduration = self.get_tiles_step(total_duration)  # interval between tiles in seconds

            # last number of sprite files
            self.sprite_end = self.get_end_number(total_duration, self.sprite_duration, self.sprite_start)
            # last number of screen files
            self.screen_end = self.get_end_number(total_duration, self.tduration, self.screen_start)

            self.absolute_sprite_prefix = os.path.join(temp_dir, self.sprite_prefix)
            self.absolute_screen_prefix = os.path.join(temp_dir, self.screen_prefix)

        @property
        def sprite_prefix(self):
            return '{}sprite'.format(self.key_prefix)

        @property
        def screen_prefix(self):
            return '{}screen'.format(self.key_prefix)

        @property
        def sprite_url_template(self):
            s3 = StrmVideoConvert.S3File(
                file_name='{}$Number$.jpg'.format(self.sprite_prefix),
                s3_dir=self.sprite_dir,
                s3_bucket=self.s3_bucket,
                base_url=self.strm_host,
            )
            return s3.url

        @property
        def screen_url_template(self):
            s3 = StrmVideoConvert.S3File(
                file_name='{}$Number$.jpg'.format(self.screen_prefix),
                s3_dir=self.screen_dir,
                s3_bucket=self.s3_bucket,
                base_url=self.strm_host,
            )
            return s3.url

        def get_sprite_s3(self, file_name):
            return StrmVideoConvert.S3File(
                file_name=file_name,
                s3_dir=self.sprite_dir,
                s3_bucket=self.s3_bucket,
                base_url=self.s3_url,
            )

        def get_screen_s3(self, file_name):
            return StrmVideoConvert.S3File(
                file_name=file_name,
                s3_dir=self.screen_dir,
                s3_bucket=self.s3_bucket,
                base_url=self.s3_url,
            )

        @staticmethod
        def get_tiles_step(duration):
            """
            How often (in seconds) make screenshot for timeline tiles
            """
            if duration < 5 * 60:
                return 2
            elif duration < 15 * 60:
                return 5
            else:
                return 10

        @property
        def tiles(self):
            """
            return: sprite size in string as "{htiles}x{vtiles}"
            """
            return '{}x{}'.format(self.htiles, self.vtiles)

        @property
        def sprite_duration(self):
            """
            Total duration for all tiles in one sprite image
            """
            return self.htiles * self.vtiles * self.tduration

        @property
        def sprite_height(self):
            return self.vtiles * self.theight

        @property
        def sprite_width(self):
            return self.htiles * self.twidth

        @property
        def sprite_resolution(self):
            return '{}x{}'.format(self.sprite_width, self.sprite_height)

        @staticmethod
        def get_end_number(total_duration, step, start):
            return int(math.floor(total_duration / step)) + start

        @property
        def mpd_session_data(self):
            """
            Данные для параметра --thumbnails <data> утилиты mp4dash
            """
            return 'tile={tile},uri={uri},start={start},duration={duration},width={width},height={height}'.format(
                tile=self.tiles,
                uri=self.sprite_url_template,
                start=self.sprite_start,
                duration=self.sprite_duration,
                width=self.sprite_width,
                height=self.sprite_height,
            )

        def get_meta_json(self):
            return {
                'tduration': self.tduration,
                'sprite': self.tiles,
                'sprite_uri': self.sprite_url_template,
                'sprite_start': self.sprite_start,
                'sprite_end': self.sprite_end,
                'screen_uri': self.screen_url_template,
                'screen_start': self.screen_start,
                'screen_end': self.screen_end,
                'sprite_duration': self.sprite_duration,
                'sprite_width': self.sprite_width,
                'sprite_height': self.sprite_height,
                'hls': True,
                'dash': self.add_to_dash,
            }

    class TimelineTilesV1(object):
        """
        New version for preview https://st.yandex-team.ru/PLAYERWEB-888
        Image options for timeline preview
        Sprite - image with multiple tiles (f.e. 5x5 tiles)
        Tile -  frame image within sprite
        Screen - separate image in max.resolution for each tile from sprite
        """
        MAX_VTILES_LOWRES = 1    # max number columns for lowres sprite
        MAX_HTILES_LOWRES = 200  # max number rows for lowres sprite

        MAX_VTILES = 1    # max vertical tiles for hires sprites sprite
        MAX_HTILES = 50    # max horizontal tiles for hires sprite

        MIN_TILE_SIDE = 90  # min side of tile (width or height depends from aspect ration)

        class Sprite:
            def __init__(self, htiles, vtiles, theight, twidth, tduration, scale_factor, url_template):
                self.htiles = htiles
                self.vtiles = vtiles
                self.scale_factor = scale_factor
                self.theight = theight * scale_factor
                self.twidth = twidth * scale_factor
                self.tduration = tduration
                self.start_number = 0
                self.url_template = url_template

            @property
            def width(self):
                return self.vtiles * self.twidth / self.scale_factor

            @property
            def height(self):
                return self.htiles * self.theight / self.scale_factor

            @property
            def resolution(self):
                return '{}x{}'.format(self.width, self.height)

            @property
            def duration(self):
                """
                Total duration for all tiles in one sprite image
                """
                return self.htiles * self.vtiles * self.tduration

            @property
            def ffprobe_tiles(self):
                return '{}x{}'.format(self.vtiles, self.htiles)

            @property
            def playlist_tiles(self):
                return '{}x{}'.format(self.htiles, self.vtiles)

        def __init__(self, total_duration, aspect_ratio,
                     temp_dir, key_prefix, add_to_dash,
                     strm_host, s3_url, s3_bucket,
                     sprite_dir, screen_dir, fps, version,
                     ):

            self.sprite_start = 0
            self.screen_start = 0
            self.strm_host = strm_host
            self.s3_url = s3_url
            self.s3_bucket = s3_bucket
            self.sprite_dir = sprite_dir
            self.screen_dir = screen_dir
            self.key_prefix = key_prefix
            self.add_to_dash = add_to_dash
            self.version = version

            self.fps = fps
            self.total_duration = total_duration
            self.tduration = self.get_tiles_step(total_duration)  # interval between tiles in seconds

            htiles_lowres = self.MAX_HTILES_LOWRES
            vtiles_lowres = self.MAX_VTILES_LOWRES

            htiles = self.MAX_HTILES
            vtiles = self.MAX_VTILES

            if total_duration < 1000:
                hduration_lowres = self.tduration * self.MAX_VTILES_LOWRES
                htiles_lowres = int(math.ceil(total_duration / hduration_lowres))

                hduration = self.tduration * self.MAX_VTILES
                htiles = int(math.ceil(total_duration / hduration))

            if aspect_ratio > 1:
                theight_lowres = self.MIN_TILE_SIDE  # tile height resolution in pixels for lowres sprite
                twidth_lowres = int(math.ceil(aspect_ratio * theight_lowres))  # always round up width to integer pixel number
            else:
                twidth_lowres = self.MIN_TILE_SIDE  # tile height resolution in pixels for lowres sprite
                theight_lowres = int(math.ceil(twidth_lowres / aspect_ratio))  # always round up width to integer pixel number

            self.sprite_lowres = self.Sprite(
                htiles=htiles_lowres,
                vtiles=vtiles_lowres,
                theight=theight_lowres,
                twidth=twidth_lowres,
                tduration=self.tduration,
                url_template=self.get_sprite_url_template(self.sprite_lowres_prefix),
                scale_factor=1,
            )

            self.sprite_hires = self.Sprite(
                htiles=htiles,
                vtiles=vtiles,
                theight=theight_lowres,
                twidth=twidth_lowres,
                tduration=self.tduration,
                url_template=self.get_sprite_url_template(self.sprite_prefix),
                scale_factor=2,  # means that hires will be X times bigger than its resolution in metadata
            )

            # last number of sprite files
            self.sprite_end = self.get_end_number(total_duration, self.sprite_hires.duration, self.sprite_start)
            self.absolute_sprite_prefix = os.path.join(temp_dir, self.sprite_prefix)
            self.absolute_sprite_lowres_prefix = os.path.join(temp_dir, self.sprite_lowres_prefix)

        @property
        def ffprobe_select_filter(self):
            interval_fps = self.fps * self.tduration
            total_tiles_low = self.sprite_lowres.vtiles * self.sprite_lowres.htiles
            # # check for rounding error - do not miss the last shot
            # if total_frames // interval_fps < total_tiles_low:
            #     interval_fps -= 1

            last_frame = interval_fps * total_tiles_low  # skip the last frame
            return 'select=not(mod(n\,{}))*not(eq(n\,{})),setpts=N/({}*TB)'.format(interval_fps, last_frame, self.fps)

        @property
        def sprite_prefix(self):
            return '{}sprite'.format(self.key_prefix)

        @property
        def sprite_lowres_prefix(self):
            return '{}spritelow'.format(self.key_prefix)

        @property
        def sprite_url_template(self):
            s3 = StrmVideoConvert.S3File(
                file_name='{}$Number$.jpg'.format(self.sprite_prefix),
                s3_dir=self.sprite_dir,
                s3_bucket=self.s3_bucket,
                base_url=self.strm_host,
            )
            return s3.url

        def get_sprite_url_template(self, sprite_prefix):
            s3 = StrmVideoConvert.S3File(
                file_name='{}$Number$.jpg'.format(sprite_prefix),
                s3_dir=self.sprite_dir,
                s3_bucket=self.s3_bucket,
                base_url=self.strm_host,
            )

            return s3.url

        @property
        def sprite_lowres_url_template(self):
            s3 = StrmVideoConvert.S3File(
                file_name='{}$Number$.jpg'.format(self.sprite_lowres_prefix),
                s3_dir=self.sprite_dir,
                s3_bucket=self.s3_bucket,
                base_url=self.strm_host,
            )
            return s3.url

        def get_sprite_s3(self, file_name):
            return StrmVideoConvert.S3File(
                file_name=file_name,
                s3_dir=self.sprite_dir,
                s3_bucket=self.s3_bucket,
                base_url=self.s3_url,
            )

        def get_tiles_step(self, duration):
            """
            How often (in seconds) make screenshot for timeline tiles
            """
            if duration < 5 * 60:
                return 2
            elif duration < 1000:
                return 5

            # Dynamic step for max allowed preview size
            time_step = duration / (self.MAX_VTILES_LOWRES * self.MAX_HTILES_LOWRES)
            fps_step = round(self.fps * time_step)
            time_step = fps_step / self.fps
            return time_step

        @staticmethod
        def get_end_number(total_duration, step, start):
            return int(math.floor(total_duration / step)) + start

        def get_hls_session_infos(self):
            """
            Вернуть список EXT-X-SESSION-DATA тегов для HLS
            :return: [[code, value] ...]
            """
            return [
                ['com.yandex.video.thumbnail.version', str(self.version)],
                # lowres sprite:
                ['com.yandex.video.thumbnail.lowres.uri', self.sprite_lowres.url_template],
                ['com.yandex.video.thumbnail.lowres.tiles', self.sprite_lowres.playlist_tiles],
                ['com.yandex.video.thumbnail.lowres.duration', self.sprite_lowres.duration],
                ['com.yandex.video.thumbnail.lowres.resolution', self.sprite_lowres.resolution],
                # hires sprite:
                ['com.yandex.video.thumbnail.uri', self.sprite_hires.url_template],
                ['com.yandex.video.thumbnail.tiles', self.sprite_hires.playlist_tiles],
                ['com.yandex.video.thumbnail.duration', self.sprite_hires.duration],
                ['com.yandex.video.thumbnail.resolution', self.sprite_hires.resolution],
            ]

        def _get_mpd_session_data(self, sprite):
            return 'tile={tile},uri={uri},start={start},duration={duration},width={width},height={height},version={version}'.format(
                tile=sprite.playlist_tiles,
                uri=sprite.url_template,
                start=sprite.start_number,
                duration=sprite.duration,
                width=sprite.width,
                height=sprite.height,
                version=str(self.version),
            )

        @property
        def mpd_session_data(self):
            """
            Данные для параметра --thumbnails-lowres <data> утилиты mp4dash
            """
            return self._get_mpd_session_data(self.sprite_hires)

        @property
        def mpd_session_data_lowres(self):
            """
            Данные для параметра --thumbnails <data> утилиты mp4dash
            """
            return self._get_mpd_session_data(self.sprite_lowres)

        def get_meta_json(self):
            return {
                'tduration': self.sprite_lowres.tduration,
                'sprite': self.sprite_hires.playlist_tiles,
                'sprite_lowres': self.sprite_lowres.playlist_tiles,
                'sprite_uri': self.sprite_hires.url_template,
                'sprite_lowres_uri': self.sprite_lowres.url_template,
                'sprite_start': self.sprite_hires.start_number,
                'sprite_end': self.sprite_end,
                'sprite_duration': self.sprite_hires.duration,
                'sprite_lowres_duration': self.sprite_lowres.duration,
                'sprite_width': self.sprite_hires.width,
                'sprite_height': self.sprite_hires.height,
                'sprite_lowres_width': self.sprite_lowres.width,
                'sprite_lowres_height': self.sprite_lowres.height,
                'hls': True,
                'dash': self.add_to_dash,
            }

    def is_ott_hls(self):
        """Return is current task for non drm ott video"""
        return (self.is_owner('OTT') or self.is_owner('KP')) and \
               self.Parameters.ffmpeg_output_format == 'hls' and \
               not self.is_drm_packaging() and \
               not self.is_mss_packaging() and \
               not self.is_ott_trailer()

    def is_mss_packaging(self):
        return self.Parameters.ffmpeg_output_format == 'mss'

    def get_output_format(self):
        if self.is_ott_hls() or self.is_drm_packaging() or self.is_mss_packaging():
            return "mp4"
        else:
            return self.Parameters.ffmpeg_output_format

    def get_secret_owner(self):
        if 'OTT' in self.owner:
            return 'OTT'
        elif self.owner.endswith('-LOW'):
            return self.owner[0:-4]
        return self.owner

    def is_youtube_url(self):
        for domain in youtube_dl_domains:
            if self.Parameters.url.split('/', 3)[2].endswith(domain):
                return True
        return False

    def is_youtube_dl_url_sni_required(self):
        for domain in youtube_dl_domains_sni_required:
            if self.Parameters.url.split('/', 3)[2].endswith(domain):
                return True
        return False

    def is_owner(self, owner):
        return (
                self.owner == owner or
                self.owner == owner + '-LOW' or
                self.owner == owner + '-TESTING'
        )

    def is_ott_trailer(self):
        return self.is_owner('OTT') and "trailer" in self.Parameters.url

    def need_preroll(self):
        # temporary always include preroll for music clips
        return self.is_owner('MUSIC') or self.Parameters.need_preroll

    def preroll_audio_transcoded_name(self):
        return "preroll_audio.mp4"

    def preroll_file(self):
        if self.is_owner('MUSIC'):
            if self.resolution_ratio_key == '169':
                return 'https://s3.mds.yandex.net/aarkhincheev/test/AIn1_new4.mp4'
            else:
                return 'https://aarkhincheev.s3.mds.yandex.net/test/AIn1_4x3_new.mp4'
        if self.resolution_ratio_key == '169':
            return 'https://aarkhincheev.s3.mds.yandex.net/test/KP_Broadcast_v7_16x9.mp4'
        else:
            return 'https://aarkhincheev.s3.mds.yandex.net/test/KP_Broadcast_v7_4x3.mp4'

    def amediateka_logo(self):
        logo_temp_path = tempfile.NamedTemporaryFile(suffix='.png').name
        self.download_wget(
                'https://s3.mds.yandex.net/ott/SANDBOX/amedia_logo.png',
                logo_temp_path
                )
        return logo_temp_path

    def need_loudnorm(self):
        return self.audio_stream and (
            self.Parameters.ffmpeg_loudnorm or
            self.Parameters.ffpmeg_loudnorm
        )

    def get_resolutions(self):
        if self.Parameters.ffpmeg_resolutions and self.Parameters.ffpmeg_resolutions != '[]':
            return json.loads(self.Parameters.ffpmeg_resolutions)
        elif self.is_drm_packaging() or self.is_owner('OTT'):
            return ott_resolutions
        elif self.is_owner('BG-STORE'):
            return browser_resolutions
        elif self.is_owner('BANNERSTORAGE'):
            return bannerstorage_resolutions
        elif self.is_strm_embed():
            return self.get_embed_resolutions()
        elif self.is_strm_hls_url() and self.Parameters.keep_aspect_ratio:
            return self.get_resolutions_base169()
        else:
            return resolutions

    def get_resolutions_base169(self):
        resolutions = [("{}{}_{}p".format(ratio[0], ratio[1], height),
                      (int((height*ratio[0]/ratio[1]) // 2) * 2, height,
                       self.get_embed_video_bitrate(height),
                       self.get_embed_audio_bitrate(height)))
                        for height in embed_base_169_heights
                            for ratio in embed_aspect_ratios if not (height > 720 and ratio != (16, 9))]
        return tuple(resolutions)

    def get_embed_resolutions(self):
        resolutions = [("{}{}_{}p".format(ratio[0], ratio[1], height),
                      (int((height*ratio[0]/ratio[1]) // 2) * 2, height))
                        for height in embed_base_169_heights
                            for ratio in embed_aspect_ratios if not (height > 720 and ratio != (16, 9))]
        resolutions.extend(embed_input_resolutions)
        return tuple(resolutions)

    def get_target_frame_rate(self):
        return self.Parameters.ffmpeg_framerate if self.Parameters.ffmpeg_framerate > 0 else self.target_frame_rate

    def get_target_keyint(self):
        return int(self.get_gop_size()*self.target_frame_rate)

    def get_gop_size(self):
        return 2

    def get_segment_time(self):
        if self.is_ott_hls() or self.Parameters.rt_parallel_encoding:
            return 2
        if self.is_drm_packaging() or self.is_mss_packaging():
            return 4
        return self.Parameters.ffmpeg_segment_time

    def is_drm_packaging(self):
        return self.Parameters.drm_widevine or self.Parameters.drm_playready or self.Parameters.drm_fairplay

    def send_status_hook(self, status, timeout=1):
        if self.Parameters.webhook_payload_url:
            data = {'task_id': self.id}
            if isinstance(status, dict):
                if 'stream_url' in status:
                    data.update({'status': 'converted'})
                elif 'error_message' in status:
                    data.update({'status': 'error'})
                elif 'warning_message' in status:
                    data.update({'status': 'warning'})
                data.update(status)
            else:
                data.update({'status': status})
            try:
                s = requests.Session()
                logging.debug('send_status_hook to %s with %s',
                              self.Parameters.webhook_payload_url, data)
                s.mount('http://', requests.adapters.HTTPAdapter(max_retries=3))
                s.post(self.Parameters.webhook_payload_url, json=data, timeout=timeout, verify=False)
            except Exception:
                logging.exception('send_status_hook(): failed send post reqest to %s',
                                  self.Parameters.webhook_payload_url)

    def sandbox_path(self, file_name):
        return os.path.join(self.temp_dir, file_name)

    def s3_sub_dir(self):
        content_id = self.get_drm_content_id()
        if content_id:
            return '{}-{}'.format(self.id, self.get_drm_content_id())
        return str(self.id)

    def hls_sb_path(self, file_name=None):
        base = os.path.join(
                self.temp_dir,
                'packaged',
                self.s3_sub_dir()
                )
        if file_name is not None:
            return os.path.join(base, file_name)
        return base

    def get_all_ts_files(self, m3u8_files):
        import m3u8
        ts_files = []
        for m3u8_file in m3u8_files:
            m3u8_obj = m3u8.load("{}/{}".format(self.temp_dir, m3u8_file))
            ts_files.extend(m3u8_obj.files)
        return ts_files

    def get_ts_files(self, m3u8_file):
        import m3u8
        return m3u8.load("{}/{}".format(self.temp_dir, m3u8_file)).files

    def s3_connect(self, s3_endpoint_url, s3_secret_key_id, s3_secret_key, s3_bucket):
        import boto3
        from boto3.s3.transfer import S3Transfer
        from botocore.exceptions import ClientError
        s3conn = boto3.client('s3', endpoint_url=s3_endpoint_url, aws_access_key_id=s3_secret_key_id,
                         aws_secret_access_key=s3_secret_key)
        try:
            s3conn.head_bucket(Bucket=s3_bucket)
        except ClientError as e:
            error_code = int(e.response['Error']['Code'])
            if error_code == 404:
                s3conn.create_bucket(Bucket=s3_bucket)
            else:
                msg = 's3 get bucket code {}'.format(error_code)
                self.send_status_hook({'warning_message': msg, 'raw_error': e.response['Error']['Message']})
                raise TemporaryError(msg)
        return S3Transfer(s3conn)
        logging.info("S3 connected")

    def get_content_type(self, file_name):
        file_extension = file_name.split('.')[-1]
        if file_extension in content_types:
            return content_types[file_extension]
        else:
            if self.src_mime_type:
                return self.src_mime_type
            else:
                return "video/mp4"

    def get_drm_widevine_content_id(self):
        return ''.join(x.encode('hex') for x in self.get_drm_content_id())

    def get_drm_content_uuid(self):
        return str(UUID(self.get_drm_content_id()))

    def get_drm_content_id(self):
        if self.Parameters.ott_content_uuid is not None:
            return self.Parameters.ott_content_uuid.replace('-', '').lower()
        return ''

    def get_drm_package_path(self):
        if self.get_drm_content_id():
            return str(self.id) + '/' + self.get_drm_content_uuid() + '.ism/'
        return str(self.id) + '/'

    def drm_path(self, file_path):
        relative_path = get_relative_path(file_path)
        return ''.join((self.base_url, self.get_drm_package_path(), relative_path))

    def get_h264_profile(self, height):
        if self.is_ott_hls() or self.is_drm_packaging() or self.is_mss_packaging():
            if height >= 576:
                return 'high'
            else:
                return 'main'
        return 'baseline'

    def get_embed_h264_level(self, height):
        if height < 480:
            return '3.0'
        elif height < 720:
            return '3.1'
        elif height < 720:
            return '3.1'
        elif height <= 720:
            return '4.1'
        elif height <= 1080:
            return '4.2'
        elif height <= 1440:
            return '5.1'
        elif height <= 2160:
            return '5.2'
        else:
            # Todo 4k minimal level and profile VH-5694
            return '-1'

    def get_h264_level(self, height, width):
        fps = self.get_target_frame_rate()
        pixels_per_frame = width * height
        if (pixels_per_frame <= 720 * 480 and fps <= 30) or (pixels_per_frame <= 720 * 576 and fps <= 25):
            return '3.0'
        elif pixels_per_frame <= 1280 * 720 and fps <= 30:
            return '3.1'
        elif pixels_per_frame <= 1280 * 1024:
            return '3.2'
        elif pixels_per_frame <= 1920 * 1080 and fps <= 30:
            return '4.0'
        elif pixels_per_frame <= 1920 * 1080 and fps > 30:
            return '4.2'
        else:
            return '-1'

    def get_embed_video_bitrate(self, height):
        base_bitrate = 8192
        if height <= 240:
            bitrate = base_bitrate * 1 / 16
        elif height <= 360:
            bitrate = base_bitrate * 3 / 16
        elif height <= 480:
            bitrate = base_bitrate * 4 / 16
        elif height <= 540:
            bitrate = base_bitrate * 5 / 16
        elif height <= 576:
            bitrate = base_bitrate * 6 / 16
        elif height <= 720:
            bitrate = base_bitrate * 12 / 16
        elif height <= 1080:
            bitrate = base_bitrate * 16 / 16
        elif height <= 1440:
            bitrate = base_bitrate * 20 / 16
        elif height <= 2160:
            bitrate = base_bitrate * 40 / 16
        else:
            bitrate = base_bitrate * 40 / 16
        return bitrate

    def get_embed_audio_bitrate(self, height):
            #if self.audio_stream and "max_bit_rate" in self.audio_stream \
            #        and int(self.audio_stream["max_bit_rate"]) >= 192000:
            #    bitrate = 192
        bitrate = 128 # STRM-1267 temp measure for mpeg-dash using lowest bitrate chunks for audio
        return bitrate

    def get_subtitles(self):
        if not self.Parameters.subtitles:
            return []
        subs_input = self.Parameters.subtitles

        subs = json.loads(subs_input)

        result = []
        sub_offset = self.get_ffprobe_duration(self.preroll_file()) if self.need_preroll() else 0
        for sub in sorted(subs, key=lambda x: x['id']):
            subs_temp_path = tempfile.NamedTemporaryFile().name
            self.download_wget(sub['url'], subs_temp_path)

            encoding = self.get_file_mime_encode(subs_temp_path)
            if encoding != 'utf-8':
                self.fail_task('Subtitles encoding not utf-8 (but {})'.format(encoding))

            path = self.sandbox_path(self.get_sub_name(sub, 'vtt'))
            self.convert_to_webttv(subs_temp_path, path, sub_offset)

            lang = ott_sub_lang_dict.get(
                (self.normalize_tag(sub['language']), self.normalize_tag(sub['title']))
            )
            lang_name = ott_sub_lang_name_dict.get(lang)

            if lang is None or lang_name is None:
                self.fail_task('Subtitles language is unknown (language={}, title={})'.format(
                    sub['language'], sub['title']
                ))

            result.append({
                    'id': sub['id'],
                    'title': lang_name,
                    'language': lang,
                    'url': self.get_sub_name(sub, 'vtt'),
                    'original_url': sub['url']
                })
        return result

    def get_hls_subtitles(self, subtitles, out_dir):
        result = []
        for sub in subtitles:
            self.run_shell('cp -n {} {}'.format(
                    self.sandbox_path(sub['url']),
                    os.path.join(out_dir, sub['url']))
                    )
            playlist_path = os.path.join(out_dir, self.get_sub_name(sub, 'm3u8'))
            logging.info('Create sub={}, path={}, url={}'.format(
                    sub,
                    playlist_path,
                    sub['url']
                    ))
            create_subtitles_playlist(
                    sub['url'],
                    self.duration,
                    playlist_path
                    )
            self.run_shell('cat {}'.format(playlist_path))
            result.append({
                    'id': sub['id'],
                    'title': sub['title'],
                    'language': sub['language'],
                    'url': playlist_path
                })
        return result

    def get_sub_name(self, sub, endwith):
        return 'sub-{}-{}-stream.{}'.format(
                sub['language'],
                sub['id'],
                endwith
                )


    def video_descriptor_id(self):
        return self.to_video_descriptor_id(self.get_product())


    def to_video_descriptor_id(self, product):
        if product == 'hd':
            return 2
        elif product == 'sd':
            return 3
        else:
            raise common.errors.TaskFailure("Product {} can't be transformed to video_descriptor_id".format(product))

    def get_mediakeys_url(self):
        if self.Parameters.production:
            return "https://ott-keys-api.ott.yandex.net/v1/keys/"
        else:
            return "https://ott-keys-api.tst.ott.yandex.net/v1/keys/"

    def get_fairplay_s3_path(self, path):
        return self.base_url + get_relative_path(path)

    def old_s3_path(self, name):
        if file_ends_with(name, ('m3u8', 'ts', 'vtt', 'mkv', 'jpg')) and not name.startswith('subtitles/'):
            if self.is_ott_trailer():
                return os.path.join(self.s3_dir, self.s3_sub_dir(), name)
            else:
                return self.s3_dir + name
        elif file_ends_with(name, ('mp4')) and not self.is_drm_packaging() and not self.is_mss_packaging():
            return self.s3_dir + name
        elif self.Parameters.drm_widevine or self.Parameters.drm_playready or self.is_mss_packaging():
            return self.s3_dir + self.get_drm_package_path() + name
        else:
            return self.s3_dir + name

    def upload_file(self, _file):
        metadata = {'task-id': str(self.id)}
        if self.Parameters.s3_metadata:
            metadata.update(self.Parameters.s3_metadata)
        if isinstance(_file, self.S3File):
            file_absolute_path = get_absolute_path(_file.relative_path, self.temp_dir)

            logging.debug(
                    'File upload. File={}, path={}'.format(
                    _file,
                    file_absolute_path
                    ))
            self.s3transfer.upload_file(
                    file_absolute_path,
                    _file.s3_bucket,
                    _file.s3_path,
                    extra_args={
                        "Metadata": metadata,
                        "ContentType": self.get_content_type(_file.file_name)
                        }
                    )
        else:
            if _file in self.file_resolutions:
                resolution = self.file_resolutions[_file]
                metadata['resolution'] = '{}x{}'.format(resolution[0], resolution[1])
            if _file.startswith('embed/'):
                self.embed_s3transfer.upload_file(
                    "{}/{}".format(self.temp_dir, _file), 'strm', _file,
                    extra_args={"Metadata": metadata, "ContentType": content_types[_file.split('.')[-1]]}
                )
            else:
                self.s3transfer.upload_file(
                    "{}/{}".format(self.temp_dir, _file),
                    self.Parameters.s3_bucket, self.s3_dir + _file,
                    extra_args={
                        "Metadata": metadata,
                        "ContentType": self.get_content_type(_file)
                    }
                )

    def upload(self, files):
        files = list(files)
        if len(files) > 0:
            logging.info("Going to upload {} files".format(len(files)))
            logging.debug(files)
            self.pool.map(self.upload_file, files)
            logging.info("Uploaded files {}, {}".format(len(files), files[-1]))
        else:
            logging.info("Empty files to upload")
        return files

    def get_file_mime_encode(self, _file):
        logging.info("Get mime-type " + _file)
        cmd = 'file -b --mime-encoding {} | tr -d "\n"'.format(_file)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
            except sp.CalledProcessError:
                out = 'unknown'
            logging.info(out)
            return out

    def get_file_mime_type(self):
        logging.info("Get mime-type ")
        cmd = 'file -b --mime-type {} | tr -d "\n"'.format(self.src_video)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
            except sp.CalledProcessError:
                out = ''
            self.src_mime_type = out
            logging.info(out)
            if out == 'image/png':
                self.src_video = self.image_loop()

    def get_md5_file(self):
        logging.info("Get md5")
        cmd = 'md5sum {} | grep -o -P "^\w+"'.format(self.src_video)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
            except sp.CalledProcessError:
                out = ''
            self.Parameters.md5 = out
            logging.info(out)

    def convert_to_webttv(self, src_sub, output_file, start_from=0):
        """
        Convert subtitles to webttv and download (if needed)
        """
        cmd = '{} -itsoffset {} -y -i {} -c:s webvtt {}'.format(
                self.ffmpeg_bin,
                start_from,
                src_sub,
                output_file
                )
        self.run_ffmpeg(cmd)

    def get_ffprobe_duration(self, src):
        """
        :type src: path or link to source file
        :return: str, number of duraction in seconds
        """
        logging.info("Get {} duraction".format(src))
        cmd = "{} -v error -show_entries format=duration -of default=noprint_wrappers=1:nokey=1 '{}'".format(
                self.ffprobe_bin,
                src
        )
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
                logging.info(out)
                return str(out).strip()
            except sp.CalledProcessError:
                msg = 'ffprobe exit code {}'.format(sp.STDOUT)
                logging.info('{} stat {}'.format(self.ffprobe_bin, os.stat(self.ffprobe_bin)))
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)

    def get_ffprobe_streams(self, src):
        logging.info("Get streams")
        cmd = "{} -v error -of flat=s=_ -print_format json {} -show_entries stream:format '{}'".format(
                self.ffprobe_bin,
                '-count_frames' if self.src_mime_type == 'image/gif' else '',
                src
        )
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
                logging.info(out)
                return json.loads(out)
            except sp.CalledProcessError:
                msg = 'ffprobe exit code {}'.format(sp.STDOUT)
                logging.info('{} stat {}'.format(self.ffprobe_bin, os.stat(self.ffprobe_bin)))
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)

    def get_ffprobe_video_frames(self):
        logging.info("Get video frames")
        video_frames = None
        cmd = "{} -v error -of flat=s=_ -print_format json -show_frames -select_streams v '{}'".format(
                self.ffprobe_bin,
                self.src_video
        )
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
                video_frames = json.loads(out)
            except sp.CalledProcessError:
                msg = 'ffprobe exit code {}'.format(sp.STDOUT)
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)
            logging.info(out)
        return video_frames

    def get_ffprobe_video_frames_count(self):
        logging.info("Get video frames count")
        video_frames_count = None
        cmd = "{} -v error -of flat=s=_ -print_format json -count_frames -select_streams v -show_entries stream=nb_read_frames '{}'".format(
                self.ffprobe_bin,
                self.src_video
        )
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
                video_frames_count = json.loads(out)
            except sp.CalledProcessError:
                msg = 'ffprobe exit code {}'.format(sp.STDOUT)
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)
            logging.info(out)
        return video_frames_count

    def get_ffmpeg_cropdetect(self):
        # http://www.ffmpeg.org/ffmpeg-filters.html#cropdetect
        logging.info("Get non-black area crop size")
        cmd = "{} -i {} -t 3 -vf cropdetect=24:16:0 -f null - 2>&1|grep cropdetect|tail -1".format(
            self.ffmpeg_bin, self.src_video
        )
        # '[Parsed_cropdetect_0 @ 0x7fb399718800] x1:0 x2:719 y1:52 y2:507 w:720 h:448 x:0 y:56 pts:52000 t:1.040000 crop=720:448:0:56'
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('cmd')) as pl:
            try:
                out = sp.check_output(cmd, shell=True, stderr=pl.stderr)
                logging.info(out)
                crop_search = re.search(' (crop=\d+:\d+:\d+:\d+)$', out)
                return crop_search.group(1) if crop_search and not crop_search.group(1).endswith(':0:0') else ''
            except sp.CalledProcessError:
                return ''

    def get_display_aspect_ratio(self):
        if 'display_aspect_ratio' in self.resolution_src and \
                self.resolution_src["display_aspect_ratio"] != '0:1' and \
                self.resolution_src["display_aspect_ratio"] != 'N/A':
            split_dar = self.resolution_src["display_aspect_ratio"].split(':')
            return float(split_dar[0]) / float(split_dar[1])
        else:
            return float(self.resolution_src["width"]) / float(self.resolution_src["height"])

    def get_resolution(self):
        logging.info("Get video resolution")
        if 'streams' in self.src_streams and 'format' in self.src_streams:
            video_streams = [
                s for s in self.src_streams['streams'] if s['codec_type'] == 'video' and (s['codec_name'] != 'mjpeg' or 'nb_frames' in s)
            ]
            audio_streams = [s for s in self.src_streams['streams'] if s['codec_type'] == 'audio']
            if len(video_streams) > 0:
                stream = video_streams[0]

                nb_read_frames = 0
                if 'nb_read_frames' in stream and int(stream["nb_read_frames"]) > 1:
                    nb_read_frames = float(stream["nb_read_frames"])
                force_duration_calculation = self.Parameters.force_duration_calculation
                if nb_read_frames == 0 and force_duration_calculation:
                    video_frames_count = self.get_ffprobe_video_frames_count()
                    try:
                        nb_read_frames = float(video_frames_count['streams'][0]['nb_read_frames'])
                    except KeyError:
                        pass

                if "duration" in stream and not force_duration_calculation:
                    self.src_duration = float(stream["duration"])
                elif "duration" in self.src_streams["format"] and not force_duration_calculation:
                    self.src_duration = float(self.src_streams["format"]["duration"])
                    stream['duration'] = self.src_duration
                elif 'avg_frame_rate' in stream and nb_read_frames > 1:
                    if stream['avg_frame_rate'] == '0/0':
                        gif_video_frames = self.get_ffprobe_video_frames()
                        if gif_video_frames and 'frames' in gif_video_frames and len(gif_video_frames['frames']) > 0:
                            self.src_duration = sum(
                                [float(f['pkt_duration_time']) for f in gif_video_frames['frames']]
                            )
                        else:
                            self.src_duration = nb_read_frames / float(2)
                    else:
                        num, den = stream["avg_frame_rate"].split('/')
                        self.src_duration = nb_read_frames * float(den) / float(num)
                    stream['duration'] = self.src_duration
                else:
                    msg = 'ffprobe: video format {} or MIME type {} is not supported'.format(
                        self.src_streams["format"]["format_name"], self.src_mime_type)
                    self.send_status_hook({'error_message': msg})
                    raise common.errors.TaskFailure(msg)

                self.new_duration = self.src_duration - self.Parameters.ffpmeg_capture_start

                if len(audio_streams) > 0:
                    self.audio_stream = audio_streams[0]
                stream['has_sound'] = isinstance(self.audio_stream, dict)
                logging.info("Video width: {} height: {}".format(stream["width"], stream["height"]))
            else:
                msg = 'ffprobe: not a video streams'
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)
        else:
            msg = 'ffprobe empty json'
            self.send_status_hook({'error_message': msg})
            raise common.errors.TaskFailure(msg)
        return stream

    def get_video_streams(self, src_streams):
        return filter(lambda x: x['codec_type'] == 'video', src_streams['streams'])

    def get_audio_streams(self, src_streams):
        return filter(lambda x: x['codec_type'] == 'audio', src_streams['streams'])

    def get_und_lang(self):
        """Logic for undefined audio stream"""
        if self.is_owner('OTT') and len(self.get_audio_streams(self.src_streams)) == 1:
            """For single OTT und stream, use ru lang"""
            return 'ru'
        return 'un'

    def fail_task(self, msg):
        self.send_status_hook({'error_message': msg})
        raise common.errors.TaskFailure(msg)

    def ensure_ott_src_streams(self, src_streams):
        audio_streams = self.get_audio_streams(src_streams)
        if not audio_streams:
            self.fail_task('Source file have empty audio streams')
        if not any(self.audio_meta_lang(stream)[0].startswith('ru') for stream in audio_streams):
            self.fail_task('Expected at least 1 rus audio stream, found 0')

    def get_nearest_ratios(self, ratio):
        logging.info("Get nearest ratio key")
        resolution_ratios = {round(float(r[0])/float(r[1]), 2): k.split('_')[0] for k, r in self.get_resolutions()}
        resolution_ratios_keys = resolution_ratios.keys()
        resolution_ratios_difs = [abs(ratio - x) for x in resolution_ratios_keys]
        result_index = resolution_ratios_difs.index(min(resolution_ratios_difs))
        return resolution_ratios[resolution_ratios_keys[result_index]]

    def package_audio_bitrate(self):
        return 256 * 1024

    def get_bitrate(self, f):
        try:
            bitrate = int(os.stat(os.path.join(self.temp_dir, f)).st_size * 8 / 1000 / self.duration)
        except:
            try:
                bitrate = self.file_resolutions[f][2]
            except KeyError:
                bitrate = self.get_audio_meta(f)['bit_rate']
        return bitrate

    def src_http_url(self):
        if self.Parameters.url.startswith('s3://'):
            s3_url = self.Parameters.url.split('/', 3)
            s3_host = "http://s3.mdst.yandex.net" if self.Parameters.s3_testing else 'http://s3.mds.yandex.net'
            return s3_host + '/' + s3_url[2] + '/' + s3_url[3]
        return self.Parameters.url

    def append_adaptive_m3u8(self, name=None):
        fname = self.temp_dir + "/" + self.m3u8_adaptive_file
        if not os.path.exists(fname):
            f = open(fname, "w")
            f.write("#EXTM3U\n#EXT-X-VERSION:3\n")
        else:
            f = open(fname, "a")
        if name:
            resolution = self.file_resolutions[name]
            f.write(
                "#EXT-X-STREAM-INF:PROGRAM-ID=1,BANDWIDTH={}000,RESOLUTION={}x{}\n".format(
                    int(resolution[2]) + int(resolution[3]), resolution[0], resolution[1]))
            f.write("{}\n".format(name))
        else:
            # запись в конец файла
            if self.Parameters.has_superresed:
                # EXT-X-SESSION-DATA:DATA-ID="com.yandex.video.levels.deephd",VALUE="1920x1080@4520481,1280x720@3106350"
                f.write("#EXT-X-SESSION-DATA:DATA-ID=\"com.yandex.video.levels.deephd\",VALUE=\"{}\"\n".format(
                    ','.join(['{}x{}@{}'.format(
                        v[0], v[1], (int(v[2])+int(v[3]))*1000) for v in sorted(
                            [m for l, m in self.file_resolutions.iteritems() if l.endswith('.m3u8') and not l.startswith('twopass_')],
                            reverse=True,
                            key=lambda r: int(r[0])
                        )[:1]]
                        )
                    )
                )

            # add timeline tiles as custom session data
            # На Tizen 3 длинный тег session_data ломает нативный плеер, если тег длиннее 230 символов и он идет после
            # X-VERSION. Поэтому пишем session-data в конец мастер плейлиста
            if self.timeline_tiles:
                if self.timeline_tiles.version == 0:
                    self.append_session_data(f, self.timeline_tiles.m3u8_uri_id, self.timeline_tiles.sprite_url_template)
                    self.append_session_data(f, self.timeline_tiles.m3u8_tiles_id, self.timeline_tiles.tiles)
                    self.append_session_data(f, self.timeline_tiles.m3u8_duration_id, self.timeline_tiles.sprite_duration)
                    self.append_session_data(f, self.timeline_tiles.m3u8_resolution_id, self.timeline_tiles.sprite_resolution)
                else:
                    for s in self.timeline_tiles.get_hls_session_infos():
                        self.append_session_data(f, s[0], s[1])

        f.close()

    @staticmethod
    def append_session_data(f, name, value):
        f.write('#EXT-X-SESSION-DATA:DATA-ID="{id}",VALUE="{value}"\n'.format(
            id=name,
            value=value,
        ))

    def get_format(self, f, url=None):
        if f in self.file_resolutions:
            return {
                'id':         f,
                'width':      self.file_resolutions[f][0],
                'height':     self.file_resolutions[f][1],
                'bitrate':    self.get_bitrate(f),
                'track_type': 'video',
                'type':       content_types[f.split('.')[-1]],
                'url':        "{}{}".format(self.base_url, f) if url is None else url
            }
        else:
            return {
                'id':         f,
                'bitrate':    self.get_bitrate(f),
                'track_type': 'audio',
                'tags':       self.get_audio_meta(f).get('tags'),
                'type':       content_types[f.split('.')[-1]],
                'url':        "{}{}".format(self.base_url, f) if url is None else url
            }

    def get_formats_files(self, files):
        return [self.get_format(f) for f in files if f.split('.')[-1] != 'm3u8' or (self.is_owner('BANNERSTORAGE') and f in self.file_resolutions)]

    def bufsize_approx(self, bitrate):
        # bitrates, x: 400 1000 2000 3000 5000 6000 7000
        # bufsizes, f(x): 100 200 400 600 1000 1200 1400
        # cubic approx: 0.0x^3 + 0.0x^2 + 0.1783x + 23.6648
        # square mean: 1.2536 %
        # return int(math.ceil(0.01783 * bitrate + 2.36648)) * 10 # with round up to tens

        # TODO:
        # Lowering decoder buffer may degrade quality noticeably, approx method requires more research, using
        # simple formula for a while
        return int(bitrate / 2)

    def build_ffmpeg_outfiles(self, i, j, resolutions, output_format, output_audio_codec, filter_scale, twopass,
                              filters, outfiles, files, is_embed, add_timeline_tiles=False):
        logging.info('new_duration {}'.format(self.new_duration))
        # VH-3987 (Too many packets buffered for output stream 2:1)
        out = '-map_metadata -1 -max_muxing_queue_size 400 '

        out_map_a = '0:a?'

        if self.Parameters.ffpmeg_capture_stop > 0:
            out += '-to {} '.format(self.Parameters.ffpmeg_capture_stop)

        if self.Parameters.ffpmeg_capture_stop > self.new_duration:
            out_map_a = "'[a{i}]'".format(i=j) if self.audio_stream else '2:a' if self.add_silence else '0:a?'
        elif self.add_silence and not self.audio_stream:
            out_map_a = '1:a'.format(i=j)
        elif self.need_loudnorm() and not self.need_preroll():
                out_map_a = "0:a:0 {}".format(
                    "-af {}".format(self.Context.loudnorm_filter) if len(self.Context.loudnorm_filter) > 0 else "")
        elif self.need_preroll():
            out_map_a = "'[audio{i}]'".format(i=j)

        if output_format == 'webm':
            out += '-c:v libvpx -crf {crf} -b:v {vbitrate}k -map {a} -c:a libvorbis -ar 48000'.format(
                crf=quality_crf[self.Parameters.ffpmeg_encoding_preset], vbitrate=resolutions[i][1][2], a=out_map_a)
        elif output_format == 'ogv':
            out += '-c:v libtheora -q:v {qscale} -map {a} -c:a libvorbis -ar 48000'.format(
                qscale=quality_scale[self.Parameters.ffpmeg_encoding_preset],
                a=out_map_a
            )
        else:
            width = int(resolutions[i][1][0])
            height = int(resolutions[i][1][1])
            out += '-c:v libx264 -preset {preset} -profile:v {profile} -level {level}'.format(
                preset='medium' if is_embed else self.Parameters.ffpmeg_encoding_preset,
                profile=self.get_h264_profile(height),
                level=self.get_embed_h264_level(height) if is_embed else self.get_h264_level(height, width)
            )

            if output_format == 'mp4':
                if is_embed:
                    out += ' -x264opts stitchable:fullrange=on' \
                           ' -g 10 -force_key_frames \'expr:gte(t,n_forced*1)\''
                else:
                    out += ' -force_key_frames \'expr:gte(t,n_forced*{gop_size})\' -x264opts ' \
                           'scenecut=40:open_gop=0:min-keyint={minkeyint}:keyint={keyint}{fullrange} ' \
                           '-movflags +faststart'.format(
                        gop_size=self.get_gop_size(),
                        minkeyint=self.get_target_keyint(),
                        keyint=self.get_gop_size()*self.get_target_keyint(),
                        fullrange=':fullrange=on' if self.is_fullrange_color(output_format) else ''
                    )
            elif output_format == 'hls':
                out += ' -force_key_frames \'expr:gte(t,n_forced*{})\''.format(
                    self.Parameters.ffmpeg_segment_time)
                if self.is_fullrange_color(output_format):
                    out += ' -x264opts fullrange=on'

            if self.is_drm_packaging() or self.is_ott_hls() or self.is_mss_packaging():
                out += ' -an'
            else:
                out += ' -map {a} -c:a {codec}{s} -ar 48000'.format(
                    a=out_map_a,
                    codec=output_audio_codec,
                    s=' -strict experimental' if self.Parameters._container else '')

            if is_embed:
                vbitrate = self.get_embed_video_bitrate(int(resolutions[i][1][1]))
                abitrate = self.get_embed_audio_bitrate(int(resolutions[i][1][1]))
                maxrate = vbitrate
                bufsize = int(int(vbitrate) * 5 / 2)
            else:
                vbitrate = int(resolutions[i][1][2])
                abitrate = resolutions[i][1][3]
                maxrate = int(vbitrate * 1.1) if self.is_owner('OTT') else vbitrate
                bufsize = self.bufsize_approx(vbitrate) \
                    if self.is_owner('OTT') \
                    else int(vbitrate * self.get_segment_time() / 2)

            out += ' -b:v {vbitrate}k -maxrate {maxrate}k -bufsize {bufsize}k'.format(vbitrate=vbitrate, maxrate=maxrate, bufsize=bufsize)
            if abitrate > 0:
                out += ' -b:a {abitrate}k'.format(abitrate=abitrate)

        name = "{path}{prefix}_{suffix}".format(
            path='embed/' if is_embed and not self.is_strm_embed() else '',
            prefix=self.Parameters.s3_key_prefix,
            suffix='_'.join([output_audio_codec, resolutions[i][0]]) if len(self.Parameters.ffmpeg_output_audio_codec.split(',')) > 1 else resolutions[i][0]
        )

        if add_timeline_tiles:
            scale_end = '[out{i}]'
            if not filter_scale.endswith(scale_end):
                raise RuntimeError('filter_scale must have "{}" in the end for using in timeline_preview. Actual: {}'.format(
                    scale_end,
                    filter_scale,
                ))

            if self.timeline_tiles.version == 0:
                # fps filter http://ffmpeg.org/ffmpeg-filters.html#fps-1
                # eof_action=pass allows include last image
                tile_filter = ",split=3[out{i}][osprite][oscreen];[osprite]fps=1/%d:eof_action=pass,scale=%d:%d:out_range=pc,tile=%s[sprite];[oscreen]fps=1/%d:eof_action=pass,scale=out_range=pc[screen]" % (
                    self.timeline_tiles.tduration,
                    self.timeline_tiles.twidth,
                    self.timeline_tiles.theight,
                    self.timeline_tiles.tiles,
                    self.timeline_tiles.tduration,
                )
            else:
                # fps filter http://ffmpeg.org/ffmpeg-filters.html#fps-1
                # eof_action=pass allows include last image
                tile_filter = ",split=2[out{i}][opreview];[opreview]%s,split=2[ospritelow][osprite];[osprite]scale=%d:%d:out_range=pc,tile=%s[sprite];[ospritelow]scale=%d:%d:out_range=pc,tile=%s[spritelow]" % (
                    self.timeline_tiles.ffprobe_select_filter,
                    self.timeline_tiles.sprite_hires.twidth,
                    self.timeline_tiles.sprite_hires.theight,
                    self.timeline_tiles.sprite_hires.ffprobe_tiles,
                    self.timeline_tiles.sprite_lowres.twidth,
                    self.timeline_tiles.sprite_lowres.theight,
                    self.timeline_tiles.sprite_lowres.ffprobe_tiles,
                )

            filter_scale = filter_scale[:-len(scale_end)] + tile_filter

        filters.append(filter_scale.format(
            i=j,
            width=resolutions[i][1][0],
            height=resolutions[i][1][1],
            fullrange='in_range=auto:out_range=pc:' if self.is_fullrange_color(output_format, is_embed=is_embed) else ''
        ))

        # https://trac.ffmpeg.org/wiki/EncodingForStreamingSites
        if output_format == 'hls':
            file_name = '{}.m3u8'.format(name) if twopass != 1 else 'twopass_{}.m3u8'.format(name)
            out_format = '-start_number 0 -hls_time {time} -hls_list_size 0 -hls_playlist_type vod -f hls {tmp}/{file_name}'.format(
                time=self.get_segment_time() or 10, tmp=self.temp_dir, file_name=file_name)
        else:
            file_name = '{}.{}'.format(name, output_format)
            if twopass == 1 and output_format != 'ogv':
                out_format = '-f {} /dev/null'.format(output_format)
            else:
                out_format = '-f {} {}/{}'.format(output_format, self.temp_dir, file_name)

        if add_timeline_tiles:
            if self.timeline_tiles.version == 0:
                # save timeline tiles to jpg files
                out_format += " -map '[sprite]' -qscale:v 1 -start_number {start} {prefix}%d.jpg".format(
                    start=self.timeline_tiles.sprite_start,
                    prefix=self.timeline_tiles.absolute_sprite_prefix,
                )
                # save timeline screens to jpg files
                out_format += " -map '[screen]' -qscale:v 1 -start_number {start} {prefix}%d.jpg".format(
                    start=self.timeline_tiles.screen_start,
                    prefix=self.timeline_tiles.absolute_screen_prefix,
                )
            else:
                # save timeline tiles to jpg files
                out_format += " -map '[sprite]' -qscale:v 4 -start_number {start} {prefix}%d.jpg".format(
                    start=self.timeline_tiles.sprite_start,
                    prefix=self.timeline_tiles.absolute_sprite_prefix,
                )

                # save timeline tiles lowres to jpg files
                out_format += " -map '[spritelow]' -qscale:v 16 -start_number {start} {prefix}%d.jpg".format(
                    start=self.timeline_tiles.sprite_start,
                    prefix=self.timeline_tiles.absolute_sprite_lowres_prefix,
                )

        if twopass < 1 or output_format == 'ogv':
            outfiles.append("-map '[out{i}]' {out} {format}".format(i=j, out=out, format=out_format))
        else:
            outfiles.append(
                "-map '[out{i}]' {out} -pass {twopass} -passlogfile ffmpeg2pass-out{i} {format}"
                .format(
                    i=j,
                    out=out,
                    tmp=self.temp_dir,
                    file_name=file_name,
                    twopass=twopass,
                    format=out_format
                )
            )

        files.append(file_name)
        if self.Parameters.keep_aspect_ratio:
            # Вычисляем выходное разрешения
            dar = self.get_display_aspect_ratio()
            ares = list(resolutions[i][1])
            if dar > 1:
                scale_width = int(float(ares[0]) / dar)
                if scale_width % 2 == 1:
                    scale_width += 1
                ares[1] = str(scale_width)
            else:
                scale_height = int(float(ares[1]) * dar)
                if scale_height % 2 == 1:
                    scale_height += 1
                ares[0] = str(scale_height)
            self.file_resolutions[file_name] = tuple(ares)
        else:
            self.file_resolutions[file_name] = resolutions[i][1]

    def run_hls_packager(self, cmd, env=None):
        self.send_status_hook("PACKAGING")
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd")) as pl:
            pr = sp.Popen(cmd, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, env=env)
            pr.wait()
            if pr.returncode != 0:
                msg = 'mp42hls exit code {}'.format(pr.returncode)
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure("mp42hls exit with none zero code {}".format(pr.returncode))


    def mp4dash(
            self,
            output_dir,
            video_files,
            audio_files=[],
            subtitles=[],
            verbose=True,
            manifest_only=False,
            force=True,
            mss=False,
            mss_client_manifest=None,
            mss_server_manifest=None,
            playready=False,
            playready_header=None,
            dash=False,
            mpd_name=None,
            widevine_header=None,
            timeline_sprites=None,
            timeline_sprites_lowres=None,
            supplemental_property=None,
            skip_subtitles_language_conversion=False
            ):
        """
        Run mp4dash packager

        :param output_dir: dir with output files and manifests
        :param video_files: collection of paths to video files
        :param audio_files: collection of tuples (lang, lang_name, path to audio files)
        :param subtitles: collection of tuples (index, title, format, lang, path to sub)
        :param verbose: be verbose
        :param manifest_only: output manifest files only
        :param force: allow output to an existing directory
        :param mss: produce an output compatible with smooth streaming
        :param mss_client_manifest: smooth streaming client manifest name
        :param mss_server_manifest: smooth streaming server manifest name
        :param playready: add PlayReady signaling to manifest
        :param playready_header: Add a PlayReady PRO element in the manifests
        :param dash: virtual param to create dash manifest
        :param mpd_name: mpd file name
        :param widevine_header: add a Widevine entry in the mpd
        :param timeline_sprites: 'param=name,' - params for adding thumbnails preview in mpd manifest
        :param timeline_sprites_lowres: 'param=name,' - params for adding thumbnails preview in lowres quality in mpd manifest
        :return: list of packaged files
        """
        subtitles_cmd = '' if subtitles is None else \
            ''.join(' "[+format={},+language={},+language_name={}]{}"'.format(f, l, t, p) for i, t, f, l, p in subtitles)

        audio_cmd = ''.join(' "[+language={},+language_name={}]{}"'.format(l, n, f) for l, n, f in audio_files)

        files = ' '.join(video_files) + audio_cmd + subtitles_cmd

        cmd = Command(self.mp4dash_bin) \
            .add('--verbose', verbose) \
            .add('--no-media', manifest_only) \
            .add('--force', force) \
            .add('--mpd-name {}', mpd_name) \
            .add('-o {}', output_dir) \
            .add('--smooth', mss) \
            .add('--smooth-client-manifest-name={}', mss_client_manifest) \
            .add('--smooth-server-manifest-name={}', mss_server_manifest) \
            .add('--playready', playready) \
            .add("--playready-header='#{}'", playready_header) \
            .add("--widevine-header='{}'", widevine_header) \
            .add("--dash-supplemental-property='{}'", supplemental_property) \
            .add("--skip-subtitles-language-conversion", skip_subtitles_language_conversion)

        if timeline_sprites:
            cmd.add("--thumbnails '{}'", timeline_sprites)

        if timeline_sprites_lowres:
            cmd.add("--thumbnails-lowres '{}'", timeline_sprites_lowres)

        cmd.add('{}', files)
        cmd.run(self.run_packager)

        packaged_files = []

        dash_manifest_path = os.path.join(output_dir, mpd_name)

        if mss:
            mss_manifest_path = os.path.join(output_dir, mss_client_manifest)

            self.run_shell("sed -i -e 's/{}\///g' {}".format(mss_server_manifest, dash_manifest_path))
            ismc = XML.parse(mss_manifest_path).getroot()

            if not manifest_only:
                ism = XML.parse(os.path.join(output_dir, mss_server_manifest)).getroot()
                self.ismindex(ismc, ism)

            packaged_files.append(mss_manifest_path)

        if dash:
            packaged_files.append(dash_manifest_path)

        if not manifest_only:
            packaged_files.extend([os.path.basename(f) for f in glob.glob(os.path.join(output_dir, 'init-*'))])
            packaged_files.extend(self.subdirs_files(output_dir))

        return packaged_files

    def mp42hls(
            self,
            file_path,
            show_info=True,
            segment_duration=4,
            segment_duration_threshold=50,
            encryption_mode='SAMPLE-AES',
            encryption_key=None,
            encryption_iv_mode='fps',
            encryption_key_format='com.apple.streamingkeydelivery',
            encryption_key_uri='skd://content-uuid',
            segment_filename_template=None,
            index_filename=None,
            hls_version=4
            ):
        if encryption_mode is not None and encryption_key is None:
            raise ValueError('You must specify encryption key')

        if encryption_key_uri is not None and not encryption_key_uri.startswith('skd://'):
            encryption_key_uri = 'skd://' + encryption_key_uri

        path_safe_name = file_path.replace("/", '_').replace('.', '__')
        meta_file_name = 'meta_{}.json'.format(path_safe_name)

        info = Command(self.mp42hls_bin) \
            .add('--hls-version {}', hls_version) \
            .add('--show-info', show_info) \
            .add('--segment-duration {}', segment_duration) \
            .add('--segment-duration-threshold {}', segment_duration_threshold) \
            .add('--encryption-mode {}', encryption_mode) \
            .add('--encryption-key {}', encryption_key) \
            .add('--encryption-iv-mode {}', encryption_iv_mode) \
            .add('--segment-filename-template {}', segment_filename_template) \
            .add('--segment-url-template {}', segment_filename_template) \
            .add('--index-filename {}', index_filename) \
            .add('--encryption-key-format {}', encryption_key_format) \
            .add('--encryption-key-uri {}', encryption_key_uri) \
            .add('{}', file_path) \
            .add('> {}', meta_file_name) \
            .run(run_function=self.run_hls_packager)

        with open(meta_file_name) as meta:
            info = json.loads(meta.read())

        self.run_shell('rm {}'.format(meta_file_name))

        return info

    def hls_package(
            self,
            file_path,
            out_dir,
            n,
            template_name_prefix,
            kid=None,
            ck=None,
            ):
        encrypt = kid is not None and ck is not None

        seg_template = '{}-{}-%d.ts'.format(n, template_name_prefix)
        index_name = '{}-{}.m3u8'.format(n, template_name_prefix)
        if encrypt:
            meta = self.mp42hls(
                    file_path,
                    encryption_key=ck,
                    encryption_key_uri=kid,
                    segment_filename_template=seg_template,
                    index_filename=index_name
                    )
        else:
            meta = self.mp42hls(
                    file_path,
                    encryption_mode=None,
                    encryption_key=None,
                    encryption_iv_mode=None,
                    encryption_key_format=None,
                    encryption_key_uri=None,
                    segment_filename_template=seg_template,
                    index_filename=index_name
                    )

        self.run_shell('mkdir -p {}'.format(out_dir))
        self.run_shell('mv {} {}'.format('*.ts', out_dir))
        self.run_shell('mv {} {}'.format('*.m3u8', out_dir))

        for key in meta['stats'].keys():
            meta['stats'][key] = int(meta['stats'][key])

        return index_name, meta

    def build_ffmpeg_command(self, resolutions, twopass=0, thumbnails=False, append_awaps_resolutions=True):
        filters = []
        outfiles = []
        files = []
        resolution_keys = [k[0] for k in resolutions]
        max_resolution = max(resolutions, key=lambda x: int(x[1][0]))
        resolution_count = len(resolutions)

        # Append video back and audio silent
        if self.Parameters.ffpmeg_capture_stop > self.new_duration:
            if self.audio_stream:
                filter_scale = "[int{i}][0:a][1:v][2:a]concat=n=2:v=1:a=1[int{i}c][a{i}];[int{i}c]"
            else:
                filter_scale = "[int{i}][1:v]concat=n=2:v=1[int{i}c];[int{i}c]"
        else:
            filter_scale = "[int{i}]"

        filter_sar = "scale=trunc(min({width}\,{height}*dar)/sar/2)*2:-1"
        filter_pad = "pad=ceil({width}/sar/2)*2:{height}:(ow-iw)/2:(oh-ih)/2:black"

        filter_scale_embed = filter_scale
        filter_scale_embed_int = ["format=yuv420p", filter_sar, filter_pad]

        # Set color format  YUV 4:2:0 planar 8-bits
        filter_scale_int = ["format=yuv420p"]

        # VH-2920 crop 1px for GIF
        if (self.resolution_src["width"] % 2 != 0 or self.resolution_src["height"] % 2 != 0) and self.src_mime_type == 'image/gif':
            filter_scale_int.insert(0, "crop={width}:{height}:{crop_width}:{crop_height}".format(
                width=self.resolution_src["width"],
                height=self.resolution_src["height"],
                crop_width=0 if self.resolution_src["width"] % 2 == 0 else 1,
                crop_height=0 if self.resolution_src["height"] % 2 == 0 else 1
            ))

        # Convert to resolution with original SAR
        if not self.Parameters.keep_aspect_ratio:
            filter_scale_int.append(filter_sar)

        # Add black area
        if not self.Parameters.ffmpeg_no_padding and not self.Parameters.keep_aspect_ratio:
            filter_scale_int.append(filter_pad)

        filter_scale += ','.join(filter_scale_int) + "[out{i}_scale]"

        # Scale and set SAR 1:1
        if self.need_preroll():
            filter_scale += ";[out{i}_scale]scale={fullrange}{width}:{height},setsar=1/1[main_out{i}];[1:v][main_out{i}]scale2ref=iw:-2[preroll][main_out_ref{i}];[preroll]setsar=1/1[preroll_sar];[preroll_sar][1:a][main_out_ref{i}][0:a]concat=n=2:v=1:a=1[out_preroll{i}]" \
                            + ("[preroll_main_audio{i}];[preroll_main_audio{i}]" + self.Context.loudnorm_filter + "[audio{i}]" if self.need_loudnorm() and len(self.Context.loudnorm_filter) > 0 else "[audio{i}]") \
                            + ";[out_preroll{i}]null[out{i}]"
        elif self.Parameters.keep_aspect_ratio:
            filter_scale += ";[out{i}_scale]scale={fullrange}if(gt(dar\,1)\,trunc({width}/sar/2)*2\,-2):if(gt(dar\,1)\,-2\,trunc({height}/sar/2)*2)[out{i}]"
        else:
            if self.is_strm_embed():
                filter_scale += ";[out{i}_scale]scale={fullrange}{width}:{height},setsar=1/1[out{i}]"
            else:
                filter_scale += ";[out{i}_scale]scale={fullrange}{width}:{height},setsar=1/1[out{i}]"

        j = 0
        for i in xrange(resolution_count):
            for output_format in self.get_output_format().split(','):
                # generate timeline tiles (preview) from minimum resolution
                add_timeline_tiles = self.timeline_tiles and max_resolution[1][0] == resolutions[i][1][0]

                # Hack for catchup to vod - generate tiles from the second max resolution
                if self.is_strm_hls_url() and output_format == 'hls' and resolution_count > 1 \
                        and i == resolution_count - 2:
                    add_timeline_tiles = self.timeline_tiles is not None

                self.build_ffmpeg_outfiles(
                    i, j, resolutions,
                    output_format, 'aac', filter_scale, twopass, filters,
                    outfiles, files,
                    self.is_strm_embed(), add_timeline_tiles=add_timeline_tiles,
                )

                if self.is_strm_hls_url() and output_format == 'hls' and i == resolution_count-1 and resolution_count > 1:
                    filters.pop()
                    outfiles.pop()
                else:
                    j += 1

        if self.Parameters.strm_embed_ready and append_awaps_resolutions and not self.is_strm_embed():
            if self.filter_crop and not self.Parameters.ffmpeg_crop_black_area:
                filter_scale_embed_int.insert(0, self.filter_crop)

            filter_scale_embed += ','.join(filter_scale_embed_int) + \
                                  "[out{i}_scale];[out{i}_scale]scale={fullrange}{width}:{height},setsar=1/1[out{i}]"
            embed_resolutions = self.get_embed_resolutions()
            for audio_codec in self.Parameters.ffmpeg_output_audio_codec.split(','):
                for i in xrange(len(embed_resolutions)):
                    if self.is_resolution_match(embed_resolutions[i][1][0], embed_resolutions[i][1][1]):
                        self.build_ffmpeg_outfiles(
                            i, j, embed_resolutions,
                            'mp4', audio_codec, filter_scale_embed, twopass,
                            filters, outfiles,
                            self.files_awaps, True, add_timeline_tiles=False,
                        )
                        if 'mp4' in self.get_output_format().split(',') and audio_codec == 'aac' \
                                and self.is_owner('STRM') and embed_resolutions[i][0] in resolution_keys:
                            filters.pop()
                            outfiles.pop()
                        else:
                            j += 1

        if thumbnails and not (self.Parameters.ffpmeg_thumbnails_t > 0 and self.Parameters.ffpmeg_first_resolution):
            if self.Parameters.ffpmeg_capture_stop > self.new_duration:
                # revert audio concat
                filter_scale_thumb = ';'.join(filter_scale.split(';')[1:]).replace('[int{i}c]', '[int{i}]')
            else:
                filter_scale_thumb = filter_scale
            filters.append(
                filter_scale_thumb.format(
                    i=len(outfiles), width=max_resolution[1][0],
                    height=max_resolution[1][1],
                    fullrange=''
                )
            )
            select = 'gt(scene\,{})+(eq(pict_type\,I)*gt(t\,1)*isnan(prev_selected_t))'.format(
                self.Parameters.ffmpeg_thumbnails_scene_threshold
            )
            filters.append('[out{i}]select={select}[out{i}_thumb]'.format(i=len(outfiles), select=select))
            outfile = "-map '[out{i}_thumb]' -qscale:v 1 -vsync vfr {prefix}%03d.jpg"
            if self.need_preroll():
                outfile += " -map '[audio{i}]' -f null /dev/null"
            outfiles.append(outfile.format(i=len(outfiles), prefix=self.get_thumbs_path_prefix()))

        video_stream = '[0:v]'
        amedia_logo_filter = ''

        if self.src_mime_type == 'video/x-ms-asf':
            video_stream = '[0:v:{}]'.format(self.resolution_src["index"])
        if self.Parameters.logo:
            video_stream = '[amedia_v]'
            amedia_logo_filter = "movie='{logo}'[wm];[wm][0:v]scale2ref=w=76/1080*ih:h=85/1080*ih:[wm][v];" \
                                 "[wm]setsar=1,format=argb,colorchannelmixer=aa=0.20[logo];" \
                                 "[v][logo] overlay=main_w-overlay_w-main_w/24:main_h/18 {out};".format(
                    logo=self.amediateka_logo(),
                    out=video_stream
                    )

        filters.insert(0, "{}{}split={}{}".format(
            amedia_logo_filter,
            self.get_ffmpeg_input_filters(video_stream),
            len(outfiles),
            "".join(map(lambda i: "[int{}]".format(i), xrange(len(outfiles))))
        ))

        return (";".join(filters), " ".join(outfiles), files)

    def get_ffmpeg_input_filters(self, video_stream):
        out = video_stream
        if len(self.ffmpeg_input_filters) > 0:
            out += ",".join(self.ffmpeg_input_filters) + "[int];[int]"

        if self.Parameters.ffpmeg_crossfade_duration > 0:
            out += "split[body][pre];[pre]trim=duration={fade_duration},format=yuva420p," \
                    "fade=d={fade_duration}:alpha=1,setpts=PTS+{fade_keyframe}/TB[jt];" \
                    "[body]trim={fade_duration},setpts=PTS-STARTPTS[main];[main][jt]overlay[fade];[fade]".format(
                fade_duration=self.Parameters.ffpmeg_crossfade_duration,
                fade_keyframe=self.duration - 2 * self.Parameters.ffpmeg_crossfade_duration
            )
        return out

    def run_ffmpeg_splitter(self):
        """
        video stream splitter:
            1. Create dir segments
            2. Run ffmpeg make segmens in to dir segments
        :return: list of relative paths to segments
        """
        segments_dir = 'segments'
        if not os.path.isdir(segments_dir):
            os.mkdir(segments_dir)
        cmdline = "{} -i {} -f segment -segment_time {} -c copy -map 0:v {}".format(
            self.ffmpeg_bin,
            self.src_video,
            self.Parameters.parallel_segment_duration,
            os.path.join(segments_dir, 'segment.%d.mp4')
        )
        self.run_ffmpeg(cmdline)
        segments = sorted(os.listdir(segments_dir), key=lambda x: int(x.split('.')[1]))
        return [os.path.join(segments_dir, f) for f in segments]

    def streams_concat(self, segments, files):
        """
        video streams concatenate:
            1. Create a file with a list of segments for each output resolution
            2. run ffmpeg to concatenate the segments for each output resolution
        :param segments: list of relative paths to segments
        :param files: list of output files
        """
        for file in files:
            if file.endswith('.mp4'):
                logging.info("file {}".format(file))
                stream_concat_list = 'stream_concat_{}.txt'.format(file)
                with open(stream_concat_list, 'wt') as stream_concat_file:
                    for segment in segments:
                        stream_concat_file.write('file \'{}/{}\'\n'.format(segment, file))
                #ToDo one run ffmpeg -i 1080p_concat.txt -i 720p_concat.txt -filter_complex "[0:v]concat[]" -map "[outv0] 1080p.mp4 -map "[outv1]" 720p.mp4
                cmdline = "{} -f concat -safe 0 -i {} -c copy {}".format(
                    self.ffmpeg_bin, stream_concat_list, self.sandbox_path(file)
                )
                self.run_ffmpeg(cmdline)

    def stream_concat(self, files):
        concat_video = tempfile.NamedTemporaryFile(suffix=".mp4").name
        stream_concat_list = '{}/stream_concat.txt'.format(self.temp_dir)
        with open(stream_concat_list, 'wt') as stream_concat_file:
            for f in files:
                stream_concat_file.write('file \'{}\'\n'.format(f))

        cmdline = "{} -f concat -safe 0 -i {} -c copy {}".format(self.ffmpeg_bin, stream_concat_list, concat_video)
        self.run_ffmpeg(cmdline)
        return concat_video

    def stream_copy_hls(self, file_name):
        concat_video = tempfile.NamedTemporaryFile(suffix=".mp4").name
        cmdline = "{} -i {} -c copy -start_number 0 -hls_time {} -hls_list_size 0 -hls_playlist_type vod -f hls {}".format(
            self.ffmpeg_bin, self.src_video, self.get_segment_time() or 10, self.sandbox_path(file_name)
        )
        self.run_ffmpeg(cmdline)
        return concat_video

    def stream_loop(self, num):
        loop_video = tempfile.NamedTemporaryFile(suffix=".mp4").name
        stream_loop_list = '{}/stream_loop.txt'.format(self.temp_dir)
        with open(stream_loop_list, 'wt') as stream_loop_file:
            for i in range(num):
                stream_loop_file.write('file \'{}\'\n'.format(self.src_video))

        cmdline = "{} -f concat -safe 0 -i {} -c {} {}".format(
            self.ffmpeg_bin,
            stream_loop_list,
            'libx264' if self.src_mime_type == 'image/gif' else 'copy',
            loop_video
        )
        self.run_ffmpeg(cmdline)
        return loop_video

    def image_loop(self):
        loop_video = tempfile.NamedTemporaryFile(suffix=".mp4").name
        cmdline = "{} -y -loop 1 -f image2 -i {} -f lavfi -i anullsrc=r=48000:n=1024 -t {} -pix_fmt yuv420p " \
                  "-force_key_frames 'expr:gte(t,n_forced*{})' -vf framerate=25 -c:v libx264 {} -c:a acc".format(
            self.ffmpeg_bin,
            self.src_video,
            self.Parameters.ffpmeg_capture_stop if self.Parameters.ffpmeg_capture_stop else
            self.Parameters.ffmpeg_segment_time,
            int(self.Parameters.ffmpeg_segment_time / 2),
            loop_video,
        )
        self.run_ffmpeg(cmdline)
        return loop_video

    def run_shell(self, cmd, env=None, log_prefix=None):
        self.send_status_hook("SHELL CMD")
        with sdk2.helpers.ProcessLog(self, logger=log_prefix or logging.getLogger("cmd")) as pl:
            pr = sp.Popen(cmd, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, env=env)
            pr.wait()
            if pr.returncode != 0:
                msg = 'shell cmd exit code {}'.format(pr.returncode)
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure("shell cmd exit")
            return pl

    def run_ffmpeg(self, cmd, env=None, send_status_hook=True):
        self.send_status_hook("VIDEO_AUDIO_CONVERSION")
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd")) as pl:
            cmd_prepared = shlex.split(cmd)

            if not isinstance(cmd_prepared, (list, tuple)):
                raise common.errors.TaskError('Incorrect ffmpeg cmd parameter type {}'.format(type(cmd_prepared)))

            pr = sp.Popen(cmd_prepared, shell=False, stdout=pl.stdout, stderr=sp.STDOUT, env=env)
            pr.wait()
            if pr.returncode != 0:
                msg = 'ffmpeg exit code {}'.format(pr.returncode)
                if send_status_hook:
                    self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure("ffmpeg exit")

    def run_fragmenter(self, cmd, env=None):
        self.send_status_hook("FRAGMENTING")
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd")) as pl:
            pr = sp.Popen(cmd, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, env=env)
            pr.wait()
            if pr.returncode != 0:
                msg = 'mp4fragment exit code {}'.format(pr.returncode)
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure("mp4fragment exit")

    def run_splitter(self, cmd, env=None):
        self.send_status_hook("SPLITTING")
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd")) as pl:
            pr = sp.Popen(cmd, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, env=env)
            pr.wait()
            if pr.returncode != 0:
                msg = 'mp4split exit code {}'.format(pr.returncode)
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure("mp4split exit")

    def run_encrypter(self, cmd, env=None):
        self.send_status_hook("ENCRYPTING")
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd")) as pl:
            pr = sp.Popen(cmd, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, env=env)
            pr.wait()
            if pr.returncode != 0:
                msg = 'mp4encrypt exit code {}'.format(pr.returncode)
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure("mp4encrypt exit")

    def run_packager(self, cmd, env=None):
        self.send_status_hook("PACKAGING")
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd")) as pl:
            pr = sp.Popen(cmd, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, env=env)
            pr.wait()
            if pr.returncode != 0:
                msg = 'mp4dash exit code {}'.format(pr.returncode)
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure("mp4dash exit with none zero code {}".format(pr.returncode))

    def transcode_fast(self):
        logging.info("Running ffmpeg to get first resolution")
        self.first_resolution = self.resolution_ratio_key + "_240p"

        first_resolution_adaptive = tuple(
            (k, r) for k, r in self.get_resolutions() if self.first_resolution == k
        )
        files_fast = []
        if (len(first_resolution_adaptive) > 0):
            files_fast = self.transcode(first_resolution_adaptive, False, False)
            self.publish_files_fast(files_fast)

        return files_fast

    def transcode(self, resolutions, two_pass_encoding=True, append_awaps_resolutions=True):
        self.ffmpeg_input_filters = []
        if self.Parameters.ffmpeg_framerate > 0:
            self.ffmpeg_input_filters.append('framerate={}'.format(self.Parameters.ffmpeg_framerate))
        elif self.src_mime_type == 'image/gif' or self.Parameters.strm_embed_ready:
            self.ffmpeg_input_filters.append('framerate=25')

        if self.Parameters.ffmpeg_setpts_pts_startpts:
            self.ffmpeg_input_filters.append('setpts=PTS-STARTPTS')

        # Crop the input video to given dimensions
        if self.Parameters.ffmpeg_crop_black_area and self.filter_crop:
            self.ffmpeg_input_filters.append(self.filter_crop)

        if self.Parameters.ffpmeg_capture_start > 0:
            cmdline_seek = "-ss {} -i '{}'".format(self.Parameters.ffpmeg_capture_start, self.src_video)
        else:
            cmdline_seek = "-i '{}'".format(self.src_video)

        if self.Parameters.ffpmeg_capture_stop > self.new_duration:
            cmdline_seek += " -f lavfi -t {durv} -i 'color=c=black:s={w}x{h}:sar={a}:r=25' -f lavfi -t {dura} -i " \
                            "'anullsrc=channel_layout=stereo:sample_rate=48000'".format(
                durv=int(self.Parameters.ffpmeg_capture_stop - self.new_duration + 2),
                dura=int(self.Parameters.ffpmeg_capture_stop - self.new_duration + 2) if self.audio_stream else
                self.Parameters.ffpmeg_capture_stop,
                w=self.resolution_src["width"],
                h=self.resolution_src["height"],
                a=self.resolution_src["sample_aspect_ratio"].replace(':', '/', 1),
            )
        elif self.add_silence and not self.audio_stream:
            cmdline_seek += " -f lavfi -t {dur} -i 'anullsrc=channel_layout=stereo:sample_rate=48000'".format(
                dur=self.new_duration
            )

        if self.need_preroll():
            cmdline_seek += " -i '{}'".format(self.preroll_file())

        if two_pass_encoding:
            # pass 1
            (filter_complex, outfiles, files) = self.build_ffmpeg_command(resolutions, 1, self.Parameters.ffpmeg_thumbnails_scene, append_awaps_resolutions)
            cmdline = "{} -y {} -filter_complex '{}' {}".format(self.ffmpeg_bin, cmdline_seek, filter_complex, outfiles)

            self.Context.ffmpeg_cmds.append(cmdline)
            if not self.Parameters.sb_parallel_encoding and not self.Parameters.rt_parallel_encoding:
                logging.info("Running ffmpeg pass 1 command: {}".format(cmdline))
                self.run_ffmpeg(cmdline)

            # pass 2
            (filter_complex, outfiles, files) = self.build_ffmpeg_command(resolutions, 2, False, append_awaps_resolutions)
            cmdline = "{} -y {} -filter_complex '{}' {}".format(
                self.ffmpeg_bin, cmdline_seek, filter_complex, outfiles
            )
            self.Context.ffmpeg_cmds.append(cmdline)
            if not self.Parameters.sb_parallel_encoding and not self.Parameters.rt_parallel_encoding:
                logging.info("Running ffmpeg pass 2 command: {}".format(cmdline))
                try:
                    self.run_ffmpeg(cmdline, send_status_hook=False)
                except common.errors.TaskFailure:
                    # Fallback to one pass encoding VH-7519
                    two_pass_encoding = False
                    self.Context.ffmpeg_cmds = []

        if not two_pass_encoding:
            (filter_complex, outfiles, files) = self.build_ffmpeg_command(resolutions, 0, self.Parameters.ffpmeg_thumbnails_scene, append_awaps_resolutions)
            cmdline = "{} -y {} -filter_complex '{}' {}".format(self.ffmpeg_bin, cmdline_seek, filter_complex, outfiles)

            self.Context.ffmpeg_cmds.append(cmdline)
            if not self.Parameters.sb_parallel_encoding and not self.Parameters.rt_parallel_encoding:
                logging.info("Running ffmpeg command: {}".format(cmdline))
                self.run_ffmpeg(cmdline)

        return files

    def get_reverse_resolutions_sorting(self):
        if self.is_owner('OTT'):
            return True
        return self.Parameters.reverse_resolutions_sorting

    def is_resolution_match(self, width, height):
        return self.resolution_src["width"] >= int(width)-8 or self.resolution_src["height"] >= int(height)-8

    def is_strm_embed(self):
        return self.is_owner('STRM') and self.Parameters.strm_embed_ready

    def is_strm_hls_url(self):
        return self.Parameters.url.startswith('https://strm.yandex.ru/kal/') and self.Parameters.url.split('?')[0].endswith('.m3u8')

    def is_fullrange_color(self, output_format, is_embed=False):
        return (self.Parameters.ffmpeg_fullrange_color and output_format in ['hls', 'mp4']) or (output_format == 'mp4' and is_embed)

    def get_strm_hls_range(self):
        url_s3_parse = urlparse(self.Parameters.url)
        m3u8_url_params = parse_qs(url_s3_parse.query)
        strm_hls_start = 0
        strm_hls_end = 0
        url_s3_key = ''
        if 'start' in m3u8_url_params and 'end' in m3u8_url_params:
            url_s3_key = url_s3_parse.path.split('/', 2)[2]
            strm_hls_start = int(math.floor(float(m3u8_url_params['start'][0]) / 10)) * 10
            strm_hls_end = int(math.ceil(float(m3u8_url_params['end'][0]) / 10)) * 10
        return (strm_hls_start, strm_hls_end, url_s3_key)

    def get_s3_secret(self, owner):
        try:
            if self.Parameters.s3_testing:
                return ("http://s3.mdst.yandex.net", sdk2.Vault.data(owner, "s3_test_secret_key_id"), sdk2.Vault.data(owner, "s3_test_secret_key"))
            else:
                return ("http://s3.mds.yandex.net", sdk2.Vault.data(owner, "s3_secret_key_id"), sdk2.Vault.data(owner, "s3_secret_key"))
        except:
            msg = 'exception get s3 secrets from Vault'
            self.send_status_hook({'warning_message': msg})
            raise TemporaryError(msg)

    def get_m3u8_high_base_path(self, playlist_path):
        import m3u8
        playlist = m3u8.load(playlist_path)
        return max(playlist.playlists, key=lambda k: k.stream_info.resolution[0]).base_path

    def get_resolutions_ratio_adaptive(self):
        if self.is_owner('OTT'):
            ratio_matched_resolutions = tuple((k, r) for k, r in self.get_resolutions() if self.resolution_ratio_key + "_" in k)
            resolutions = []
            for key, resolution in ratio_matched_resolutions:
                width, height, b, d, p = resolution
                width, height = int(width), int(height)
                if self.resolution_src["width"] >= width and self.resolution_src["height"] >= height:
                    resolutions.append((key, resolution))
                elif self.resolution_src["width"] >= width and self.resolution_src["height"] < height:
                    resolutions.append((key, (width, self.resolution_src["height"], b, d, p)))
                elif self.resolution_src["height"] >= height and self.resolution_src["width"] < width:
                    resolutions.append((key, (self.resolution_src["width"], height, b, d, p)))
        else:
            resolutions = [
                (k, r) for k, r in self.get_resolutions() if
                (self.resolution_ratio_key + "_" in k and self.first_resolution != k and
                 self.is_resolution_match(r[0], r[1]))
                or (k == self.resolution_ratio_key + '_240p' and self.first_resolution is None and self.resolution_src["width"] >= int(r[0]) / 2 and self.resolution_src["height"] >= int(r[1]) / 2)
            ]
        return sorted(tuple(resolutions), reverse=self.get_reverse_resolutions_sorting(), key=lambda r: int(r[1][0]))

    def get_semaphore_priority(self):
        semaphore_priority = 'low'
        if not self.owner.endswith('-LOW') and \
                self.Parameters.priority.cls == ctt.Priority.Class.SERVICE and \
                self.Parameters.priority.scls == ctt.Priority.Subclass.HIGH:
            semaphore_priority = 'high'
        return semaphore_priority

    def on_save(self):
        if self.owner.endswith('-LOW'):
            self.Parameters.priority.cls = ctt.Priority.Class.SERVICE
            self.Parameters.priority.scls = ctt.Priority.Subclass.HIGH

        if self.is_owner('ZEN'):
            self.Parameters.kill_timeout = 1800

    def on_enqueue(self):
        self.Requirements.client_tags &= ctc.Tag.Group.LINUX & ~ctc.Tag.IPV4

        size = 0

        try:
            size = int(requests.head(self.src_http_url()).headers['content-length']) * 10 / 1024 / 1024
        except:
            pass
        if self.Parameters.url and self.is_strm_hls_url():
            strm_hls_start, strm_hls_end, url_s3_key = self.get_strm_hls_range()
            kill_timeout = (strm_hls_end-strm_hls_start)*3
            if self.Parameters.ffmpeg_two_pass:
                kill_timeout *= 2
            if kill_timeout > self.Parameters.kill_timeout:
                self.Parameters.kill_timeout = kill_timeout
            disk_space = (strm_hls_end-strm_hls_start)*5
            if disk_space > self.Requirements.disk_space:
                self.Requirements.disk_space = disk_space
        elif size > self.Requirements.disk_space:
            self.Requirements.disk_space = size+100

    def download_resource(self, resource_type):
        logging.info("Downloading resource {}".format(resource_type))
        self.send_status_hook("RESOURCE_DOWNLOADING")
        try:
            res = list(sdk2.Resource.find(resource_type, state=[ctr.State.READY],
                                          attrs=dict(released=ctt.ReleaseStatus.STABLE)).limit(1))[0]
            res_data = sdk2.ResourceData(res)
        except Exception as e:
            logging.exception("Failed to download resource %s", resource_type)
            raise TemporaryError("Downloading resource {}, error: {}".format(resource_type, e))
        logging.info("Resource {} downloaded: {}".format(resource_type, str(res.description)))
        return res_data

    def configure_toolset(self):
        if self.Parameters._container:
            self.ffmpeg_bin = '/usr/bin/ffmpeg'
            self.ffprobe_bin = '/usr/bin/ffprobe'
            self.youtube_dl_bin = '/usr/bin/youtube-dl'
        else:
            ffmpeg = self.download_resource(resource_types.FFMPEG_BIN)
            self.ffmpeg_bin = str(ffmpeg.path / "ffmpeg")
            self.ffprobe_bin = str(ffmpeg.path / "ffprobe")
            self.youtube_dl_bin = sys.executable + ' -m youtube_dl'

        if self.is_ott_hls() or self.is_drm_packaging() or self.is_mss_packaging():
            bento4_bundle = self.download_resource(BENTO4_BINARIES)
            self.mp4dash_bin = str(bento4_bundle.path / "mp4dash")
            self.mp4fragment_bin = str(bento4_bundle.path / "mp4fragment")
            self.mp4encrypt_bin = str(bento4_bundle.path / "mp4encrypt")
            self.mp4split_bin = str(bento4_bundle.path / "mp4split")
            self.mp42hls_bin = str(bento4_bundle.path / "mp42hls")

        if self.Parameters.ffmpeg_timeline_tiles:
            jpegtran = self.download_resource(strm_resources.JPEGTRAN_BINARY)
            self.jpegtran_bin = str(jpegtran.path)


    def configure(self):
        self.s3_dir = self.Parameters.s3_dir
        if self.s3_dir and self.s3_dir[-1] != "/":
            self.s3_dir += "/"
        #  https://st.yandex-team.ru/VH-5031 internal only video
        s3_host = 's3.mdst.yandex.net' if self.Parameters.s3_testing else 's3.mds.yandex.net'
        if self.Parameters.nda:
            if re.match('internal-vh-[\w-]+-converted', self.Parameters.s3_bucket):
                strm_host = s3_host
            else:
                msg = 'S3 bucket is not internal'
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)
        elif self.Parameters.outstaff:
            if re.match('external-vh-[\w-]+-converted', self.Parameters.s3_bucket):
                strm_host = 'strm-ext.yandex.net'
            else:
                msg = 'S3 bucket is not external'
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)
        else:
            strm_host = 'streaming.tst.strm.yandex.net' if self.Parameters.s3_testing else 'strm.yandex.ru'
        secret_owner = self.get_secret_owner()
        self.s3_endpoint_url, s3_secret_key_id, s3_secret_key = self.get_s3_secret(secret_owner)
        self.s3transfer = self.s3_connect(self.s3_endpoint_url, s3_secret_key_id, s3_secret_key, self.Parameters.s3_bucket)
        self.base_url = "https://{}/{}/{}".format(strm_host, self.Parameters.s3_bucket, self.s3_dir)
        self.bucket_url = "https://{}/{}".format(strm_host, self.Parameters.s3_bucket)
        self.strm_host = "https://{}".format(strm_host)
        self.s3_url = "https://{}".format(s3_host)

        if self.is_ott_hls() or self.is_ott_trailer() or \
                (self.is_drm_packaging() and self.Parameters.package_clear_hls_as_well):
            self.m3u8_adaptive_file = "master.m3u8"
            self.m3u8_adaptive_url = "{}{}/{}".format(
                    self.base_url,
                    self.s3_sub_dir(),
                    self.m3u8_adaptive_file
                    )
        else:
            self.m3u8_adaptive_file = "{}.m3u8".format(self.Parameters.s3_key_prefix)
            self.m3u8_adaptive_url = "{}{}".format(self.base_url, self.m3u8_adaptive_file)

        self.temp_dir = tempfile.mkdtemp()
        if self.Parameters.strm_embed_ready or self.is_strm_hls_url():
            embed_s3_endpoint_url, embed_s3_secret_key_id, embed_s3_secret_key = self.get_s3_secret('STRM')
            self.embed_s3transfer = self.s3_connect(embed_s3_endpoint_url, embed_s3_secret_key_id, embed_s3_secret_key, 'strm')
            os.mkdir(self.temp_dir + '/embed')
        self.src_mime_type = ''
        self.file_resolutions = {}
        self.audio_stream = None
        self.pool = ThreadPool(128) if self.Parameters.rt_parallel_encoding else ThreadPool(16)
        self.add_silence = False
        self.is_youtube_dl = False
        self.filter_crop = ''

        if self.is_youtube_url():
            self.is_youtube_dl = True
            self.src_video = self.sandbox_path(str(self.Parameters.s3_key_prefix))
        else:
            self.src_video = tempfile.NamedTemporaryFile(suffix=".mp4").name

        if self.Parameters.strm_embed_ready or self.Parameters.silence:
            self.add_silence = True

        if self.is_drm_packaging():
            if self.get_output_format() != "mp4":
                raise common.errors.TaskFailure("ISO BMFF(mp4) output format required for DRM packaging")
            try:
                if self.Parameters.production:
                    self.drm_kek = sdk2.Vault.data(secret_owner, "drm_mediakeys_kek")
                    self.drm_kek_id = sdk2.Vault.data(secret_owner, "drm_mediakeys_kekId")
                else:
                    self.drm_kek = sdk2.Vault.data(secret_owner, "drm_mediakeys_testing_kek")
                    self.drm_kek_id = sdk2.Vault.data(secret_owner, "drm_mediakeys_testing_kekId")
            except:
                msg = 'exception get drm keys secrets from Vault'
                self.send_status_hook({'error_message': msg})
                raise TemporaryError(msg)

            for handler in logging.root.handlers:
                handler.setFormatter(self.DrmSecretFormatter(handler.formatter))

        if self.Parameters.rt_parallel_encoding:
            if self.Parameters.production:
                rt_token = sdk2.Vault.data(secret_owner, "faas_rt")
                rt_host = 'https://video-faas.n.yandex-team.ru/'
            else:
                rt_token = sdk2.Vault.data(secret_owner, 'faas_rt_test')
                rt_host = 'https://video-faas-prestable.n.yandex-team.ru/'

            self.rt_client = RTClient(
                'transcoder-test',
                self.Parameters.s3_dir,
                self.Parameters.s3_key_prefix,
                rt_token,
                rt_host,
                self.retry_session
                )

        self.configure_toolset()

        try:
            if self.is_drm_packaging():
                task_tags = self.server.task[self.id].read()["tags"]

                task_tags = task_tags + ["DRM"]

                if self.Parameters.drm_widevine:
                    task_tags = task_tags + ["WIDEVINE"]

                if self.Parameters.drm_playready:
                    task_tags = task_tags + ["PLAYREADY"]

                if self.Parameters.drm_fairplay:
                    task_tags = task_tags + ['FAIRPLAY']

                task_tags = list(set(task_tags))

                self.server.task[self.id].update({"tags": task_tags})
        except Exception as e:
            logging.error("Unable to set tags: " + str(e))

        if self.is_owner('OTT'):
            self.log_resource.ttl = 365
            self.log_resource.backup_task = True

    def shutdown(self):
        self.pool.close()
        shutil.rmtree(self.temp_dir)

    def retry_session(self, retries=3, backoff_factor=1.0, status_forcelist=(500, 502, 504)):
        session = requests.Session()
        retry = Retry(
            total=retries,
            read=retries,
            connect=retries,
            backoff_factor=backoff_factor,
            status_forcelist=status_forcelist,
            method_whitelist=frozenset(['GET', 'POST'])
        )
        adapter = HTTPAdapter(max_retries=retry)
        session.mount('http://', adapter)
        session.mount('https://', adapter)
        return session

    def download_wget(self, src=None, out=None):
        if src is None:
            src = self.Parameters.url
        if out is None:
            out = self.src_video
        with sdk2.helpers.ProcessLog(self,
                                     logger=logging.getLogger("cmd")) as pl:
            pr = sp.Popen(
                    "wget -t20 --progress=dot:giga -O '{}' '{}'".format(out, src),
                    shell=True,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT
                    )
            pr.wait()
            if pr.returncode != 0:
                msg = 'wget {} exit code {}'.format(src, pr.returncode)
                self.send_status_hook({'warning_message': msg})
                raise TemporaryError(msg)

    def download_youtube_dl(self):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd")) as pl:
            pr = sp.Popen('{} -v -f "(bestvideo[ext=mp4]+bestaudio/best)" --merge-output-format mkv '
                          '--ffmpeg-location {} -o "{}" -i {}{}'.format(
                self.youtube_dl_bin,
                self.ffmpeg_bin,
                self.src_video,
                self.Parameters.url,
                ' --external-downloader wget' if self.is_youtube_dl_url_sni_required() else ''
            ),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT)
            pr.wait()
            if pr.returncode != 0:
                self.Context.youtube_dl_error = pr.returncode
                msg = 'youtube-dl exit code {}'.format(pr.returncode)
                self.send_status_hook({'warning_message': msg})
                raise TemporaryError(msg)
            elif os.path.isfile(self.src_video + '.mkv'):
                    self.src_video += '.mkv'

    def download_video(self):
        self.send_status_hook("VIDEO_DOWNLOADING")

        logging.info("Downloading video")

        url_s3_key = None
        url_s3_bucket = None

        if self.Parameters.url.startswith('s3://'):
            url = self.Parameters.url.split('/', 3)
            url_s3_bucket = url[2]
            try:
                url_s3_key = unquote_plus(str(url[3]))
            except:
                url_s3_key = url[3]
        elif self.Parameters.url.startswith('http://') and '.s3.mds.yandex.net/' in self.Parameters.url:
            url = self.Parameters.url.split('/', 3)
            url_s3_bucket = url[2].split('.')[0]
            try:
                url_s3_key = unquote_plus(str(url[3]))
            except:
                url_s3_key = url[3]
        # https://strm.yandex.ru/kal/fresh/fresh0.m3u8 to s3://strm/fresh/fresh0.m3u8
        elif self.is_strm_hls_url():
            url_s3_bucket = 'strm'
            strm_hls_start, strm_hls_end, url_s3_key = self.get_strm_hls_range()
            if (strm_hls_end - strm_hls_start) <= 0:
                msg = 'start or end get params does not exist'
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)

        if url_s3_bucket and url_s3_key:
            from botocore.exceptions import ClientError
            try:
                self.s3transfer.download_file(url_s3_bucket, url_s3_key, self.src_video)
            except ClientError as e:
                msg = 's3 boto download error code {}'.format(e.response['Error']['Code'])
                self.send_status_hook({'warning_message': msg, 'raw_error': e.response['Error']['Message']})
                if self.Parameters.url.startswith('s3://'):
                    raise TemporaryError(msg)
                else:
                    self.download_wget()
            if self.is_strm_hls_url():
                m3u8_high_base_path = self.get_m3u8_high_base_path(self.src_video)
                m3u8_high_resolution_key = '_'.join(m3u8_high_base_path.split('/')[-1].split('.')[0].split('_')[-2:])
                strm_hls_files = []
                for ts in range(strm_hls_start, strm_hls_end, 10):
                    # fresh/fresh0_169_720p-1513063930000.mp4
                    mp4_s3_key = '{}_{}-{}000.mp4'.format(url_s3_key[:-5], m3u8_high_resolution_key, ts)
                    try:
                        self.embed_s3transfer.download_file(
                            url_s3_bucket,
                            mp4_s3_key,
                            self.sandbox_path(mp4_s3_key.split('/')[-1])
                        )
                    except ClientError as e:
                        logging.warning('s3 boto download error code {}'.format(e.response['Error']['Code']))
                        if e.response['Error']['Code'] == '404':
                            continue
                        else:
                            raise
                    strm_hls_files.append(self.sandbox_path(mp4_s3_key.split('/')[-1]))
                if len(strm_hls_files) > 0:
                    self.src_video = self.stream_concat(strm_hls_files)
                else:
                    msg = 'not found chunks'
                    self.send_status_hook({'error_message': msg})
                    raise common.errors.TaskFailure(msg)
        elif self.is_youtube_dl:
            if 'youtube_dl_error' in self.Context:
                self.Requirements.environments.append(
                    PipEnvironment('youtube-dl').prepare()
                )
            else:
                self.Requirements.environments.append(
                    PipEnvironment('youtube-dl', use_wheel=True).prepare()
                )
            self.download_youtube_dl()
        else:
            self.download_wget()

        logging.info("Downloaded video")

    def eval_frame_rate(self, s):
        numerator, denominator = s.split('/')
        try:
            return float(numerator) / float(denominator)
        except Exception:
            return float(numerator)

    def is_integer_frame_rate(self, s):
        return s.endswith('/1')

    def analyze_video(self):
        self.send_status_hook("METADATA_GETTTING")
        self.get_file_mime_type()
        if not self.Parameters.md5:
            self.get_md5_file()
        self.src_streams = self.get_ffprobe_streams(self.src_video)
        if self.is_owner('OTT') and not self.Parameters.skip_audio_track_requirements:
            self.ensure_ott_src_streams(self.src_streams)
        if not self.Parameters.ffprobe_input_result:
            self.Parameters.ffprobe_input_result = self.src_streams
        self.resolution_src = self.get_resolution()
        if self.Parameters.ffmpeg_crop_black_area:
            self.filter_crop = self.get_ffmpeg_cropdetect()

        if self.is_owner('OTT') and self.is_drm_packaging():
            self.resolution_ratio_key = "169"
        else:
            self.resolution_ratio_key = self.get_nearest_ratios(self.get_display_aspect_ratio())

        self.duration = self.new_duration if self.Parameters.ffpmeg_capture_stop == 0 else self.Parameters.ffpmeg_capture_stop

        video_streams = self.get_video_streams(self.src_streams)
        frame_rate_spec = video_streams[0]['avg_frame_rate']
        self.target_frame_rate = int(math.ceil(self.eval_frame_rate(frame_rate_spec)))


    def get_lang_title(self, lang):
        name = ott_audio_lang_name_dict.get(lang)
        if name is None:
            self.fail_task('Unknown name for lang={}'.format(lang))
        return name

    def get_lang(self, lang_code, handler_name):
        _id = (
            self.normalize_tag(lang_code),
            self.normalize_tag(handler_name)
        )
        logging.info(_id)
        logging.info(ott_audio_lang_dict)

        lang = ott_audio_lang_dict.get(_id)

        if lang:
            return lang
        if self.Parameters.skip_audio_track_requirements or not self.is_owner('OTT'):
            return 'rus'
        self.fail_task('Unknown lang for lang_code={}, handler_name={}'.format(lang_code, handler_name))

    def get_audio_meta(self, audio_name):
        """
        Find audio ffprobe stream meta.

        During the program runtime, audio name can be changed with some prefixes (like: f_, e_, etc)
        This method ignore prefixes and try to find meta information or raise error

        :param audio_name: str, name of audio with prefixes
        :return: dict, ffprobe single stream description
        """
        for key in self.audio_files_meta.keys():
            if key in audio_name:
                return self.audio_files_meta[key]

        raise common.errors.TaskFailure('No meta for audio was found in script for key={}'.format(key))

    def normalize_tag(self, value):
        if value is None:
            return value
        return str(value).decode('utf-8').lower().strip()

    def audio_meta_lang(self, audio_meta):
        meta_lang = get_audio_lang(audio_meta)
        meta_lang_name = None
        if 'tags' in audio_meta and 'handler_name' in audio_meta['tags']:
            meta_lang_name = audio_meta['tags']['handler_name']

        lang = self.get_lang(meta_lang, meta_lang_name)
        lang_name = self.get_lang_title(lang)

        return lang, lang_name

    def audio_template_name(self, file_name):
        meta = self.get_audio_meta(file_name)
        lang, _ = self.audio_meta_lang(meta)
        return 'audio_{lang}_{sample_rate}'.format(
                lang=lang,
                sample_rate=meta['sample_rate'],
                )

    @staticmethod
    def video_stream_meta(file_name):
        # video name [[e_]f_]oscar_169_432p.mp4
        name_parts = file_name.split('_')
        ration = name_parts[-2]
        resolution = name_parts[-1].split('.')[0]  # remove extension
        return ration, resolution

    @staticmethod
    def video_template_name(ration, resolution):
        return 'video_{ratio}_{resolution}'.format(
                ratio=ration,
                resolution=resolution
                )

    def get_ott_audio_codec(self):
        return self.Parameters.ott_audio_codec

    def extract_audios(self):
        """
        Extracts audio stream from video.

        Creates global variables:
            :self.audio_files_original: set of audio names, which may be considered as "original"
                Example:
                    > self.audio_files_original
                    > ('audio_0.mp4')
                Note: len(self.audio_files_original) < all audio streams of content
                Note: self.audio_files_original can be empty
            :self.audio_files_meta: dict audio_name -> meta_information
                Example:
                    > self.audio_files_meta
                    > {'audio_0.mp4' : {'codec_name' : 'aac', ...}}
                Note: meta_information matches ffprobe each stream output

        :return: list of paths of files. Each file = one audio stream
        """

        # Get ffprobe for each audio and save it in ffprobe_stdout
        cmd = "{} -v fatal -of flat=s=_ -print_format json -select_streams a -show_entries " \
                      "stream_tags=language,handler_name:stream=index,codec_name,codec_type,profile,sample_rate,channel_layout '{}'" \
            .format(self.ffprobe_bin, self.src_video)
        pl = self.run_shell(cmd, log_prefix='ffprobe_audio_streams_info')

        ffprobe_stdout = open(str(pl.stdout.path), 'r').read()

        # Checks ffprobe output json, and creates streams list
        stream_json = json.loads(ffprobe_stdout)

        if 'streams' in stream_json:
            streams = stream_json["streams"]
        else:
            msg = 'ffprobe empty json'
            self.send_status_hook({'error_message': msg})
            raise common.errors.TaskFailure(msg)

        ###
        # Logic
        ###

        if self.need_preroll():
            self.run_ffmpeg("{} -y -i '{}' {}".format(
                    self.ffmpeg_bin,
                    self.preroll_file(),
                    "-map 0:a:0 -c {} -b:a 256k -ar 48k {}".format(
                            self.get_ott_audio_codec(),
                            self.sandbox_path(self.preroll_audio_transcoded_name())
                            )
                    ))

        self.audio_files_meta = {}

        prepared_streams = []
        for index, audio_stream in enumerate(streams):
            file_name = 'audio_{}.mp4'.format(index)
            # Save global audio meta
            self.audio_files_meta[file_name] = audio_stream
            prepared_streams.append(dict(audio_stream, index=index, file_name=file_name))

        return list(self.pool.map(self.extract_audio, prepared_streams))


    def loudnorm(self, audio_track_path):
        """
        Loudnorm audio
        Example:
            > loudnorm('../packaged/audio.mp4')
            > '../packaged/loudnorm_audio.mp4'

        :param audio_track_path: str, path to audio_track to apply loudnorm
        :return: str, loudnormed audio track path, with prefix 'loudnorm_'
        """

        pl = self.run_shell(
            "{} -i {} -af loudnorm=I=-23:TP=-2.0:LRA=7:print_format=json -f null /dev/null".format(
                self.ffmpeg_bin,
                self.sandbox_path(audio_track_path)
            ),
            log_prefix='ffmpeg_loudnorm_info'
        )

        ffmpeg_loudnorm_stdout = open(str(pl.stdout.path), 'r').read()
        loudnorm_json = json.loads(ffmpeg_loudnorm_stdout[ffmpeg_loudnorm_stdout.rfind('{'):])

        loudnorm_audio_track_path = "loudnorm_" + audio_track_path

        input_i = loudnorm_json["input_i"]
        # https://st.yandex-team.ru/DIRECT-108722 temporary
        if self.is_owner('CANVAS'):
            input_i = '0.00' if (float(loudnorm_json["input_i"]) > 0) else loudnorm_json["input_i"]
            if (float(input_i) < -99):
                input_i = '-99.00'
        self.run_ffmpeg(
            "{} -i {} -af loudnorm=I=-23:TP=-2.0:LRA=7:" \
            "measured_I={}:measured_TP={}:measured_LRA={}:measured_thresh={}:offset={}:" \
            "linear=true:print_format=summary -c {} -b:a 256k -ar 48k {}".format(
            self.ffmpeg_bin,
            self.sandbox_path(audio_track_path),
            input_i,
            loudnorm_json["input_tp"],
            loudnorm_json["input_lra"],
            loudnorm_json["input_thresh"],
            loudnorm_json["target_offset"],
            self.get_ott_audio_codec(),
            self.sandbox_path(loudnorm_audio_track_path)
        ))

        return loudnorm_audio_track_path

    def extract_audio(self, audio_stream):
        file_name = audio_stream['file_name']
        lang = get_audio_lang(audio_stream)

        logging.info("Found audio track index={}:codec={}:rate={}:lang={} track".format(
            audio_stream["index"],
            audio_stream["codec_name"],
            audio_stream["sample_rate"],
            lang
            ))

        # Extract audio
        self.run_ffmpeg("{} -i '{}' {}".format(
            self.ffmpeg_bin,
            self.src_video,
            "-map 0:a:{} {}".format(
                str(audio_stream['index']),
                self.sandbox_path(file_name)
                )
            ))

        file_name = self.loudnorm(file_name)

        if self.need_preroll():
            file_name_with_preroll = "prerol_" + file_name
            self.run_ffmpeg(
                "{} -i {} -i {} -filter_complex '{}' -map '{}' -c:a '{}' -b:a 256k -ar 48k -metadata:s '{}' {}".format(
                    self.ffmpeg_bin,
                    self.sandbox_path(self.preroll_audio_transcoded_name()),
                    self.sandbox_path(file_name),
                    "[0:0] [1:0] concat=n=2:v=0:a=1 [a]",
                    "[a]",
                    self.get_ott_audio_codec(),
                    "language={}".format(lang),
                    self.sandbox_path(file_name_with_preroll)
                    ))
            file_name = file_name_with_preroll

        return file_name

    def fragment(self, files):
        '''
        Fragment files
        Example:
            > files = ['../packaged/video.mp4', '../packaged/audio.mp4']
            > self.fragment(files)
            > ['../packaged/f_video.mp4', '../packaged/f_audio.mp4']


        :param files: list of paths of files (single file = single stream)
        :return: list of paths of fragmented files
        '''
        files_fragmented = []
        for f in files:
            frag_file_name = "f_" + f
            self.run_fragmenter(
                "{} --verbosity 3 --fragment-duration 4000 --timescale 10000000 {} {}".format(self.mp4fragment_bin, self.sandbox_path(f), self.sandbox_path(frag_file_name)))
            files_fragmented.append(frag_file_name)

        logging.info("files fragmented: " + str(files_fragmented))

        return files_fragmented

    def get_package_content_keys_specs(self, drm, product):
        package_content_keys_spec = []

        video_key = self.content_keys_spec[drm][product]
        package_content_keys_spec.append({
            "kid": video_key['video']['keyId'],
            "type": "video",
            "product": product
        })

        audio_key = self.content_keys_spec[drm][product]
        package_content_keys_spec.append({
            "kid": audio_key['audio']['keyId'],
            "type": "audio",
            "product": product
        })

        return package_content_keys_spec

    def get_kids_keys(self, drm, product):
        kids_keys = []
        if self.content_keys_spec[drm][product]["audio"]:
            kids_keys.append(
                (self.content_keys_spec[drm][product]["audio"]['keyId'], self.content_keys_spec[drm][product]["audio"]['ck']))

        if self.content_keys_spec[drm][product]["video"]:
            kids_keys.append(
                (self.content_keys_spec[drm][product]["video"]['keyId'], self.content_keys_spec[drm][product]["video"]['ck']))

        return kids_keys

    def get_kid(self, type='video'):
        return str(self.content_keys_spec['widevine' if self.Parameters.drm_widevine else 'playready']['hd'][type]['keyId'])

    def get_key(self, type='video'):
        return str(self.content_keys_spec['widevine' if self.Parameters.drm_widevine else 'playready']['hd'][type]['ck'])

    def get_playready_kid(self, type='video'):
        return str(self.content_keys_spec['playready']['hd'][type]['keyId'])

    def get_playready_key(self, type='video'):
        return str(self.content_keys_spec['playready']['hd'][type]['ck'])

    def get_fairplay_kid(self, type='video'):
        return str(self.content_keys_spec['fairplay']['hd'][type]['keyId'])

    def get_fairplay_custom_data(self, key_id, type):
        """See OTT-2083"""
        version = 0
        return ':'.join([
            str(version),
            key_id,
            self.get_drm_content_uuid().replace('-', ''),
            str(self.id),
            'HD',
            type.upper()
            ])

    def get_fairplay_key(self, type='video'):
        return str(self.content_keys_spec['fairplay']['hd'][type]['ck'])

    def get_product_for_backward_compatibility(self):
        return 'hd' # See OTT-5495

    def get_product(self):
        return 'hd' if self.resolution_src["width"] >= 960 and self.resolution_src["height"] >= 720 else 'sd'

    def is_video_of_product(self, file, product):
        width = self.file_resolutions[file][0]
        product_resolutions = filter(lambda r: product in r[1][4] and r[0].startswith(self.resolution_ratio_key), ott_resolutions)
        matches = width in map(lambda r: r[1][0], product_resolutions)
        logging.info("getting product for {}, width={}".format(file, width))
        return matches

    def compute_video_products(self, fragmented_video_file):
        result = []
        for _product in ('hd', 'sd'):
            # Video file starts with `f_` prefix, since it's already segmented
            if self.is_video_of_product(fragmented_video_file[2:], _product):
                result.append(_product)
        return tuple(result)

    def get_selected_drms(self):
        drms = []
        if self.Parameters.drm_widevine:
            drms.append('widevine')
        if self.Parameters.drm_playready:
            drms.append('playready')
        if self.Parameters.drm_fairplay:
            drms.append('fairplay')
        return drms

    def get_prh_version(self):
        return '4.0'

    def encrypt(self, files):
        files_encrypted = []
        for f in files:
            kid = self.get_kid()
            key = self.get_key()
            logging.info("kid: {}, file: {}".format(kid, f))

            pssh = []
            if self.Parameters.drm_widevine:
                pssh.append("edef8ba979d64acea3c827dcd51d21ed:")

            if self.Parameters.drm_playready:
                playready_header = self.create_playready_header(
                    [(self.get_playready_kid(), self.get_playready_key())],
                    self.id,
                    self.get_drm_content_uuid(),
                    self.get_product_for_backward_compatibility(),
                    convert2base64=False,
                    prh_version=self.get_prh_version())
                prh_file = tempfile.NamedTemporaryFile(suffix='.prh', delete=False)
                prh_file.write(playready_header)
                prh_file.close()
                pssh.append("9a04f07998404286ab92e65be0885f95:" + prh_file.name)

            enc_file_name = "e_{}".format(f)
            self.run_encrypter(
                "{} --show-progress --global-option 'mpeg-cenc.piff-compatible:true' --method MPEG-CENC --key 1:{}:random --property 1:KID:{} {} {} {}".format(
                    self.mp4encrypt_bin,
                    key,
                    kid,
                    ' '.join(map(lambda p: "--pssh " + p, pssh)),
                    self.sandbox_path(f),
                    self.sandbox_path(enc_file_name)))
            files_encrypted.append(enc_file_name)

        return files_encrypted

    def create_playready_header(self, kids_keys, batch_id, content_id, product_id, la='https://playready-proxy.ott.yandex.ru/service/la/rightsmanager.asmx', prh_version='4.0', convert2base64=True):
        custom_attrs = "<CONTENT_ID>{}</CONTENT_ID><PRODUCT_ID>{}</PRODUCT_ID><BATCH_ID>{}</BATCH_ID>".format(
            content_id, product_id, batch_id
        )

        if prh_version == '4.0' and len(kids_keys) == 1:
            header_bin = playready.generate_header40(
                "LA_URL:{}#LUI_URL:https://playready-proxy.ott.yandex.ru/web/lui#CUSTOMATTRIBUTES:{}".format(la, custom_attrs.encode('base64')), kids_keys[0][0], kids_keys[0][1])
        elif prh_version == '4.1' and len(kids_keys) == 1:
            header_bin = playready.generate_header41(
                "LA_URL:{}#LUI_URL:https://playready-proxy.ott.yandex.ru/web/lui#CUSTOMATTRIBUTES:{}".format(la, custom_attrs.encode('base64')), kids_keys[0][0], kids_keys[0][1])
        elif prh_version == '4.2':
            header_bin = playready.generate_header42(
                "LA_URL:{}#LUI_URL:https://playready-proxy.ott.yandex.ru/web/lui#CUSTOMATTRIBUTES:{}".format(la, custom_attrs.encode('base64')), kids_keys)
        else:
            raise Exception("Can't generate header {}", prh_version)

        return header_bin.encode('base64').replace('\n', '') if convert2base64 else header_bin

    def parse_ismc(self, ismc):
        fragments = {}
        for stream in ismc.findall('StreamIndex'):
            url = stream.get('Url')
            m = re.match(".*Fragments\(([\w\-]+)=.*", url)
            if m is None:
                msg = "can't parse track name in client manifest"
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)

            track_name = m.group(1)

            logging.info("Fragments found: " + url)

            t = 0
            fragment_mv_commands = []
            for idx, chunk in enumerate(stream.findall('c')):
                fragment_mv_commands.append("mv {} {}".format(
                    self.sandbox_path("packaged/Fragments\(" + track_name + "=" + str(idx + 1) + "\)"),
                    self.sandbox_path("packaged/QualityLevels\({bitrate}\)/Fragments\(" + track_name + "=" + str(t) + "\)")

                ))
                t += int(chunk.get('d'))

            for quality in stream.findall('QualityLevel'):
                bitrate = quality.get('Bitrate')
                fragments[bitrate + "_" + track_name] = fragment_mv_commands

        return fragments

    def split_ism(self, ism, fragments):
        ns = {'smil': 'http://www.w3.org/2001/SMIL20/Language'}
        for track in ism.findall('smil:body/smil:switch/smil:audio', ns) + \
                ism.findall('smil:body/smil:switch/smil:video', ns):
            src = track.get('src')
            bitrate = str(track.get('systemBitrate'))
            param_track_name = track.find('smil:param/[@name="trackName"]', ns)
            track_id = track.find('smil:param/[@name="trackID"]', ns).get('value')
            track_name = param_track_name.get('value') if param_track_name is not None else re.match('\{.*\}(.*)', track.tag).group(1)

            self.run_shell("mkdir -p {}".format(self.sandbox_path("packaged/QualityLevels\({}\)".format(bitrate))))

            self.run_splitter(
                ("{}  --verbose --track-id {} --pattern-parameters N --media-segment {} {}").format(
                    self.mp4split_bin,
                    track_id,
                    self.sandbox_path("'packaged/Fragments(" + track_name + "=%llu)'"),
                    self.sandbox_path("packaged/{}".format(src))
                ))

            frags_key = bitrate + "_" + track_name
            batch_script_name = "mv_{}.sh".format(frags_key)
            with open(batch_script_name, 'w') as batch:
                for mv_cmd in fragments[frags_key]:
                    batch.write(mv_cmd.format(bitrate=bitrate)+'\n')

            self.run_shell("sh {}".format(batch_script_name))

    def ismindex(self, ismc, ism):
        fragments = self.parse_ismc(ismc)
        self.split_ism(ism, fragments)

    def parse_mss_info(self, ismc_path):
        ismc = XML.parse(ismc_path).getroot()
        audio_streams = []

        streams = ismc.findall('StreamIndex[@Type="audio"]')
        default_lang = "rus" if len(streams) == 1 else "und"
        default_lang_title = "русский" if len(streams) == 1 else "undefined"
        index = 0
        for s in streams:
            lang = s.get('Language')
            lang_title = s.get('Name')
            for quality in s.findall('QualityLevel'):
                audio_streams.append(
                    {
                        "language": lang or default_lang,
                        "index": index,
                        "title": lang_title or default_lang_title
                    }
                )
                index += 1

        video_streams = []

        for s in ismc.findall('StreamIndex[@Type="video"]'):
            for quality in s.findall('QualityLevel'):
                video_streams.append(int(quality.get('Bitrate')))

        return {
            "video": video_streams,
            "audio": audio_streams
        }

    @staticmethod
    def parse_xml_duration(duration):
        h, m, s = 0, 0, 0

        pos = 2
        if 'H' in duration:
            h_pos = duration.find('H')
            h = float(duration[pos:h_pos])
            pos = h_pos + 1
        if 'M' in duration:
            m_pos = duration.find('M')
            m = float(duration[pos:m_pos])
            pos = m_pos + 1
        if 'S' in duration:
            s_pos = duration.find('S')
            s = float(duration[pos:s_pos])

        return h * 3600 + m * 60 + s

    def parse_dash_info(self, dash_path, subtitles):
        dash = XML.parse(dash_path).getroot()
        ns = {'mpd': 'urn:mpeg:dash:schema:mpd:2011'}

        duration = self.parse_xml_duration(dash.attrib['mediaPresentationDuration'])
        audio_streams = []

        streams = dash.findall('mpd:Period/mpd:AdaptationSet[@mimeType="audio/mp4"]', ns)
        index = 0
        for s in streams:
            lang = s.get('lang')
            name = s.get('label')
            for r in s.findall('mpd:Representation', ns):
                bandwidth = int(r.get('bandwidth'))
                audio_streams.append(
                    {
                        "language": lang,
                        "index":    index,
                        "title":    name,
                        "bitrate":  bandwidth,
                        "codec":    parse_rfc6381_codec(r.get('codecs')),
                        "size":     int(bandwidth * duration)
                    }
                )
                index += 1

        video_streams = []
        video_index = 0
        for s in dash.findall('mpd:Period/mpd:AdaptationSet[@mimeType="video/mp4"]', ns):

            prop = s.find('mpd:SupplementalProperty[@schemeIdUri="urn:mpeg:yandex:labels:quality"]', ns)
            bandwidth_to_label = self.parse_quality_label(prop.attrib['value'])

            for r in s.findall('mpd:Representation', ns):
                bandwidth = int(r.get('bandwidth'))
                video_streams.append({
                    "index":   video_index,
                    "label":   bandwidth_to_label[bandwidth],
                    "bitrate": bandwidth,
                    "width":   int(r.get('width')),
                    "height":  int(r.get('height')),
                    "deep_hd": self.Parameters.has_superresed,
                    "size":    int(bandwidth * duration),
                    "codec":   parse_rfc6381_codec(r.get('codecs'))
                })
                video_index += 1

        subtitles_streams = []
        index = 0
        for s in dash.findall('mpd:Period/mpd:AdaptationSet[@mimeType="text/vtt"]', ns):
            lang = s.get('lang')
            name = s.get('label')
            for i in range(len(s.findall('mpd:Representation', ns))):
                subtitles_streams.append(
                    {
                        "language": lang,
                        "index":    index,
                        "title":    name,
                        'url':      subtitles[index]['original_url']
                    }
                )
                index += 1

        return {
            "video":     video_streams,
            "audio":     audio_streams,
            "subtitles": subtitles_streams
        }

    def clear_hls_meta(self, hls_files, subtitles):
        for playlist in filter(lambda x: x.endswith('master.m3u8'), hls_files):
            return self.fairplay_meta(playlist, subtitles)
        return None

    def parse_quality_label(self, label_value):
        bandwidth_to_label = {}
        for prop_resolution in label_value.split(','):
            res, label = prop_resolution.split(':')
            _, band = res.split('@')
            bandwidth_to_label[int(band)] = label
        return bandwidth_to_label

    def fairplay_meta(self, playlist_path, subtitles):
        import m3u8
        playlist = m3u8.load(playlist_path)
        bandwidth_to_label = self.parse_quality_label(
            next(s.value for s in playlist.session_data if s.data_id == 'com.yandex.video.labels.quality')
        )

        video_meta = []
        audio_meta = []
        subtitles_meta = []

        for index, audio in enumerate(filter(lambda x: x.type == 'AUDIO', playlist.media)):
            audio_meta.append({
                'index':    index,
                'title':    audio.name,
                'language': audio.language,
                'bitrate': self.package_audio_bitrate(),
                'size':    int(self.new_duration * self.package_audio_bitrate())
            })

        for index, sub in enumerate(filter(lambda x: x.type == 'SUBTITLES', playlist.media)):
            subtitles_meta.append({
                'index':    index,
                'title':    sub.name,
                'language': sub.language,
                'url':      subtitles[index]['original_url']
            })

        for index, video in enumerate(playlist.playlists):
            width, height = video.stream_info.resolution
            bandwidth = video.stream_info.average_bandwidth
            video_meta.append({
                'index':   index,
                'label':   bandwidth_to_label[bandwidth],
                'deep_hd': self.Parameters.has_superresed,
                'bitrate': bandwidth,
                'codec':   parse_rfc6381_codec(video.stream_info.codecs),
                'size':    int(self.new_duration * bandwidth),
                'width':   width,
                'height':  height
            })

        meta = {
            'streams': {
                'audio':    audio_meta,
                'subtitle': subtitles_meta,
                'video':    video_meta,
            }
        }

        if self.timeline_tiles is not None:
            meta['streams']['thumbnail_tiles'] = self.timeline_tiles.get_meta_json()

        return meta

    def subdirs_files(self, dir):
        files_all = []
        for d in filter(lambda f: os.path.isdir(os.path.join(dir, f)), os.listdir(dir)):
            for subdir, _, files in os.walk(os.path.join(dir, d)):
                for file_name in files:
                    files_all.append(os.path.join(os.path.relpath(subdir, dir), file_name))
        return files_all

    def package_ott_hls(self, video_files, audio_files, subtitles):
        """
        Package hls:
            1. Fragment, encrypt and package every single files from video_files + audio_files
                in "packaged/ts-<track_type>-<index>/" dir.
                "packaged/ts-<track_type>-<index>/" contains *.ts encryptes fragments and playlist
                stream.m3u8 with meta encryption info.
                All operations are done with mp42hls bento4 utility
            2. Creates master playlist in "packaged/master.m3u8"
            3. Push all drm data to output

        :param video_files: list[str], not enc segments video files
        :param audio_files: list[str], not enc segmets audio files
        :return: every stream manifest path and one master playlist manifest path for publishing
        """
        logging.info("Create clear hls")
        packaged_file_list = []
        out_dir = self.hls_sb_path()

        ###
        ### Fragment and Package
        ###

        # Audio
        hls_audio_metas = []
        for index, audio_file in enumerate(audio_files):
            # There are best place to get language information about audio
            lang, lang_name = self.audio_meta_lang(self.get_audio_meta(audio_file))

            audio_m3u8_path, _ = self.hls_package(
                    file_path=self.sandbox_path(audio_file),
                    out_dir=out_dir,
                    n=index,
                    template_name_prefix=self.audio_template_name(audio_file)
                    )
            hls_audio_metas.append((audio_m3u8_path, lang, lang_name))
            packaged_file_list.append(audio_m3u8_path)

        # Video
        hls_video_metas = []
        for index, video_file in enumerate(video_files):
            ration, resolution = self.video_stream_meta(video_file)
            video_m3u8_path, video_meta = self.hls_package(
                    file_path=self.sandbox_path(video_file),
                    out_dir=out_dir,
                    n=index,
                    template_name_prefix=self.video_template_name(ration, resolution)
                    )
            video_meta['resolution'] = resolution
            hls_video_metas.append((video_m3u8_path, video_meta))
            packaged_file_list.append(video_m3u8_path)

        ###
        # Create master playlist
        ###

        # Create playlist
        pl_path = self.hls_sb_path('{}'.format('master.m3u8'))
        playlist = PlaylistM3U8(self.sandbox_path(pl_path))

        # Audio
        for index, audio_meta_tuple in enumerate(hls_audio_metas):
            audio_m3u8_path, lang, lang_name = audio_meta_tuple
            playlist.audio(
                    get_relative_path(audio_m3u8_path),
                    name=lang_name,
                    lang=lang,
                    default=index == 0
                    )

        # Subtitles
        for sub in self.get_hls_subtitles(subtitles, out_dir):
            packaged_file_list.append(sub['url'])
            playlist.subtitle(
                    get_relative_path(sub['url'], '/'),
                    name=sub['title'],
                    lang=sub['language']
                    )

        # Video
        for video_m3u8_path, video_meta in hls_video_metas:
            playlist.video(
                    path=get_relative_path(video_m3u8_path),
                    avg_bandwidth=video_meta['stats']['avg_segment_bitrate'],
                    bandwidth=video_meta['stats']['max_segment_bitrate'],
                    codecs=video_meta['video']['codec'],
                    width=video_meta['video']['width'],
                    height=video_meta['video']['height'],
                    skip_audio_track_requirements=self.Parameters.skip_audio_track_requirements
                    )

        # Timeline tiles
        # На Tizen 3 длинный тег session_data ломает нативный плеер, если тег длиннее 230 символов и он идет после
        # X-VERSION. Поэтому пишем session-data в конец мастер плейлиста
        if self.timeline_tiles:
            if self.timeline_tiles.version == 0:
                playlist.session_data(self.timeline_tiles.m3u8_uri_id, self.timeline_tiles.sprite_url_template)
                playlist.session_data(self.timeline_tiles.m3u8_tiles_id, self.timeline_tiles.tiles)
                playlist.session_data(self.timeline_tiles.m3u8_duration_id, self.timeline_tiles.sprite_duration)
                playlist.session_data(self.timeline_tiles.m3u8_resolution_id, self.timeline_tiles.sprite_resolution)
            else:
                for s in self.timeline_tiles.get_hls_session_infos():
                    playlist.session_data(s[0], s[1])

        # DeepHD
        if self.Parameters.has_superresed:
            deephd_video_meta = max(hls_video_metas, key=lambda x: int(x[1]['video']['height']))[1]
            playlist.deephd(
                deephd_video_meta['video']['width'],
                deephd_video_meta['video']['height'],
                deephd_video_meta['stats']['max_segment_bitrate']
            )

        label_quality = []
        for _, video_meta in hls_video_metas:
            name = video_meta['resolution']
            if self.Parameters.has_superresed and video_meta == deephd_video_meta:
                name += ' DeepHD'
            label_quality.append('{width}x{height}@{bitrate}:{name}'.format(
                width=video_meta['video']['width'],
                height=video_meta['video']['height'],
                bitrate=video_meta['stats']['avg_segment_bitrate'],
                name=name
            ))

        playlist.session_data(
            'com.yandex.video.labels.quality',
            ','.join(label_quality)
        )

        playlist_hd_path = playlist.create()
        packaged_file_list.append(playlist_hd_path)

        return packaged_file_list

    def package_fairlay_hls(self, video_files, audio_files, subtitles):
        """
        Package fairplay hls:
            1. Fragment, encrypt and package every single files from video_files + audio_files
                in "packaged/ts-<track_type>-<index>/" dir.
                "packaged/ts-<track_type>-<index>/" contains *.ts encryptes fragments and playlist
                stream.m3u8 with meta encryption info.
                All operations are done with mp42hls bento4 utility
            2. Creates master playlist in "packaged/master.m3u8"
            3. Push all drm data to output

        :param video_files: list[str], not enc segments video files
        :param audio_files: list[str], not enc segmets audio files
        :return: every stream manifest path and one master playlist manifest path for publishing
        """
        packaged_file_list = []
        output_info = []
        out_dir = self.hls_sb_path()

        ###
        ### Encrypt and Package
        ###

        # Audio
        hls_audio_metas = []
        for index, audio_file in enumerate(audio_files):
            # There are best place to get language information about audio
            audio_meta = self.get_audio_meta(audio_file)
            lang, lang_name = self.audio_meta_lang(audio_meta)

            audio_kid = self.get_fairplay_kid(type='audio')

            audio_m3u8_path, _ = self.hls_package(
                file_path=self.sandbox_path(audio_file),
                out_dir=out_dir,
                n=index,
                template_name_prefix="fp-{}".format(self.audio_template_name(audio_file)),
                kid=self.get_fairplay_custom_data(audio_kid, 'audio'),
                ck=self.get_fairplay_key(type='audio')
            )

            hls_audio_metas.append((audio_m3u8_path, lang, lang_name))
            packaged_file_list.append(audio_m3u8_path)
            output_info.append({
                "product": self.get_product(),
                "drm": "fairplay",
                "url": self.get_fairplay_s3_path(audio_m3u8_path),
                "type": 'audio',
                "kid": audio_kid,
                "meta": audio_meta
            })


        # Video
        hls_video_metas = []
        for index, video_file in enumerate(video_files):
            video_kid = self.get_fairplay_kid(type='video')
            ration, resolution = self.video_stream_meta(video_file)
            video_m3u8_path, video_meta = self.hls_package(
                file_path=self.sandbox_path(video_file),
                out_dir=out_dir,
                n=index,
                template_name_prefix="fp-{}".format(self.video_template_name(ration, resolution)),
                kid=self.get_fairplay_custom_data(video_kid, 'video'),
                ck=self.get_fairplay_key(type='video')
            )
            video_meta['resolution'] = resolution
            hls_video_metas.append((video_m3u8_path, video_meta))
            packaged_file_list.append(video_m3u8_path)

            output_info.append({
                "product": self.get_product(),
                "drm": "fairplay",
                "url": self.get_fairplay_s3_path(video_m3u8_path),
                "type": 'video',
                "kid": video_kid,
                "hls_meta": video_meta
            })

        ###
        # Create master playlist
        ###
        playlist = PlaylistM3U8(
            self.sandbox_path(
                self.hls_sb_path('master.{}.m3u8'.format(self.get_product()))
            )
        )

        # Audio
        for index, audio_meta_tuple in enumerate(hls_audio_metas):
            audio_m3u8_path, lang, lang_name = audio_meta_tuple
            playlist.audio(
                get_relative_path(audio_m3u8_path),
                name=lang_name,
                lang=lang,
                default=index == 0
            )
            playlist.add_session_key(self.parse_ext_x_session_key(audio_m3u8_path, out_dir))

        # Subtitles
        for sub in self.get_hls_subtitles(subtitles, out_dir):
            playlist.subtitle(
                    get_relative_path(sub['url'], '/'),
                    name=sub['title'],
                    lang=sub['language']
                    )

        # Video
        for video_m3u8_path, video_meta in hls_video_metas:
            playlist.video(
                    path=get_relative_path(video_m3u8_path),
                    avg_bandwidth=video_meta['stats']['avg_segment_bitrate'],
                    bandwidth=video_meta['stats']['max_segment_bitrate'],
                    codecs=video_meta['video']['codec'],
                    width=video_meta['video']['width'],
                    height=video_meta['video']['height'],
                    skip_audio_track_requirements=self.Parameters.skip_audio_track_requirements
                    )
            playlist.add_session_key(self.parse_ext_x_session_key(video_m3u8_path, out_dir))

        # Timeline tiles
        # На Tizen 3 длинный тег session_data ломает нативный плеер, если тег длиннее 230 символов и он идет после
        # X-VERSION. Поэтому пишем session-data в конец мастер плейлиста
        if self.timeline_tiles:
            if self.timeline_tiles.version == 0:
                playlist.session_data(self.timeline_tiles.m3u8_uri_id, self.timeline_tiles.sprite_url_template)
                playlist.session_data(self.timeline_tiles.m3u8_tiles_id, self.timeline_tiles.tiles)
                playlist.session_data(self.timeline_tiles.m3u8_duration_id, self.timeline_tiles.sprite_duration)
                playlist.session_data(self.timeline_tiles.m3u8_resolution_id, self.timeline_tiles.sprite_resolution)
            else:
                for s in self.timeline_tiles.get_hls_session_infos():
                    playlist.session_data(s[0], s[1])

        deephd_video_meta = max(hls_video_metas, key=lambda x: int(x[1]['video']['height']))[1]
        label_quality = []
        for _, video_meta in hls_video_metas:
            name = video_meta['resolution']
            if self.Parameters.has_superresed and video_meta == deephd_video_meta:
                name += ' DeepHD'
            label_quality.append('{width}x{height}@{bitrate}:{name}'.format(
                width=video_meta['video']['width'],
                height=video_meta['video']['height'],
                bitrate=video_meta['stats']['avg_segment_bitrate'],
                name=name
            ))

        playlist.session_data(
            'com.yandex.video.labels.quality',
            ','.join(label_quality)
        )

        playlist_path = playlist.create()
        packaged_file_list.append(playlist_path)
        self.drm_package.append({
                 "product": self.get_product(),
                 "url": self.get_fairplay_s3_path(playlist_path),
                 "keys": get_fairplay_keys(output_info, 'hd'),
                 "meta": self.fairplay_meta(playlist_path, subtitles),
                 "drm": "fairplay",
                 "type": "HLS"
            })
        output_info.append({'master': self.get_fairplay_s3_path(playlist_path)})

        self.Parameters.fairplay_package = json.dumps(output_info)

        return packaged_file_list


    def clear_hls_hook(self, hls_files, subtitles):
        self.send_status_hook({
            'status': 'VIDEO_FULL_UPLOADED',
            'stream': json.dumps(self.resolution_src) if self.resolution_src else '',
            'video_descriptor_id': self.video_descriptor_id(),
            'stream_url': self.m3u8_adaptive_url,
            'meta': self.clear_hls_meta(hls_files, subtitles)
        }, timeout=5)


    def package_clear_hls(self, video_files, audio_files, subtitles):
        hls_to_publish = self.package_ott_hls(
            video_files,
            audio_files,
            subtitles
        )
        self.publish_ott_hls_files()
        self.Parameters.clear_hls_package = self.clear_hls_meta(hls_to_publish, subtitles)
        return hls_to_publish


    def parse_ext_x_session_key(self, m3u8_path, out_dir):
        with open(get_absolute_path(m3u8_path, out_dir), 'r') as submanifest:
            for line in submanifest:
                if line.startswith('#EXT-X-KEY:'):
                    return line.replace('EXT-X-KEY', 'EXT-X-SESSION-KEY')
        self.fail_task(
            '#EXT-X-KEY tag was not found in {}, function was called in non-drm scenario'.format(m3u8_path)
        )

    def compute_supplemental_property(self, video_files):
        tracks_properties = []
        temp_dash = tempfile.NamedTemporaryFile(suffix='.mpd').name
        self.mp4dash(
            output_dir=os.path.dirname(temp_dash),
            mpd_name=os.path.basename(temp_dash),
            video_files=[self.sandbox_path(f) for f in video_files]
        )

        dash = XML.parse(temp_dash).getroot()
        ns = {'mpd': 'urn:mpeg:dash:schema:mpd:2011'}
        representations = dash.findall('mpd:Period/mpd:AdaptationSet[@mimeType="video/mp4"]/mpd:Representation', ns)
        if len(representations) != len(video_files):
            raise ValueError('Temp dash contains not all video metadata')
        representations_iter = iter(representations)

        for video_file in video_files:
            _, resolution = self.video_stream_meta(video_file)
            stream = next(representations_iter)
            tracks_properties.append({
                'width': int(stream.get('width')),
                'height': int(stream.get('height')),
                'bandwidth': int(stream.get('bandwidth')),
                'resolution': resolution
            })

        tracks_properties = sorted(
            tracks_properties,
            key=lambda x: x['width'],
            reverse=True
        )

        qualities = [
            '{width}x{height}@{bandwidth}:{resolution}'.format(
                width=track['width'],
                height=track['height'],
                bandwidth=track['bandwidth'],
                resolution=track['resolution']
            ) for track in tracks_properties
        ]
        video_properties = {
            'urn:mpeg:yandex:labels:quality': ','.join(qualities)
        }

        if self.Parameters.has_superresed:
            video_properties['urn:mpeg:yandex:labels:deephd'] = qualities[0].split(':')[0]

        return '#' + json.dumps({'video': video_properties}).encode('base64')[:-1]

    def package(self, video_files, audio_files, subtitles, types, manifest_only=False):
        packaged_output_dir = self.sandbox_path("packaged")

        mss = 'playready' in types or 'mss' in types
        drm_playready = 'playready' in types
        mss_client_manifest = 'manifest.ismc' if mss else None
        mss_server_manifest = 'server.ism' if mss else None
        dash_manifest_name = 'manifest.mpd'

        playready_header = self.create_playready_header(
                [(self.get_playready_kid(), self.get_playready_key())],
                self.id,
                self.get_drm_content_uuid(),
                self.get_product_for_backward_compatibility()
                ) if 'playready' in types else None
        widevine_header = 'provider:{}#content_id:{}'.format(
                self.Parameters.drm_widevine_header_provider,
                self.get_drm_widevine_content_id()
                ) if 'widevine' in types else None

        create_dash = True  # For now always create dash manifest

        # add thumbnails info if required
        timeline_sprites = None
        timeline_sprites_lowres = None
        if self.timeline_tiles and self.Parameters.ffmpeg_timeline_tiles_dash:
            timeline_sprites = self.timeline_tiles.mpd_session_data
            if self.timeline_tiles.version > 0:
                timeline_sprites_lowres = self.timeline_tiles.mpd_session_data_lowres

        return self.mp4dash(
                packaged_output_dir,
                video_files=map(lambda f: "{}".format(self.sandbox_path(f)), video_files),
                audio_files=map(lambda f: (self.audio_meta_lang(self.get_audio_meta(f)) + (self.sandbox_path(f), )), audio_files),
                subtitles=map(lambda s: (s['id'], s['title'], 'webvtt', s['language'], self.sandbox_path(s['url'])), subtitles),
                verbose=True,
                manifest_only=manifest_only,
                force=True,
                mss=mss,
                mss_client_manifest=mss_client_manifest,
                mss_server_manifest=mss_server_manifest,
                playready=drm_playready,
                playready_header=playready_header,
                mpd_name=dash_manifest_name,
                dash=create_dash,
                widevine_header=widevine_header,
                timeline_sprites=timeline_sprites,
                timeline_sprites_lowres=timeline_sprites_lowres,
                supplemental_property=self.compute_supplemental_property(video_files),
                skip_subtitles_language_conversion=True
                )

    def parse_drm_package_params(self, files, drms, path_function, subtitles):
        dash_manifests = filter(lambda x: x.endswith('.mpd'), files)
        mss_manifests = filter(lambda x: x.endswith('.ismc'), files)
        result = []
        for dash in dash_manifests:
            parsed_dash = self.parse_dash_info(dash, subtitles)
            dash_drms = filter(lambda x: x in ('widevine', 'playready'), drms)
            for drm in dash_drms:
                result.append({
                    'product': self.get_product_for_backward_compatibility(),
                    'video_descriptor_id': self.to_video_descriptor_id(self.get_product()),
                    'drm': drm,
                    'url': path_function(dash),
                    'type': 'dash',
                    'keys': self.get_package_content_keys_specs(drm, self.get_product_for_backward_compatibility()),
                    'meta': {'streams': parsed_dash}
                })
        for mss in mss_manifests:
            parsed_mss = self.parse_mss_info(mss)
            result.append({
                'product': self.get_product_for_backward_compatibility(),
                'video_descriptor_id': self.to_video_descriptor_id(self.get_product()),
                'drm': 'playready',
                'url': path_function(mss),
                'type': 'mss',
                'keys': self.get_package_content_keys_specs('playready', self.get_product_for_backward_compatibility()),
                'meta': {'streams': parsed_mss}
            })

        return result

    def publish_files_fast(self, files_fast):
        logging.info("Finished fast converting")
        self.send_status_hook("VIDEO_FAST_UPLOADING")
        if 'hls' in self.get_output_format():
            logging.info("Upload fast video ts to S3")
            for file in files_fast:
                if file.split('.')[-1] == 'm3u8':
                    self.upload(self.get_ts_files(file))
                    self.append_adaptive_m3u8(file)
            self.upload(files_fast)
            if self.timeline_tiles:
                logging.info("Upload fast timeline preview to S3")
                self.publish_timeline_tiles()
            self.upload([self.m3u8_adaptive_file])
            self.send_status_hook({
                'status': 'VIDEO_FAST_UPLOADED',
                'stream_url': self.m3u8_adaptive_url,
                'formats': [self.get_format(f) for f in files_fast if f.split('.')[-1] != 'm3u8']
            }, timeout=5)
        else:
            self.upload(files_fast)
            self.send_status_hook({'status': 'VIDEO_FAST_UPLOADED',
                                   'formats': [self.get_format(f) for f in files_fast if f.split('.')[-1] != 'm3u8']},
                                    timeout=5)

    def publish_ott_hls_files(self):
        self.send_status_hook("VIDEO_UPLOADING")
        logging.info("Upload hls ts and m3u8 to S3")
        # Only .m3u8 for streams (exclude master playlist) and subs
        to_upload = []
        for root, dirnames, filenames in os.walk(self.sandbox_path('packaged')):
            for filename in filenames:
                if filename.endswith(('.m3u8', '.ts', '.vtt')):
                    to_upload.append(
                            self.S3File(
                                os.path.join(root, filename),
                                self.s3_dir,
                                self.Parameters.s3_bucket,
                                self.s3_url
                            )
                        )
        self.upload(to_upload)

    def publish_dash_mss_files(self, files):
        self.send_status_hook("VIDEO_UPLOADING")
        return self.upload(
              map(
                  lambda f: self.S3File(
                        f,
                        os.path.join(self.Parameters.s3_dir, self.get_drm_package_path()),
                        self.Parameters.s3_bucket,
                        self.s3_url
                      ),
                  files
                  )
              )

    def publish_internal_mp4_files(self, files):
        try:
            return self.upload(
                    map(
                        lambda f: self.S3File(
                            f,
                            os.path.join(
                                'mp4',
                                self.Parameters.s3_dir,
                                self.get_drm_content_id(),
                                str(self.id)
                                ),
                            self.Parameters.s3_internal_bucket,
                            self.s3_url
                        ),
                    files
                    )
                )
        except Exception as e:
            logging.info("Publish internal mp4 error {}".format(e.__class__.__name__))
            logging.error(traceback.format_exc())
        return []

    def publish_ott_trailer_files(self, files):
        self.send_status_hook("VIDEO_UPLOADING")
        to_upload = []

        for _file in filter(lambda f: file_ends_with(f, 'm3u8'), files):
            to_upload.extend(self.get_ts_files(_file))
            to_upload.append(_file)
            self.append_adaptive_m3u8(_file)
        self.append_adaptive_m3u8()
        to_upload.append(self.m3u8_adaptive_file)

        self.upload(
            map(
                lambda f: self.S3File(
                    f,
                    os.path.join(
                        self.Parameters.s3_dir,
                        str(self.id) + '-' + self.get_drm_content_id()
                    ),
                    self.Parameters.s3_bucket,
                    self.s3_url
                ),
                to_upload
            )
        )

        self.send_status_hook({
            'status':     'VIDEO_FULL_UPLOADED',
            'stream':     json.dumps(self.resolution_src) if self.resolution_src else '',
            'stream_url': self.m3u8_adaptive_url,
            'formats':    json.dumps(self.get_formats_files(files)),
            'video_descriptor_id': self.video_descriptor_id()
        }, timeout=5)

    def publish_files(self, files, files_fast):
        self.send_status_hook("VIDEO_UPLOADING")
        files_m3u8 = [f for f in files if f.split('.')[-1] == 'm3u8']
        if len(files_m3u8) > 0:
            logging.info("Upload ts to S3")
            for file in files_m3u8:
                self.upload(self.get_ts_files(file))
                self.append_adaptive_m3u8(file)
            logging.info("Upload m3u8 to S3")
            self.append_adaptive_m3u8()
            files.append(self.m3u8_adaptive_file)

        self.upload(files)
        if len(self.files_awaps) > 0:
            self.upload([f for f in self.files_awaps if f not in files])

        files.extend(files_fast)

        if not self.is_drm_packaging() and not self.is_mss_packaging():
            self.send_status_hook({
                'status': 'VIDEO_FULL_UPLOADED',
                'stream': json.dumps(self.resolution_src) if self.resolution_src else '',
                'stream_url': self.m3u8_adaptive_url if len(files_m3u8) > 0 else '',
                'formats': json.dumps(self.get_formats_files(files)),
                'video_descriptor_id': self.video_descriptor_id()
            }, timeout=5)

        return (files_m3u8)

    def publish_thumbnails(self):
        logging.info("Uploading thumbnails to S3")
        thumbnails = sorted([
                self.S3File(
                    os.path.basename(x),
                    self.get_s3_dir('thumbs'),
                    self.Parameters.s3_bucket,
                    self.s3_url
                )
                for x in
                glob.glob("{prefix}*.jpg".format(prefix=self.get_thumbs_path_prefix()))
            ])
        self.upload(thumbnails)
        self.send_status_hook({'status': 'THUMBNAILS_UPLOADED',
                               'thumbnail_urls': json.dumps([x.url for x in thumbnails])})
        return thumbnails

    def publish_timeline_tiles(self):
        logging.info("Uploading timeline tiles to S3")
        # collect tiles files with relative path from temp_dir
        files = sorted([x.split(self.temp_dir)[1].lstrip('/')
                        for x in glob.glob(
                            "{prefix}*.jpg".format(prefix=self.timeline_tiles.absolute_sprite_prefix)
                        )
        ])
        if not files:
            raise TemporaryError('timeline tiles are empty')

        if self.timeline_tiles.version == 0:
            self.convert_to_progressive_jpg(files)
        else:  # remove jpeg tran from new versions
            # load lowres files and mix with hires files
            files_set = set(files)
            lowres = [x.split(self.temp_dir)[1].lstrip('/')
                            for x in glob.glob(
                    "{prefix}*.jpg".format(prefix=self.timeline_tiles.absolute_sprite_prefix)
                )]
            files_set.update(lowres)
            files = sorted(list(files_set))

        self.upload([self.timeline_tiles.get_sprite_s3(x.split('/')[-1]) for x in files])

        if self.timeline_tiles.version == 0:
            # create timeline screens only for 0 version
            logging.info("Uploading timeline screenshots to S3")
            screen_files = sorted([x.split(self.temp_dir)[1].lstrip('/')
                                   for x in glob.glob(
                    "{prefix}*.jpg".format(prefix=self.timeline_tiles.absolute_screen_prefix)
            )])

            self.convert_to_progressive_jpg(screen_files)
            self.upload([self.timeline_tiles.get_screen_s3(x.split('/')[-1]) for x in screen_files])
        else:
            screen_files = []

        return files, screen_files

    def get_thumbs_path_prefix(self):
        return '{tmp}/{prefix}_thumb'.format(
            tmp=self.temp_dir,
            prefix=self.Parameters.s3_key_prefix,
        )

    def get_s3_dir(self, inside_dir):
        s3_dir = self.Parameters.s3_dir
        if self.is_ott_hls() or self.is_drm_packaging() or self.is_mss_packaging() or self.is_ott_trailer():
            s3_dir = os.path.join(s3_dir, self.get_drm_content_id(), inside_dir, str(self.id))
        return s3_dir

    def send_drm_error(self, subj, body):
        if self.Parameters.production:
            self.server.notification(
                subject=subj,
                body=body,
                recipients=['tabroot', 'rikunov', 'baltor'],
                transport=ctn.Transport.EMAIL,
                urgent=True
            )

    def retrieve_rt_video(self, task_id):
        logging.info('Block task until rt is not ready with task_id=%s', task_id)
        rt_response = self.rt_client.retrieve_video(task_id)
        logging.info('RT success response for task_id=%s, %s', task_id, rt_response)
        files = list(
            map(
                    operator.itemgetter('Url'),
                    rt_response['Streams']
                )
            )
        self.pool.map(
            lambda url: self.download_wget(url, self.sandbox_path(os.path.basename(url))),
            files
            )
        return list(map(os.path.basename, files))

    def convert_to_progressive_jpg(self, files):
        """
        Convert given files to progressive jpeg using jpegtran utility
        :param files: list of file names with relative path
        """
        logging.info("Convert files to progressive jpg")
        for f in files:
            self.run_shell(
                '{bin} -progressive -outfile {file} {file}'.format(
                    bin=self.jpegtran_bin,
                    file=get_absolute_path(f, self.temp_dir),
                ),
                log_prefix='jpegtran.',
            )

    def require_keys(self):
        keys = []
        for drm in self.get_selected_drms():
            key = {'drm': drm}
            key['keys'] = []
            key['keys'].append({
                'product': 'hd',
                'type': 'video'
            })
            key['keys'].append({
                'product': 'hd',
                'type': 'audio'
            })
            keys.append(key)

        logging.info("Require keys: {}".format(keys))

        drm_list = list(set([k['drm'] for k in keys]))
        description = 'Sandbox(drms: {}; products: {};)'.format(
                ', '.join(drm_list),
                ', '.join(['hd'])
        )

        data = {
              'serviceId': 10,
              'kekId': self.drm_kek_id,
              'kek': self.drm_kek,
              'description': description,
              'batchId': self.id,
              'reuseOldKeys': False,
              'contentUuid': self.get_drm_content_id(),
              'contentKeys': keys
        }

        generate_keys_start_time = time.time()
        try:
            s = self.retry_session()
            response = s.post(
                self.get_mediakeys_url(),
                data=json.dumps(data),
                headers={'Content-Type': 'application/json'},
                timeout=(5, 500),
                verify=False
            )

            status_code = response.status_code

            if status_code != 201:
                self.send_drm_error(
                    'Content keys generation failed with status {} for Sandbox task {}'.format(response.status_code, self.id),
                    "MediaKeys API response: {}".format(response)
                )
                raise TemporaryError("MediaKeys API response: {}".format(response))

            json_keys = response.json()['keys']
            drms = set([content_key['drmType'] for content_key in json_keys])
            self.content_keys_spec = {}
            for drm in drms:
                self.content_keys_spec[drm] = {}
                current_drm_keys = [content_key for content_key in json_keys if content_key['drmType'] == drm]
                products = set([content_key['product'] for content_key in current_drm_keys])
                for product in products:
                    self.content_keys_spec[drm][product] = {}
                    current_product_keys = [content_key for content_key in current_drm_keys if content_key['product'] == product]
                    tracks = set([content_key['type'] for content_key in current_product_keys])
                    for track in tracks:
                        track_key = [content_key for content_key in current_product_keys if content_key['type'] == track][0]
                        self.content_keys_spec[drm][product][track] = {
                            'keyId': track_key['keyId'],
                            'ck': track_key['ck'],
                        }
        except Exception as e:
            self.send_drm_error(
                "MediaKeys API request failed",
                "Sandbox task {} failed: {}".format(self.id, str(e))
            )
            logging.info("Require keys failed: {}".format(e.__class__.__name__))
            logging.error(traceback.format_exc())
            raise TemporaryError("Require keys error: " + str(e))
        else:
            logging.info("Require keys done with status: {}".format(status_code))
        finally:
            logging.info("Require keys took {} sec".format(time.time() - generate_keys_start_time))

    def loudnorm_probe(self):
        if self.audio_stream:
            pl = self.run_shell(
                "{} -i {} -map 0:a:0 -af loudnorm=I=-23:TP=-2.0:LRA=7:print_format=json -f null /dev/null".format(
                    self.ffmpeg_bin,
                    self.sandbox_path(self.src_video)
                ),
                log_prefix='loudnorm.probe'
            )

            loudnorm_probe = open(str(pl.stdout.path), 'r').read()
            loudnorm_probe_json = json.loads(loudnorm_probe[loudnorm_probe.rfind('{'):])

            if loudnorm_probe_json['input_i'] == '-inf'\
                    or loudnorm_probe_json["input_tp"] == '-inf' \
                    or loudnorm_probe_json["target_offset"] == 'inf':
                self.Context.loudnorm_filter = ''
            else:
                input_i = loudnorm_probe_json["input_i"]
                # https://st.yandex-team.ru/DIRECT-108722 temporary
                if self.is_owner('CANVAS'):
                    input_i = '0.00' if (float(loudnorm_probe_json["input_i"]) > 0) else loudnorm_probe_json["input_i"]
                    if (float(input_i) < -99):
                        input_i = '-99.00'

                self.Context.loudnorm_filter = \
                    "loudnorm=I=-23:TP=-2.0:LRA=7:" \
                    "measured_I={}:measured_TP={}:measured_LRA={}:measured_thresh={}:offset={}:" \
                    "linear=true:print_format=summary".format(
                        input_i,
                        loudnorm_probe_json["input_tp"],
                        loudnorm_probe_json["input_lra"],
                        loudnorm_probe_json["input_thresh"],
                        loudnorm_probe_json["target_offset"]
                    )

    def check_parameters(self):
        if self.Parameters.drm_widevine and not self.Parameters.drm_playready:
            self.fail_task("Widevine-only packaging has been disabled, please package DASH-CENC Widevine and PlayReady")

    def ffmpeg_cmd_prepare_format(self, cmd):
        for r in [(self.temp_dir, '{temp_dir}'), (self.ffmpeg_bin, '{ffmpeg_bin}'), (self.src_video, '{src_video}')]:
            cmd = cmd.replace(r[0], r[1])
        return cmd

    def transcode_segment(self):
        """
        video streams segment transcode:
            1. Get SB resource FFMPEG_BIN
            2. Get segment SB resource from parent task
            3. Run ffmpeg segment encoding command from parent task
            4. Make SB resource encoded segment and save to output param resource_id
        """
        ffmpeg = self.download_resource(resource_types.FFMPEG_BIN)
        ffmpeg_bin = str(ffmpeg.path / "ffmpeg")
        skynet_get(self.Parameters.url, '.')
        segments_resource = resource_types.OTHER_RESOURCE(
            self,
            'output video segments',
            os.path.splitext(self.Parameters.parallel_parent['segment'])[0]
        )
        segments_resource_data = sdk2.ResourceData(segments_resource)
        segments_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        ffmpeg_cmds = json.loads(self.Parameters.parallel_parent['ffmpeg_cmds'])
        for cmd in ffmpeg_cmds:
            ffmpeg_cmd = cmd.format(ffmpeg_bin=ffmpeg_bin,
                                    src_video=self.Parameters.parallel_parent['segment'],
                                    temp_dir=segments_resource.path)
            logging.info(ffmpeg_cmd)
            self.run_ffmpeg(ffmpeg_cmd)
        self.Parameters.resource_id = segments_resource
        segments_resource_data.ready()

    def on_execute(self):
        if self.Parameters.sb_parallel_encoding and self.Parameters.url.startswith('rbtorrent:'):
            self.transcode_segment()
            return

        self.check_parameters()
        self.configure()

        if self.Parameters.rt_parallel_encoding:
            try:
                rt_task_id = self.rt_client.submit_video(
                    self.src_http_url(),
                    self.get_segment_time(),
                    self.Parameters.ffpmeg_encoding_preset,
                    self.Parameters.keep_aspect_ratio,
                    '169',  # only 16x9 for now supported
                    self.get_resolutions()
                    )
            except RTClient.RTSubmitError as e:
                logging.error(traceback.format_exc())
                raise TemporaryError(e)
            logging.info('RT video send to task_id %s', rt_task_id)

        self.download_video()
        self.analyze_video()

        if self.need_loudnorm():
            with self.memoize_stage["loudnorm_probe"]:
                self.loudnorm_probe()

        if self.is_drm_packaging():
            self.require_keys()

        if self.Parameters.ffmpeg_timeline_tiles \
                and not self.is_strm_embed() \
                and not self.Parameters.rt_parallel_encoding:
            key_prefix = '{}_'.format(self.Parameters.s3_key_prefix)
            # put tiles as other OTT content
            if self.is_ott_hls() or self.is_drm_packaging() or self.is_mss_packaging() or self.is_ott_trailer():
                key_prefix = ''

            if self.Parameters.ffmpeg_timeline_tiles_version in ['v1', 'v2']:
                logging.info('Initialize timeline tiles %s', self.Parameters.ffmpeg_timeline_tiles_version)
                self.timeline_tiles = self.TimelineTilesV1(
                    total_duration=self.duration,
                    aspect_ratio=self.get_display_aspect_ratio(),
                    temp_dir=self.temp_dir,
                    key_prefix=key_prefix,
                    add_to_dash=self.Parameters.ffmpeg_timeline_tiles_dash,
                    s3_url=self.s3_url,
                    s3_bucket=self.Parameters.s3_bucket,
                    strm_host=self.strm_host,
                    sprite_dir=self.get_s3_dir('sprites'),
                    screen_dir=self.get_s3_dir('screens'),
                    fps=self.get_target_frame_rate(),
                    version=1 if self.Parameters.ffmpeg_timeline_tiles_version == 'v1' else 2,
                )
            else:
                logging.info('Initialize timeline tiles V0 (old version)')
                self.timeline_tiles = self.TimelineTilesV0(
                    total_duration=self.duration,
                    aspect_ratio=self.get_display_aspect_ratio(),
                    temp_dir=self.temp_dir,
                    key_prefix=key_prefix,
                    add_to_dash=self.Parameters.ffmpeg_timeline_tiles_dash,
                    s3_url=self.s3_url,
                    s3_bucket=self.Parameters.s3_bucket,
                    strm_host=self.strm_host,
                    sprite_dir=self.get_s3_dir('sprites'),
                    screen_dir=self.get_s3_dir('screens'),
                )
        else:
            self.timeline_tiles = None

        self.files_awaps = []
        files_fast = []
        files_m3u8 = []
        thumbnails = []
        tiles_files = []
        screen_files = []
        subtitles = self.get_subtitles()

        self.first_resolution = None
        if self.Parameters.ffpmeg_first_resolution and self.Parameters.ffpmeg_thumbnails_t > 0:
            select = 'gte(t\,{})*isnan(prev_selected_t)'.format(self.Parameters.ffpmeg_thumbnails_t)
            cmdline = "{bin} -i {input} -vf select='{select}' -qscale:v 1 -vsync vfr {prefix}%03d.jpg".format(
                bin=self.ffmpeg_bin,
                input=self.src_video,
                select=select,
                prefix=self.get_thumbs_path_prefix()
            )
            logging.info("Running ffmpeg fast thumbnail command: {}".format(cmdline))
            self.run_ffmpeg(cmdline)
            thumbnails = self.publish_thumbnails()

        if self.Parameters.ffpmeg_stream_loop > 0:
            self.src_video = self.stream_loop(self.Parameters.ffpmeg_stream_loop)

        if self.Parameters.ffpmeg_first_resolution and len(self.get_resolutions_ratio_adaptive()) > 1 and not not self.Parameters.rt_parallel_encoding:
            files_fast = self.transcode_fast()

        try:
            resolutions_ratio_adaptive = self.get_resolutions_ratio_adaptive()
            if self.owner == 'ZEN' and self.src_mime_type == 'image/gif' and len(resolutions_ratio_adaptive) == 0:
                resolutions_ratio_adaptive = (("169_144p", ("256", "114", "200", "32")),)

            # https://st.yandex-team.ru/BANNERSTORAGE-5697 temporary
            if self.is_owner('BANNERSTORAGE') and not self.Parameters.has_superresed:
                max_out_height = max(map(lambda x: int(x[1][1]), resolutions_ratio_adaptive))
                if max_out_height < 720:
                    resolutions_ratio_adaptive = tuple(
                    (k, r) for k, r in self.get_resolutions() if self.resolution_ratio_key + "_" in k and int(r[1]) <= 720
                )

            if len(resolutions_ratio_adaptive) > 0:
                files = self.transcode(resolutions_ratio_adaptive, self.Parameters.ffmpeg_two_pass)
                targets = {}
                if self.Parameters.sb_parallel_encoding:
                    # Start parallel encoding segments in subtasks
                    with self.memoize_stage.set_sb_parallel_encoding_tasks(commit_on_entrance=False):
                        for segment in self.run_ffmpeg_splitter():
                            segment_name = os.path.basename(segment)
                            logging.info('Create subtask for {}'.format(segment_name))
                            segment_resource = resource_types.OTHER_RESOURCE(self, 'input video segment', segment)
                            sdk2.ResourceData(segment_resource).ready()
                            parent_context = {
                                'parent_id': self.id,
                                'segment_resource_id': segment_resource.id,
                                'segment': segment_name,
                                'ffmpeg_cmds': [self.ffmpeg_cmd_prepare_format(cmd) for cmd in self.Context.ffmpeg_cmds]
                            }
                            subtask = StrmVideoConvert(
                                self,
                                owner=self.owner,
                                url = segment_resource.skynet_id,
                                description="Child of {} task for segment {}".format(self.id, segment_name),
                                sb_parallel_encoding = True,
                                parallel_parent = parent_context
                            )
                            subtask.Requirements.client_tags = self.Requirements.client_tags
                            subtask.save().enqueue()
                            targets[subtask.id] = 'resource_id'
                            self.Context.childs.append(subtask.id)
                        raise sdk2.WaitOutput(targets, wait_all=True)
                    # End parallel encoding, catch status WAIT_OUT -> ENQUEUED
                    if len(self.Context.childs) > 0:
                        segments = []
                        for subtask_id in self.Context.childs:
                            resource_data = sdk2.ResourceData(sdk2.Task[subtask_id].Parameters.resource_id)
                            segments.append(resource_data.path)
                        self.streams_concat(segments, files)

                if self.is_strm_hls_url():
                    m3u8_file_name = "{}_{}.m3u8".format(self.Parameters.s3_key_prefix, resolutions_ratio_adaptive[-1][0])
                    self.stream_copy_hls(m3u8_file_name)
                    files.append(m3u8_file_name)

                if self.is_youtube_dl:
                    src_orig_video = os.path.basename(self.src_video)
                    self.upload_file(src_orig_video)
                    self.src_streams["format"]["filename"] = "{}/{}/{}{}".format(
                            self.s3_endpoint_url,
                            self.Parameters.s3_bucket,
                            self.Parameters.s3_dir,
                            src_orig_video)
                    self.resolution_src["format"] = self.src_streams["format"]

                if self.Parameters.ffpmeg_thumbnails_scene:
                    thumbnails = self.publish_thumbnails()

                if self.timeline_tiles:
                    tiles_files, screen_files = self.publish_timeline_tiles()

                audios = None
                if self.is_ott_hls() or self.is_mss_packaging() or self.is_drm_packaging():
                    audios = self.extract_audios()

                if self.Parameters.rt_parallel_encoding:
                    files = self.retrieve_rt_video(rt_task_id)

                if self.is_owner('OTT'):
                    to_publish = list(files)
                    if audios is not None:
                        to_publish.extend(audios)
                    internal_files = self.publish_internal_mp4_files(to_publish)

                self.drm_package = []
                if self.is_drm_packaging():
                    files_video_frag = self.fragment(files)
                    files_audio_frag = self.fragment(audios)

                    if self.Parameters.drm_playready or self.Parameters.drm_widevine:
                        """Logic of playready and widevine packaging a lot the same"""

                        types = []
                        if self.Parameters.drm_playready:
                            types.append('playready')
                        if self.Parameters.drm_widevine:
                            types.append('widevine')

                        files_packaged = self.package(
                            self.encrypt(files_video_frag),
                            self.encrypt(files_audio_frag),
                            subtitles,
                            types
                        )

                        self.drm_package = self.parse_drm_package_params(
                                files_packaged,
                                types,
                                self.drm_path,
                                subtitles
                                )
                        self.publish_dash_mss_files(files_packaged)

                    if self.Parameters.drm_fairplay:
                        '''Fairplay logic is different'''
                        self.package_fairlay_hls(
                                files_video_frag,
                                files_audio_frag,
                                subtitles
                                )
                        self.publish_ott_hls_files()

                    for _package in self.drm_package:
                        _package.update({
                            'video_descriptor_id': self.to_video_descriptor_id(self.get_product())
                            })
                        if self.timeline_tiles is not None:
                            _package['meta']['streams']['thumbnail_tiles'] = self.timeline_tiles.get_meta_json()

                    if self.Parameters.package_clear_hls_as_well:
                        self.run_shell("rm -rf {}/*".format(self.sandbox_path("packaged")))
                        hls_files = self.package_clear_hls(files_video_frag, files_audio_frag, subtitles)

                        self.send_status_hook({
                            'status': 'DRM_VIDEO_FULL_UPLOADED',
                            'stream': json.dumps(self.resolution_src) if self.resolution_src else '',
                            'package': json.dumps(self.drm_package),
                            'video_descriptor_id': self.video_descriptor_id(),
                            'stream_url': self.m3u8_adaptive_url,
                            'meta': self.clear_hls_meta(hls_files, subtitles)
                        }, timeout=5)
                    else:
                        self.send_status_hook({
                            'status': 'DRM_VIDEO_FULL_UPLOADED',
                            'stream': json.dumps(self.resolution_src) if self.resolution_src else '',
                            'package': json.dumps(self.drm_package)
                        }, timeout=5)

                    self.Parameters.drm_package = json.dumps(self.drm_package)
                elif self.is_ott_hls():
                    hls_files = self.package_clear_hls(self.fragment(files), self.fragment(audios), subtitles)
                    self.clear_hls_hook(hls_files, subtitles)
                elif self.is_mss_packaging():
                    files_packaged = self.package(
                            self.fragment(files),
                            self.fragment(audios),
                            'all',
                            ('mss', 'dash')
                            )
                    self.publish_dash_mss_files(files_packaged)
                    ismc_manifest = next(iter(filter(lambda x: x.endswith('.ismc'), files_packaged)))
                    self.Parameters.stream_url = self.drm_path(ismc_manifest)
                elif self.is_ott_trailer():
                    self.publish_ott_trailer_files(files)
                else:
                    (files_m3u8) = self.publish_files(files, files_fast)

            else:
                msg = 'resolution not available'
                self.send_status_hook({'error_message': msg})
                raise common.errors.TaskFailure(msg)
        except Exception as e:
            logging.error(traceback.format_exc())
            raise TemporaryError(e)
        else:
            if self.resolution_src:
                self.Parameters.stream = json.dumps(self.resolution_src)
            self.Parameters.has_sound = isinstance(self.audio_stream, dict)
            self.Parameters.video_descriptor_id = self.video_descriptor_id()
            if self.get_output_format() != 'hls' and len(files) > 0:
                try:
                    self.Parameters.formats = json.dumps(
                        [self.get_format(f.file_name, '{}/{}/{}'.format(self.s3_url, f.s3_bucket, f.s3_path))
                         for f in internal_files]
                        )
                except NameError:
                    self.Parameters.formats = json.dumps(self.get_formats_files(files))
            if len(files_m3u8) > 0 or self.is_ott_hls() or self.is_ott_trailer() or \
                    (self.is_drm_packaging() and self.Parameters.package_clear_hls_as_well):
                self.Parameters.stream_url = self.m3u8_adaptive_url
            if len(subtitles) > 0:
                self.Parameters.subtitle_package = json.dumps(subtitles)
            if len(thumbnails) > 0:
                self.Parameters.thumbnail_urls = json.dumps([x.url for x in thumbnails])
            if len(tiles_files) > 0 or len(screen_files) > 0:
                self.Parameters.timeline_tiles_urls = json.dumps(self.timeline_tiles.get_meta_json())
            if len(self.files_awaps) > 0:
                self.Parameters.vast_formats = json.dumps([self.get_format(f) for f in self.files_awaps])
        finally:
            self.shutdown()
