# coding: utf-8
import time

import requests
from requests.auth import AuthBase
from requests.compat import urljoin

try:
    from past.builtins import xrange
except ImportError:
    pass


class RTClient(object):
    class RTSubmitError(BaseException):
        pass

    class RTRetrieveError(BaseException):
        pass

    class HeaderAuth(AuthBase):
        def __init__(self, name, secret):
            self.name = name
            self.secret = secret

        def __call__(self, req):
            req.headers[self.name] = self.secret
            return req

    def __init__(
        self,
        s3_bucket,
        s3_dir,
        s3_prefix,
        token,
        host,
        session_factory=requests.Session,
        timeout=(5, 30)
    ):
        self.s3_params = {
            "Bucket":          s3_bucket,
            "DirInsideBucket": s3_dir,
            "KeyPrefix":       s3_prefix
        }
        self.auth = self.HeaderAuth('Authorization', token)
        self.session_factory = session_factory
        self.timeout = timeout
        self.rt_host = host

    def submit_video(
        self,
        url,
        segment_length_sec,
        preset,
        keep_aspect_ratio,
        ratio,
        resolutions
    ):
        """
        Submit video by s3 url to real time transcoding
        For now only mp4 supported

        :returns: rt task id
        :rtype: str
        """

        items = []

        for name, resolution in resolutions:
            if not name.startswith(ratio):
                continue
            width, height, video_bitrate, audio_bitrate, _ = resolution
            items.append({
                "Name":           name,
                "Width":          int(width),
                "Height":         int(height),
                "VideoBitrate":   [int(video_bitrate) * 1000],
                "AudioBitrate":   int(audio_bitrate) * 1000,
                "AllowedFormats": ["EMp4"]
            })
        import logging
        import json
        j = {
            "DoneWebhookUrl":   "http://no-host.yandex.net",
            "InputVideoUrl":    url,
            "KeepAspectRatio":  keep_aspect_ratio,
            "OutputFormats":    ["EMp4"],
            "OutputStreams":    ["EVideo"],
            "Preset":           preset,
            "S3Params":         self.s3_params,
            "SegmentLengthSec": segment_length_sec,
            "UserRatios":       {
                "Items": [
                    {
                        "RationWidth": int(ratio[:-1]),
                        "RatioHeight": int(ratio[-1]),
                        "Resolutions": items
                    }
                ]
            }
        }
        logging.info(json.dumps(j))
        resp = self.session_factory().post(
            urljoin(self.rt_host, '/create-task'),
            auth=self.auth,
            timeout=self.timeout,
            json=j
        )

        if resp.status_code in xrange(400, 500):
            raise self.RTSubmitError(
                'Integration error, code={}, text={}'.format(
                    resp.status_code,
                    resp.text
                )
            )
        if resp.status_code in (500, 600):
            raise self.RTSubmitError(
                'Server error, code={}, text={}'.format(
                    resp.status_code,
                    resp.text
                )
            )
        try:
            body = resp.json()
        except JSONDecodeError:
            raise self.RTSubmitError('Unparsable response ' + resp.text)
        if 'Error' in body:
            raise self.RTSubmitError('Error response ' + str(body['Error']))
        if 'TaskId' not in body:
            raise self.RTSubmitError('No TaskId found: ' + resp.text)
        return body['TaskId']

    def retrieve_video(self, task_id, max_error=3, check_delay=3):
        """
        Block thread and try to retrieve video until success

        :returns: rt response with streams
        :rtype: dict
        """
        error_count = 0
        session = self.session_factory()
        while True:
            try:
                resp = session.get(
                    urljoin(self.rt_host, '/get-task'),
                    timeout=self.timeout,
                    params={'id': task_id}
                )
                status = resp.json()['Info']['StatusStr']
                if status == 'ETSDone':
                    return resp.json()['Info']
                if status == 'ETSError':
                    error_code = resp.json()['Info'].get('ErrorCodeStr', 'unknown')
                    raise self.RTRetrieveError('RT error={}, response={}'.format(error_code, str(resp.text)))
            except Exception as e:
                error_count += 1
                if error_count >= max_error:
                    raise self.RTRetrieveError(
                        'Max errors exceeded, last error: {}, last status_code, last response'.format(
                            str(e),
                            str(resp.status_code),
                            str(resp.text)
                        )
                    )
            time.sleep(check_delay)
