# -*- coding: utf-8 -*-
import io


class Writer(object):

    def __init__(self, output_file):
        self.writer = io.open(output_file, 'w+', encoding='utf-8')

    def write(self, string, *params):
        unicode_params = [
            param if type(param) == unicode else unicode(str(param), 'utf-8')
            for param in params
            ]
        self.writer.write(unicode(string).format(*unicode_params))

    def linebreak(self):
        self.write('\n')

    def do_write(self):
        self.writer.close()


class MasterPlaylist(object):

    def __init__(self, output_file, version=4):
        self.writer = Writer(output_file)
        self.writer.write('#EXTM3U')
        self.writer.linebreak()
        self.writer.write('#EXT-X-VERSION:{}'.format(version))
        self.writer.linebreak()

        self.audio_group_id = None
        self.subtitles_group_id = None
        self.output_file = output_file
        self.session_keys = set()

    def audio(self, path, name, lang, default=False, id='audio'):
        self.writer.write('#EXT-X-MEDIA:TYPE=AUDIO,')
        self.writer.write('GROUP-ID="{}",', id)
        self.writer.write('NAME="{}",', name)
        self.writer.write('LANGUAGE="{}",', lang)
        if default and self.audio_group_id is None:
            self.writer.write('AUTOSELECT=YES,DEFAULT=YES,')
        self.writer.write('URI="{}"', path)
        self.writer.linebreak()

        self.audio_group_id = id

    def subtitle(self, path, name, lang, default=False, id='subs'):
        self.writer.write('#EXT-X-MEDIA:TYPE=SUBTITLES,')
        self.writer.write('GROUP-ID="{}",', id)
        self.writer.write('NAME="{}",', name)
        self.writer.write('LANGUAGE="{}",', lang)
        if default and self.subtitles_group_id is None:
            self.writer.write('AUTOSELECT=YES,DEFAULT=NO,')
        self.writer.write('URI="{}"', path)
        self.writer.linebreak()

        self.subtitles_group_id = id

    def video(self, path, avg_bandwidth, bandwidth, codecs, width, height, skip_audio_track_requirements=False):
        if not skip_audio_track_requirements and self.audio_group_id is None:
            raise ValueError('Illegal attempt to write video to playlist without audio')

        self.writer.write('#EXT-X-STREAM-INF:')
        self.writer.write('AVERAGE-BANDWIDTH={},', avg_bandwidth)
        self.writer.write('BANDWIDTH={},', bandwidth)
        self.writer.write('CODECS="{}",', codecs)
        self.writer.write('RESOLUTION={}x{},', width, height)
        if self.audio_group_id is not None:
            self.writer.write('AUDIO="{}"', self.audio_group_id)
        if self.subtitles_group_id is not None:
            self.writer.write(',SUBTITLES="{}"', self.subtitles_group_id)
        self.writer.linebreak()
        self.writer.write(path)
        self.writer.linebreak()

    def deephd(self, width, height, bandwidth):
        value = '{}x{}@{}'.format(width, height, bandwidth)
        self.session_data("com.yandex.video.levels.deephd", value)

    def session_data(self, data_id, value):
        self.writer.write('#EXT-X-SESSION-DATA:DATA-ID="{}",VALUE="{}"'.format(data_id, value))
        self.writer.linebreak()

    def create(self):
        for session_key in self.session_keys:
            self.writer.write(session_key)
        self.writer.do_write()
        return self.output_file

    def add_session_key(self, ext_x_session_key):
        self.session_keys.add(ext_x_session_key)


class Command(object):

    def __init__(self, bin):
        self.bin = bin
        self.args = []

    def add(self, parameter=None, arg=True):
        if arg and parameter is not None:
            self.args.append(parameter.format(str(arg)))
        return self

    def run(self, run_function):
        return run_function(str(self))

    def __str__(self):
        return self.bin + ' ' + ' '.join(self.args)


def fairplay_language(lang):
    if lang.startswith('ru'):
        return 'ru'
    if lang.startswith('en'):
        'en'
    return 'und'


def create_subtitles_playlist(subtitle_file, duration, output_file):
    import math

    writer = Writer(output_file)
    writer.write('#EXTM3U')
    writer.linebreak()
    writer.write('#EXT-X-VERSION:{}', 3)
    writer.linebreak()
    writer.write('#EXT-X-MEDIA-SEQUENCE:{}', 0)
    writer.linebreak()
    writer.write('#EXT-X-TARGETDURATION:{}', int(math.ceil(float(duration))))
    writer.linebreak()
    writer.write('#EXTINF:{},', duration)
    writer.linebreak()
    writer.write(subtitle_file)
    writer.linebreak()
    writer.write('#EXT-X-ENDLIST')

    writer.do_write()
    return output_file
